{
    LogEdit1.pas - LOGIC Editor
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    *************************************************************

    UPDATES:
        Modified by : Eric Fullerton
        Oct.4,2001  : Fixed a trailing slash appearing in the directory
                      when saving a modified logic file. Also checks for
                      existing directory instead of the file.
                                                (Starting @ Line No.350)
        Oct.9,2001  : Added a popup menu to all text editing windows.                                        
}

unit LogEdit1;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, Menus, dcsystem, dcparser, dcstring, dccommon, dcmemo,
  dcSyntax, AGIcommands, dclib, ResourceWin1, LogCompile;

type
  TLogEditWin = class(TForm)
    StatusBar1: TStatusBar;
    MainMenu1: TMainMenu;
    FileMenu: TMenuItem;
    FileCompile: TMenuItem;
    N1: TMenuItem;
    FileClose: TMenuItem;
    SaveDialog1: TSaveDialog;
    FileSave: TMenuItem;
    FileCompileRun: TMenuItem;
    FileOpen: TMenuItem;
    FileNew: TMenuItem;
    OpenDialog1: TOpenDialog;
    FileSaveAs: TMenuItem;
    EditMenu: TMenuItem;
    EditCut: TMenuItem;
    EditCopy: TMenuItem;
    EditPaste: TMenuItem;
    N2: TMenuItem;
    EditUndo: TMenuItem;
    N3: TMenuItem;
    EditFind: TMenuItem;
    EditFindAgain: TMenuItem;
    FileChangeLogNum: TMenuItem;
    N4: TMenuItem;
    FilePrint: TMenuItem;
    PopupMenu1: TPopupMenu;
    PopupUndo: TMenuItem;
    N5: TMenuItem;
    PopupCut: TMenuItem;
    PopupCopy: TMenuItem;
    PopupPaste: TMenuItem;
    N6: TMenuItem;
    PopupFind: TMenuItem;
    PopupFindAgain: TMenuItem;
    EditMemo: TDCMemo;
    MemoSource1: TMemoSource;
    EditReplace: TMenuItem;
    EditGoto: TMenuItem;
    N7: TMenuItem;
    EditModify: TMenuItem;
    EditIndent: TMenuItem;
    EditOutdent: TMenuItem;
    N8: TMenuItem;
    EditUppercase: TMenuItem;
    EditLowercase: TMenuItem;
    EditInsertText: TMenuItem;
    EditDate: TMenuItem;
    EditTime: TMenuItem;
    PopupReplace: TMenuItem;
    PopupGoto: TMenuItem;
    N9: TMenuItem;
    PopupModifySelection: TMenuItem;
    PopupInsertText: TMenuItem;
    PopupIndent: TMenuItem;
    PopupOutdent: TMenuItem;
    PopupDate: TMenuItem;
    PopupTime: TMenuItem;
    N10: TMenuItem;
    PopupUppercase: TMenuItem;
    PopupLowercase: TMenuItem;
    EditCapitalize: TMenuItem;
    PopupCapitalize: TMenuItem;
    N11: TMenuItem;
    PopupHelpwith: TMenuItem;
    Bookmarks1: TMenuItem;
    GotoBookmarks1: TMenuItem;
    Bookmark01: TMenuItem;
    Bookmark11: TMenuItem;
    Bookmark21: TMenuItem;
    Bookmark31: TMenuItem;
    Bookmark41: TMenuItem;
    Bookmark51: TMenuItem;
    Bookmark61: TMenuItem;
    Bookmark71: TMenuItem;
    Bookmark81: TMenuItem;
    Bookmark91: TMenuItem;
    Bookmark02: TMenuItem;
    Bookmark12: TMenuItem;
    Bookmark22: TMenuItem;
    Bookmark32: TMenuItem;
    Bookmark42: TMenuItem;
    Bookmark52: TMenuItem;
    Bookmark62: TMenuItem;
    Bookmark72: TMenuItem;
    Bookmark82: TMenuItem;
    Bookmark92: TMenuItem;
    N12: TMenuItem;
    SyntaxParser1: TSyntaxParser;
    SaveasHTML1: TMenuItem;
    N13: TMenuItem;
    HTML1: TMenuItem;
    RTF1: TMenuItem;
    BMP1: TMenuItem;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FileCloseClick(Sender: TObject);
    procedure FileCompileClick(Sender: TObject);
    procedure FileSaveAsClick(Sender: TObject);
    procedure FileSaveClick(Sender: TObject);
    procedure FileCompileRunClick(Sender: TObject);
    procedure FileNewClick(Sender: TObject);
    procedure FileOpenClick(Sender: TObject);
    procedure EditMemoChange(Sender: TObject);
    procedure EditCutClick(Sender: TObject);
    procedure EditCopyClick(Sender: TObject);
    procedure EditPasteClick(Sender: TObject);
    procedure EditUndoClick(Sender: TObject);
    procedure EditFindClick(Sender: TObject);
    procedure EditFindAgainClick(Sender: TObject);
    procedure FileChangeLogNumClick(Sender: TObject);
    procedure FilePrintClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure EditMemoSelectionChange(Sender: TObject);
    procedure UpdateCursorPos(Sender: TObject);
    procedure ContextHelp1Click(Sender: TObject);
    procedure EditReplaceClick(Sender: TObject);
    procedure EditGotoClick(Sender: TObject);
    procedure EditIndentClick(Sender: TObject);
    procedure EditOutdentClick(Sender: TObject);
    procedure EditUppercaseClick(Sender: TObject);
    procedure EditLowercaseClick(Sender: TObject);
    procedure EditDateClick(Sender: TObject);
    procedure EditTimeClick(Sender: TObject);
    procedure EditCapitalizeClick(Sender: TObject);
    procedure PopupMenu1Popup(Sender: TObject);
    procedure PopupHelpwithClick(Sender: TObject);
    procedure Bookmark01Click(Sender: TObject);
    procedure Bookmark02Click(Sender: TObject);
    procedure Bookmark12Click(Sender: TObject);
    procedure Bookmark22Click(Sender: TObject);
    procedure Bookmark32Click(Sender: TObject);
    procedure Bookmark42Click(Sender: TObject);
    procedure Bookmark52Click(Sender: TObject);
    procedure Bookmark11Click(Sender: TObject);
    procedure Bookmark21Click(Sender: TObject);
    procedure Bookmark31Click(Sender: TObject);
    procedure Bookmark41Click(Sender: TObject);
    procedure Bookmark51Click(Sender: TObject);
    procedure Bookmark61Click(Sender: TObject);
    procedure Bookmark71Click(Sender: TObject);
    procedure Bookmark81Click(Sender: TObject);
    procedure Bookmark91Click(Sender: TObject);
    procedure Bookmark62Click(Sender: TObject);
    procedure Bookmark72Click(Sender: TObject);
    procedure Bookmark82Click(Sender: TObject);
    procedure Bookmark92Click(Sender: TObject);
    procedure EditMemoKeyPress(Sender: TObject; var Key: Char);
    procedure CheckCtrlSpace;
    function  CurrentChar : char;
    function  GetStringLeft(var s : string) : boolean;
    procedure ShowBox(Strings : TStrings);
    procedure EditMemoHintPopup(Sender: TObject; Strings: TStrings;
      var AllowPopup: Boolean; var PopupType: TPopupType);
    procedure ShowCommandHint(const s : string; APoint : TPoint; CheckBold : boolean);
    procedure CheckMethodParams(APoint : TPoint);
    procedure EditMemoHintInsert(Sender: TObject; var s: String);
    function GetCommandArgs: String;
    function GetTestCommandArgs: String;
    procedure EditMemoStateChange(Sender: TObject; State: TMemoStates);
    procedure EditMemoKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure BMP1Click(Sender: TObject);
    procedure HTML1Click(Sender: TObject);
    procedure RTF1Click(Sender: TObject);
  private
    { Private declarations }
    function GetLogicNumber(NumberEditText:string) : integer;
    function SaveSource : boolean;
    function CompileThisLogic : boolean;
    function GetParamText(APoint, Pt : TPoint) : string;
    function IsCommandHintVisibe(CheckHint: boolean) : boolean;
    procedure HideCommandHint;
  public
    { Public declarations }
    LogEditorNum : word;
    LogicNum : byte;
    ContentsIsLogic : boolean;  // if false, contents is a text file
    TextFilename : string;
    NewLogic : boolean;
    Modified : boolean;
    function AskClose : boolean;
    procedure SaveTextFile;
  end;

const MaxLogEditWins = 20;
      CurrentLineNumberInfo = 'Line:%3d Col:%3d';

var
     LogEditWin: array[1..MaxLogEditWins] of TLogEditWin;
     LogEditWinUsed : array[1..MaxLogEditWins] of boolean;
     FMethodPt      : TPoint;
     LogEditWidth,
     LogEditHeight,
     LogEditTop,
     LogEditLeft    : Integer;

function EditLogic(LogNum:byte;new:boolean) : integer;  // return value = editor window number
function EditTextFile(Filename:string) : integer;  // return value = editor window number

implementation

uses Defines, VolMan2, LogDecode, GetResourceNum1, main1;

{$R *.DFM}


type
  TMSource = class(TCustomMemoSource);
  TMMemo = class(TCustomDCMemo);

{*************************************************************}
function TLogEditWin.AskClose : boolean;
{*************************************************************}
var SaveMessageResult : integer;
    WindowName : PChar;
begin
  AskClose := True;
  if Modified then
  begin
    GetMem(WindowName,Length(LogEditWin[LogEditorNum].Caption)+1);
    StrPCopy(WindowName,LogEditWin[LogEditorNum].Caption);
    if ContentsIsLogic then
    begin  // logic editor
      SaveMessageResult := Application.MessageBox('Do you want to save changes?',WindowName,MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then SaveSource
      else if SaveMessageResult = IDCANCEL then AskClose := False;
      if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then Modified := False;
    end
    else
    begin // text editor
      SaveMessageResult := Application.MessageBox('Do you want to save changes?',WindowName,MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then FileSaveClick(Application)
      else if SaveMessageResult = IDCANCEL then AskClose := False;
      if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then Modified := False;
    end;
    FreeMem(WindowName,Length(LogEditWin[LogEditorNum].Caption)+1);
  end;
end;

{*************************************************************}
function EditLogic(LogNum:byte;new:boolean) : integer;
{*************************************************************}
var ResourceData : TResource;
    CurWinNum : byte;
    ThisWinNum : byte;
    ResourceLoadSuccess : boolean;
    LogicText : TStringList;
    TempStream : TMemoryStream;
begin
  EditLogic := 0;
  ThisWinNum := 0;
  if (not new) then
  begin
    CurWinNum := 0;
    for CurWinNum := 1 to MaxLogEditWins do
    if LogEditWinUsed[CurWinNum] and LogEditWin[CurWinNum].ContentsIsLogic
    and (not LogEditWin[CurWinNum].NewLogic) and (LogEditWin[CurWinNum].LogicNum = LogNum) then
      ThisWinNum := CurWinNum;
    if ThisWinNum > 0 then LogEditWin[ThisWinNum].Show;  // logic is already open in another window so set focus to it
  end;
  if ThisWinNum = 0 then
  begin  // the logic is not already open in another window
    for CurWinNum := 1 to MaxLogEditWins do
      if not LogEditWinUsed[CurWinNum] then ThisWinNum := CurWinNum;
    if ThisWinNum = 0 then
      ShowMessage('Error: Too many editors already open.')
    else
    begin
      LogEditWinUsed[ThisWinNum] := True;
      LogEditWin[ThisWinNum] := TLogEditWin.Create(Main);
      if Settings.LogicEditor.MaximizeEditorWindow then LogEditWin[ThisWinNum].WindowState := wsMaximized;
      LogEditWin[ThisWinNum].LogEditorNum := ThisWinNum;
      LogEditWin[ThisWinNum].LogicNum := LogNum;
      LogEditWin[ThisWinNum].ContentsIsLogic := True;
      LogEditWin[ThisWinNum].NewLogic := New;
      LogEditWin[ThisWinNum].FileCompile.Visible := True;
      LogEditWin[ThisWinNum].FileCompileRun.Visible := True;
      LogEditWin[ThisWinNum].FileNew.Visible := False;
      LogEditWin[ThisWinNum].FileOpen.Visible := False;
      LogEditWin[ThisWinNum].FileSaveAs.Visible := False;
      LogEditWin[ThisWinNum].FileChangeLogNum.Visible := True;
      LogEditWin[ThisWinNum].FileCompile.Enabled := LogEditWin[ThisWinNum].FileCompile.Visible;
      LogEditWin[ThisWinNum].FileCompileRun.Enabled := LogEditWin[ThisWinNum].FileCompileRun.Visible;
      LogEditWin[ThisWinNum].FileNew.Enabled := LogEditWin[ThisWinNum].FileNew.Visible;
      LogEditWin[ThisWinNum].FileOpen.Enabled := LogEditWin[ThisWinNum].FileOpen.Visible;
      LogEditWin[ThisWinNum].FileSaveAs.Enabled := LogEditWin[ThisWinNum].FileSaveAs.Visible;
      LogEditWin[ThisWinNum].FileChangeLogNum.Enabled := LogEditWin[ThisWinNum].FileChangeLogNum.Visible;
      Main.tbFileSave.Enabled := LogEditWin[ThisWinNum].FileSaveAs.Enabled;

      if LogEditWin[ThisWinNum].NewLogic then
      begin
        LogEditWin[ThisWinNum].Caption := 'Logic editor - New Logic';
      end
      else
      begin
        LogEditWin[ThisWinNum].Caption := 'Logic editor - ' + ResourceName(LOGIC,Lognum);
        LogEditWin[ThisWinNum].StatusBar1.Panels[0].Text := 'Reading logic....';
        if FileExists(GameDir+'src\Logic'+IntToStr(LogNum)+'.txt') then
        begin  // source already exists, read that
          LogEditWin[ThisWinNum].EditMemo.Lines.LoadFromFile(GameDir+'src\Logic'+IntToStr(LogNum)+'.txt');
          LogEditWin[ThisWinNum].StatusBar1.Panels[0].Text := '';
          LogEditWin[ThisWinNum].EditMemo.ReadOnly := False;
          LogEditWin[ThisWinNum].Modified := False;
        end
        else
        begin // source does not exist, must decode logic
          ResourceData := ReadResource(LOGIC,LogNum);
          if ResourceData.Size > 0 then
          begin
            LogicText := TStringList.Create;
            LogEditWin[ThisWinNum].StatusBar1.Repaint;
            if DecodeLogic(ResourceData,LogicText) then {DecodeLogic frees the memory after it has used it.}
            begin
              TempStream := TMemoryStream.Create;
              LogicText.SaveToStream(TempStream);
              TempStream.Position := 0;
              LogEditWin[ThisWinNum].EditMemo.Lines.LoadFromStream(TempStream);
              TempStream.Free;
              LogEditWin[ThisWinNum].StatusBar1.Panels[0].Text := '';
              LogEditWin[ThisWinNum].EditMemo.ReadOnly := False;
              LogEditWin[ThisWinNum].Modified := False;
            end
            else
            begin
              ShowMessage('Error loading Logic. The resource may be corrupt.');
              LogEditWin[ThisWinNum].Close;
              LogEditWinUsed[ThisWinNum] := False;
            end;
            LogicText.Free;
          end  // if ResourceData.Size > 0
          else
          begin
            ShowMessage('Error: Could not load resource. The files may be open or the resource may be corrupt.');
            LogEditWin[ThisWinNum].Close;
            LogEditWinUsed[ThisWinNum] := False;
          end;
        end;
      end;  // if not LogEditWin[ThisWinNum].NewLogic
    end;
  end;  // if ThisWinNum = 0 (i.e. the logic is not already open in another window)
  EditLogic := ThisWinNum;
  LogEditWin[ThisWinNum].Resize;
  LogEditWin[ThisWinNum].UpdateCursorPos(LogEditWin[ThisWinNum]);
end;

{*************************************************************}
function EditTextFile(Filename:string) : integer;
{*************************************************************}
var
    CurWinNum : byte;
    ThisWinNum : byte;
begin
  EditTextFile := 0;
  ThisWinNum := 0;
  CurWinNum := 0;
  for CurWinNum := 1 to MaxLogEditWins do
  if LogEditWinUsed[CurWinNum] and (not LogEditWin[CurWinNum].ContentsIsLogic)
  and (Lowercase(LogEditWin[CurWinNum].TextFilename) = Lowercase(Filename)) then
    ThisWinNum := CurWinNum;
  if ThisWinNum > 0 then LogEditWin[ThisWinNum].Show;  // text file is already open in another window so set focus to it
  if ThisWinNum = 0 then
  begin  // the text file is not already open in another window
    for CurWinNum := 1 to MaxLogEditWins do
      if not LogEditWinUsed[CurWinNum] then ThisWinNum := CurWinNum;
    if ThisWinNum = 0 then
      ShowMessage('Error: Too many editors already open.')
    else
    begin
      if (Filename <> '') and (not CanAccessFile(Filename)) then
        ShowMessage('Error: Could not read file.')
      else
      begin
        LogEditWinUsed[ThisWinNum] := True;
        LogEditWin[ThisWinNum] := TLogEditWin.Create(Main);
        if Settings.LogicEditor.MaximizeEditorWindow then LogEditWin[ThisWinNum].WindowState := wsMaximized;
        LogEditWin[ThisWinNum].LogEditorNum := ThisWinNum;
        LogEditWin[ThisWinNum].ContentsIsLogic := False;
        LogEditWin[ThisWinNum].FileCompile.Visible := False;
        LogEditWin[ThisWinNum].FileCompileRun.Visible := False;
        LogEditWin[ThisWinNum].FileNew.Visible := True;
        LogEditWin[ThisWinNum].FileOpen.Visible := True;
        LogEditWin[ThisWinNum].FileSaveAs.Visible := True;
        LogEditWin[ThisWinNum].FileChangeLogNum.Visible := False;
        LogEditWin[ThisWinNum].FileCompile.Enabled := LogEditWin[ThisWinNum].FileCompile.Visible;
        LogEditWin[ThisWinNum].FileCompileRun.Enabled := LogEditWin[ThisWinNum].FileCompileRun.Visible;
        LogEditWin[ThisWinNum].FileNew.Enabled := LogEditWin[ThisWinNum].FileNew.Visible;
        LogEditWin[ThisWinNum].FileOpen.Enabled := LogEditWin[ThisWinNum].FileOpen.Visible;
        LogEditWin[ThisWinNum].FileSaveAs.Enabled := LogEditWin[ThisWinNum].FileSaveAs.Visible;
        Main.tbFileSave.Enabled := LogEditWin[ThisWinNum].FileSaveAs.Enabled;
        LogEditWin[ThisWinNum].FileChangeLogNum.Enabled := LogEditWin[ThisWinNum].FileChangeLogNum.Visible;
        if Filename = '' then LogEditWin[ThisWinNum].Caption := 'Text editor'
        else
        begin
          LogEditWin[ThisWinNum].Caption := 'Text editor - ' + ExtractFileName(Filename);
          LogEditWin[ThisWinNum].EditMemo.Lines.LoadFromFile(Filename)
        end;
        LogEditWin[ThisWinNum].Modified := False;
        LogEditWin[ThisWinNum].TextFilename := Filename;
      end;
    end;  // if ThisWinNum > 0
  end;  // if ThisWinNum = 0 (i.e. the logic is not already open in another window)
  EditTextFile := ThisWinNum;
  LogEditWin[ThisWinNum].Resize;
  LogEditWin[ThisWinNum].UpdateCursorPos(LogEditWin[ThisWinNum]);
end;

{*************************************************************}
function TLogEditWin.GetLogicNumber(NumberEditText:string) : integer;
{*************************************************************}
var NewNum : byte;
    LogicIndex : integer;
begin
  GetLogicNumber := -1;
  GetResourceNum.WindowFunction := LogEditGetNum;
  GetResourceNum.NumberEdit.Text := NumberEditText;
  GetResourceNum.ShowModal;
  if GetResourceNum.OKPressed then
  begin
    NewNum := StrToInt(GetResourceNum.NumberEdit.Text);  // GetResourceNum verifies that this is a number between 0 and 255.
      LogicIndex := ResourceIndex(LOGIC,NewNum);
      if (LogicIndex < 0) or (AskYesNo('Change logic number','Logic '+IntToStr(NewNum)+' already exists. Do you wish to replace it?')) then
        GetLogicNumber := NewNum;
  end;
end;

{*************************************************************}
function TLogEditWin.SaveSource : boolean;
{*************************************************************}
var FoundDir : boolean;
    SourceFilename : string;
    NewLogicNum : integer;
    Code: Integer;
begin
  SaveSource := False;

  if NewLogic then
  begin
    NewLogicNum := GetLogicNumber('');
    if NewLogicNum >= 0 then
    begin
      NewLogic := False;
      LogicNum := NewLogicNum;
      LogEditWin[LogEditorNum].Caption := 'Logic editor - ' + ResourceName(LOGIC,LogicNum);
    end;
  end;

  if (not NewLogic) then
  begin
    Code := GetFileAttributes(PChar(GameDir+SourceDir));
    FoundDir := (Code <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
    if not FoundDir then FoundDir := CreateDir(GameDir+SourceDir);
    if not FoundDir then
      ShowMessage('Error creating directory "'+GameDir+SourceDir+'". The source code could not be saved.')
    else
    begin
      SourceFilename := GameDir+SourceDir+'\Logic'+IntToStr(LogicNum)+'.txt';
      if FileExists(SourceFilename) and (not CanAccessFile(SourceFilename)) then
       ShowMessage('Error saving source code to "'+SourceFilename+'".')
      else
      begin
        EditMemo.Lines.SaveToFile(SourceFilename);
        SaveSource := True;
        Modified := False;
        StatusBar1.Panels[0].Text := 'File saved.';
        EditUndo.Enabled := False;
        PopupUndo.Enabled := False;
      end;
    end;
  end;
end;

{*************************************************************}
function TLogEditWin.CompileThisLogic : boolean;
{*************************************************************}
var CompiledLogic : TResource;
begin
  CompileThisLogic := False;
  CompiledLogic := CompileLogic(LogEditorNum);
  if CompiledLogic.Size > 0 then
  begin
    StatusBar1.Panels[0].Text := 'Compile successful.';
    AddResource(CompiledLogic,LOGIC,LogicNum); // AddResource will free the memory assigned to CompiledLogic
    CompileThisLogic := True;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FormClose(Sender: TObject; var Action: TCloseAction);
{*************************************************************}
begin
  LogEditWinUsed[LogEditorNum] := false;
  action := caFree;
end;

{*************************************************************}
procedure TLogEditWin.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
{*************************************************************}
begin
  CanClose := AskClose;
end;

{*************************************************************}
procedure TLogEditWin.FileCloseClick(Sender: TObject);
{*************************************************************}
begin
  Close;
end;

{*************************************************************}
procedure TLogEditWin.FileCompileClick(Sender: TObject);
{*************************************************************}
begin
  SaveSource;
  if not NewLogic then CompileThisLogic;
end;

{*************************************************************}
procedure TLogEditWin.SaveTextFile;
{*************************************************************}
begin
  EditMemo.Lines.SaveToFile(TextFilename);
  StatusBar1.Panels[0].Text := 'File saved.';
  Modified := False;
  EditUndo.Enabled := False;
  PopupUndo.Enabled := False;
end;

{*************************************************************}
procedure TLogEditWin.FileSaveAsClick(Sender: TObject);
{*************************************************************}
begin
  SaveDialog1.Filename := TextFilename;
  if SaveDialog1.Execute then
  begin
    if FileExists(SaveDialog1.Filename) and (not CanAccessFile(SaveDialog1.Filename)) then
      ShowMessage('Error: Could not write to file.')
    else
    begin
      TextFilename := SaveDialog1.Filename;
      LogEditWin[LogEditorNum].Caption := 'Text editor - ' + ExtractFileName(TextFilename);
      SaveTextFile;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FileSaveClick(Sender: TObject);
{*************************************************************}
begin
  if ContentsIsLogic then SaveSource
  else
  begin
    if TextFilename <> '' then
    begin
      if FileExists(TextFilename) and (not CanAccessFile(TextFilename)) then
        ShowMessage('Error: Could not write to file "'+TextFilename+'".')
      else
      begin
        SaveTextFile;
      end;
    end
    else
    begin
      FileSaveAsClick(Sender);
      Modified := False;
      EditUndo.Enabled := False;
      PopupUndo.Enabled := False;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FileCompileRunClick(Sender: TObject);
{*************************************************************}
begin
  SaveSource;
  if (not NewLogic) and CompileThisLogic then
    Main.GameRunClick(Sender);
end;

{*************************************************************}
procedure TLogEditWin.FileNewClick(Sender: TObject);
{*************************************************************}
begin
  if AskClose then
  begin
    TextFilename := '';
    LogEditWin[LogEditorNum].Caption := 'Text editor';
    EditMemo.Lines.Clear;
    StatusBar1.Panels[0].Text := '';
    Modified := False;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FileOpenClick(Sender: TObject);
{*************************************************************}
var CurWinNum : word;
    ThisWinNum : word;
begin
  if AskClose and OpenDialog1.Execute then
  begin
    if not CanAccessFile(OpenDialog1.Filename) then
      ShowMessage('Error: Could not read file.')
    else
    begin
      ThisWinNum := 0;
      for CurWinNum := 1 to MaxLogEditWins do
      if (CurWinNum <> LogEditorNum) and LogEditWinUsed[CurWinNum] and (not LogEditWin[CurWinNum].ContentsIsLogic)
      and (Lowercase(LogEditWin[CurWinNum].TextFilename) = Lowercase(OpenDialog1.Filename)) then
        ThisWinNum := CurWinNum;
      if ThisWinNum > 0 then
        ShowMessage('File is already open in another window.')
      else
      begin
        EditMemo.Lines.LoadFromFile(OpenDialog1.Filename);
        LogEditWin[LogEditorNum].Caption := 'Text editor - ' + ExtractFileName(OpenDialog1.Filename);
        TextFilename := OpenDialog1.Filename;
        StatusBar1.Panels[0].Text := '';
        Modified := False;
      end;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.EditMemoChange(Sender: TObject);
{*************************************************************}
begin
  if not Modified then Modified := True;
  StatusBar1.Panels[0].Text := '';
  EditUndo.Enabled := EditMemo.GetSource.UndoAvailable;
  PopupUndo.Enabled := EditUndo.Enabled;
end;

{*************************************************************}
procedure TLogEditWin.UpdateCursorPos(Sender: TObject);
{*************************************************************}
begin
  with EditMemo.CaretPoint do
    StatusBar1.Panels[1].Text := Format(CurrentLineNumberInfo, [Y+1, X+1]);
end;

{*************************************************************}
procedure TLogEditWin.EditCutClick(Sender: TObject);
{*************************************************************}
begin
  EditMemo.CutToClipboard;
end;

{*************************************************************}
procedure TLogEditWin.EditCopyClick(Sender: TObject);
{*************************************************************}
begin
  EditMemo.CopyToClipboard;
end;

{*************************************************************}
procedure TLogEditWin.EditPasteClick(Sender: TObject);
{*************************************************************}
begin
  EditMemo.PasteFromClipboard;
end;

{*************************************************************}
procedure TLogEditWin.EditUndoClick(Sender: TObject);
{*************************************************************}
begin
  with EditMemo do
    if HandleAllocated then SendMessage(Handle, EM_UNDO, 0, 0);
end;

{*************************************************************}
procedure TLogEditWin.EditFindClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.ShowSearchDialog;
end;

{*************************************************************}
procedure TLogEditWin.EditFindAgainClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.FindNext;
end;

{*************************************************************}
procedure TLogEditWin.FileChangeLogNumClick(Sender: TObject);
{*************************************************************}
var NewLogicNum : integer;
begin
  if NewLogic then NewLogicNum := GetLogicNumber('')
  else NewLogicNum := GetLogicNumber(IntToStr(LogicNum));
  if NewLogicNum >= 0 then
  begin
    if (not NewLogic) and (NewLogicNum = LogicNum) then
      ShowMessage('Logic number not changed!')
    else
    begin
      NewLogic := False;
      LogicNum := NewLogicNum;
      LogEditWin[LogEditorNum].Caption := 'Logic editor - ' + ResourceName(LOGIC,LogicNum);
      SaveSource;
      CompileThisLogic;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FilePrintClick(Sender: TObject);
{*************************************************************}
var PrintCaption : string;
begin
  if ContentsIsLogic then
  begin
   if NewLogic then PrintCaption := 'New Logic'
   else PrintCaption := 'Logic '+IntToStr(LogicNum);
  end
  else
  begin
    if TextFilename = '' then PrintCaption := 'Untitled'
    else PrintCaption := ExtractFilename(TextFilename);
  end;
  EditMemo.PrintWithDialog;
end;

{*************************************************************}
procedure TLogEditWin.FormResize(Sender: TObject);
{*************************************************************}
begin
  StatusBar1.Panels[0].Width := StatusBar1.Width - 106;
end;

{*************************************************************}
procedure TLogEditWin.EditMemoSelectionChange(Sender: TObject);
{*************************************************************}
begin
  UpdateCursorPos(EditMemo);
end;

{*************************************************************}
procedure TLogEditWin.ContextHelp1Click(Sender: TObject);
{*************************************************************}
begin

end;

{*************************************************************}
procedure TLogEditWin.EditReplaceClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.ShowReplaceDialog;
end;

{*************************************************************}
procedure TLogEditWin.EditGotoClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.ShowGoToLineDialog;
end;

{*************************************************************}
procedure TLogEditWin.EditIndentClick(Sender: TObject);
{*************************************************************}
begin
  if EditMemo.SelLength > 0 then MemoSource1.IndentBlock;
end;

{*************************************************************}
procedure TLogEditWin.EditOutdentClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.OutDentBlock;
end;

{*************************************************************}
procedure TLogEditWin.EditUppercaseClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.UpperCaseBlock;
end;

{*************************************************************}
procedure TLogEditWin.EditLowercaseClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.LowerCaseBlock;
end;

{*************************************************************}
procedure TLogEditWin.EditDateClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.InsertDate;
end;

{*************************************************************}
procedure TLogEditWin.EditTimeClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.InsertTime;
end;

{*************************************************************}
procedure TLogEditWin.EditCapitalizeClick(Sender: TObject);
{*************************************************************}
begin
  MemoSource1.CapitalizeBlock;
end;

{*************************************************************}
procedure TLogEditWin.PopupMenu1Popup(Sender: TObject);
{*************************************************************}
begin
  PopupCut.Enabled := EditMemo.SelLength > 0;
  PopupCopy.Enabled := PopupCut.Enabled;
  PopupPaste.Enabled := EditMemo.CanPaste;
  PopupHelpwith.Caption := 'Help with "'+MemoSource1.TextAtCursor+'"';
end;

{*************************************************************}
procedure TLogEditWin.PopupHelpwithClick(Sender: TObject);
{*************************************************************}
begin
  Application.HelpJump(MemoSource1.TextAtCursor);
end;

procedure TLogEditWin.Bookmark01Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark0;
end;

procedure TLogEditWin.Bookmark02Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark0;
end;

procedure TLogEditWin.Bookmark12Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark1;
end;

procedure TLogEditWin.Bookmark22Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark2;
end;

procedure TLogEditWin.Bookmark32Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark3;
end;

procedure TLogEditWin.Bookmark42Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark4;
end;

procedure TLogEditWin.Bookmark52Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark5;
end;

procedure TLogEditWin.Bookmark11Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark1;
end;

procedure TLogEditWin.Bookmark21Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark2;
end;

procedure TLogEditWin.Bookmark31Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark3;
end;

procedure TLogEditWin.Bookmark41Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark4;
end;

procedure TLogEditWin.Bookmark51Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark5;
end;

procedure TLogEditWin.Bookmark61Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark6;
end;

procedure TLogEditWin.Bookmark71Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark7;
end;

procedure TLogEditWin.Bookmark81Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark8;
end;

procedure TLogEditWin.Bookmark91Click(Sender: TObject);
begin
  MemoSource1.ToggleBookMark9;
end;

procedure TLogEditWin.Bookmark62Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark6;
end;

procedure TLogEditWin.Bookmark72Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark7;
end;

procedure TLogEditWin.Bookmark82Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark8;
end;

procedure TLogEditWin.Bookmark92Click(Sender: TObject);
begin
  MemoSource1.GoToBookmark9;
end;

{*************************************************************}
procedure TLogEditWin.EditMemoKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
{*************************************************************}
begin
  if (Key = VK_TAB) and (EditMemo.SelLength > 0) then
  begin
    if Shift = [ssShift] then MemoSource1.OutdentBlock else MemoSource1.IndentBlock;
    Key := VK_NONAME;
  end;
end;

{*************************************************************}
procedure TLogEditWin.EditMemoKeyPress(Sender: TObject; var Key: Char);
{*************************************************************}
begin
  if (Key = ' ') then
    if (GetAsyncKeyState(VK_CONTROL) and $80000000) <> 0 then
    begin
      CheckCtrlSpace;
      Key := #0;
    end;
end;

{*************************************************************}
procedure TLogEditWin.CheckCtrlSpace;
{*************************************************************}
var
  AStrings : TStringList;
  s        : string;
  sels,args: string;
  ii,i     : Integer;
  iArgs    : Integer;
begin
  AStrings := TStringList.Create;
  try
//    if (CurrentChar = '.') then
    if (CurrentChar = '.') or (CurrentChar = '') or (CurrentChar = ' ') then
    begin
      AStrings.Text := s;
      for i := 0 to 181 do  // Add all 181 AGI Commands to the popup box
      begin
        args := '';
        for ii := 1 to AGICommands.AGICommand[i].numArgs do // There are 7 possible arguments for each command
        begin
          iArgs := AGICommands.AGICommand[i].argTypes[ii];
          if args = '' then args := ArgTypeName[iArgs] else args := args + ',' + ArgTypeName[iArgs];
        end;
        if args <> '' then
          AStrings.Add('~'+AGICommands.AGICommand[i].Name+'~'+'('+args+')')
        else
          AStrings.Add('~'+AGICommands.AGICommand[i].Name+'~');
      end;
      AStrings.Sort;
      ShowBox(AStrings);
    end
    else
      with TMMemo(EditMemo), GetSource do
      begin
        BeginUpdate(acSetSelRect);
        MarkWordOnTemplate;

        sels := LowerCase(SelStrings.Text);

        Delete(sels, Pos('#13#10', sels), MaxInt);

        if AStrings.Count > 0 then
        begin
          ShowBox(AStrings);
          {$IFDEF NEWVER}
          if (SelectionType <> stNotSelected) then
            HintPos := SelectionRect.TopLeft
          else
            HintPos := CaretPoint;
          {$ENDIF}
        end;
        SelectionType := stNotSelected;
        EndUpdate;
      end;
  finally
    AStrings.Free;
  end;
end;

{*************************************************************}
function  TLogEditWin.GetStringLeft(var s : string) : boolean;
{*************************************************************}
var
  ALeft      : integer;
  ARight     : integer;
begin
  result := false;
  with EditMemo, TMSource(GetSource), CaretPoint do
  begin
    WordBounds(X - 1, Y, ALeft, ARight, wsWordOnly);
    if (ARight > ALeft) or ((ARight = ALeft) and (ARight > 0)) then
    begin
      s := Copy(Strings[Y], ALeft , ARight - ALeft + 1);
      result := true;
    end;
  end;
end;

{*************************************************************}
function TLogEditWin.CurrentChar : char;
{*************************************************************}
var
  s : string;
begin
  with EditMemo, TMSource(GetSource) do
  begin
    s := Strings[CaretPoint.Y];
    if (CaretPoint.X > 0) and (CaretPos.X <= length(s)) then
      result := s[CaretPoint.X ]
    else
      result := #0;
  end;
end;

{*************************************************************}
procedure TLogEditWin.ShowBox(Strings : TStrings);
{*************************************************************}
begin
  if Strings.Count > 0 then
  begin
    EditMemo.HidePopupHint;
    EditMemo.ShowPopupListBox(Strings);
  end;
end;

{*************************************************************}
function DeleteBoldToken(const s : string) : string;
{*************************************************************}
var
  P : integer;
begin
  result := s;
  P := Pos('~', result);
  while P <> 0 do
  begin
    delete(result, P, 1);
    P := Pos('~', result);
  end;
end;

{*************************************************************}
procedure TLogEditWin.EditMemoHintPopup(Sender: TObject; Strings: TStrings;
  var AllowPopup: Boolean; var PopupType: TPopupType);
{*************************************************************}
var
  ALeft      : integer;
  ARight     : integer;
  Pt : TPoint;
  s  : string;
  ch : char;
begin
  ch := CurrentChar;
  AllowPopup := ch in ['(', '.'];
  if AllowPopup then
  begin
    if ch  = '(' then
    begin
      s := GetCommandArgs;
      ShowCommandHint(s, EditMemo.CaretPoint, true);
      allowpopup:=false;
    end
{    else
    if (ch = '.') and GetStringLeft(s)  then   // experimental
    begin
      PopupType := ptListBox;
      Strings.Text := s;
    end
}    else
      AllowPopup := false;
  end;
end;

{*************************************************************}
procedure TLogEditWin.ShowCommandHint(const s : string; APoint : TPoint; CheckBold : boolean);
{*************************************************************}
begin
  if s = '' then
    exit;
  with EditMemo, TextToPixelPoint(APoint) do
  begin
    ShowPopupHint(s, ClientToScreen(Point(X, Y + LineHeight + 1)), false);
    if CheckBold then
      CheckMethodParams(CaretPoint);
    FMethodPt := APoint;
  end;
end;

{*************************************************************}
function CommaCount(const s : string; APos : integer) : integer;
{*************************************************************}
var
  P : integer;
begin
  P := Pos(',' , s);
  result := 0;
  while (P <> 0) and (P <= APos) do
  begin
    inc(result);
    P := PosEx(',', s, P + 1);
  end;
end;

{*************************************************************}
function MakeParamBold(const s : string; Count : integer) : string;
{*************************************************************}
var
  P    : integer;
  OldP : integer;
  cnt  : integer;
begin
  P := Pos(',' , s);
  if P = 0 then
  begin
    result := s;
    exit;
  end
  else
    result := DeleteBoldToken(s);
  OldP := 0;
  P := Pos(',' , result);
  if Count <> 0 then
  begin
    cnt := 0;
    while P <> 0 do
    begin
      inc(cnt);
      OldP := P;
      P := PosEx(',', result, P + 1);
      if cnt >= Count then
        break;
    end;
  end;
  Insert('~', result, OldP + 1);
  if P <> 0 then
    Insert('~', result, P + 1);
end;

{*************************************************************}
procedure TLogEditWin.CheckMethodParams(APoint : TPoint);
{*************************************************************}
var
  ACount : integer;
begin
  with TMMemo(EditMemo), PopupHint do
  begin
    ACount := CommaCount(GetParamText(APoint, FMethodPt), APoint.X + 1);
    Caption := MakeParamBold(Caption, ACount)
  end;
end;


{*************************************************************}
function TLogEditWin.GetParamText(APoint, Pt : TPoint) : string;
{*************************************************************}
begin
  with EditMemo, APoint do
    result := Copy(Lines[Y], Pt.X + 1, X - Pt.X);
end;


{*************************************************************}
procedure TLogEditWin.EditMemoHintInsert(Sender: TObject; var s: String);
{*************************************************************}
var
  P : integer;
begin
  P := Pos('|', s);
  if P <> 0 then
    delete(s, P , MaxInt);
end;

{*************************************************************}
function TLogEditWin.GetCommandArgs: String;
{*************************************************************}
var
 ii,i   : Integer;
 iArgs  : Integer;
 args,s : String;
 isFound: Boolean;
begin
   for i := 0 to 181 do  // Search through all 181 AGI Commands
   begin
     args := '';
     GetStringLeft(s);
     if s = AGICommands.AGICommand[i].Name then
       for ii := 1 to AGICommands.AGICommand[i].numArgs do // There are 7 possible arguments for each command
       begin
         iArgs := AGICommands.AGICommand[i].argTypes[ii];
         if args = '' then args := ArgTypeName[iArgs] else args := args + ',' + ArgTypeName[iArgs];
         isFound := True;
       end;
     if isFound then break;
   end;

   if args <> '' then
     Result := '~'+s+'~'+'('+args+')'
   else
     // Arguments for the current Command are not found
     // As a last resort let's look for it in the AGI TestCommands
     Result := GetTestCommandArgs;
end;

{*************************************************************}
function TLogEditWin.GetTestCommandArgs: String;
{*************************************************************}
var
 ii,i   : Integer;
 iArgs  : Integer;
 args,s : String;
 isFound: Boolean;
begin
   for i := 0 to 18 do  // Search through the 18 AGI TestCommands
   begin
     args := '';
     GetStringLeft(s);
     if s = AGICommands.TestCommand[i].Name then
       for ii := 1 to AGICommands.TestCommand[i].numArgs do // There are 7 possible arguments for each command
       begin
         iArgs := AGICommands.TestCommand[i].argTypes[ii];
         if args = '' then args := ArgTypeName[iArgs] else args := args + ',' + ArgTypeName[iArgs];
         isFound := True;
       end;
     if isFound then break;
   end;

   if args <> '' then
     Result := '~'+s+'~'+'('+args+')'
   else
     Result := '';
end;

{*************************************************************}
procedure TLogEditWin.EditMemoStateChange(Sender: TObject; State: TMemoStates);
{*************************************************************}
begin
  if ComponentState * [csLoading, csReading] <> [] then
    exit;

  if IsCommandHintVisibe(true) then
  begin
    if ((msEdited in State) or (msPositionChanged in State)) and (CurrentChar = ')') then
      HideCommandHint
    else
    if (msPositionChanged in State) then
      with EditMemo, CaretPoint do
        if (Y <> FMethodPt.Y) or (X < FMethodPt.X) then
          HideCommandHint
        else
          CheckMethodParams(CaretPoint);
  end;
end;

{*************************************************************}
function TLogEditWin.IsCommandHintVisibe(CheckHint: boolean) : boolean;
{*************************************************************}
begin
  result := (FMethodPt.X <> -1) and (FMethodPt.Y <> -1) and (not CheckHint or TMMemo(EditMemo).PopupHint.Visible);
end;

{*************************************************************}
procedure TLogEditWin.HideCommandHint;
{*************************************************************}
begin
  EditMemo.HidePopupHint;
  FMethodPt := Point(-1, -1);
end;

{*************************************************************}
procedure TLogEditWin.BMP1Click(Sender: TObject);
{*************************************************************}
begin
  with Main.ResourceExtractDialog do
  begin
    Filter := 'Windows or OS/2 Bitmap (*.bmp)|*.bmp';
    DefaultExt := 'bmp';
    Title := 'Export as BMP';
    Filename := '';
  end;
  if Main.ResourceExtractDialog.Execute then
    EditMemo.DrawToBMPFile(Main.ResourceExtractDialog.Filename);
end;

{*************************************************************}
procedure TLogEditWin.HTML1Click(Sender: TObject);
{*************************************************************}
begin
  with Main.ResourceExtractDialog do
  begin
    Filter := 'Hyper Text Markup Language (*.htm,*.html)|*.htm,*.html';
    DefaultExt := 'htm';
    Title := 'Export as HTML';
    Filename := '';
  end;
  if Main.ResourceExtractDialog.Execute then
    EditMemo.ExportToHtmlFile(Main.ResourceExtractDialog.Filename);
end;

{*************************************************************}
procedure TLogEditWin.RTF1Click(Sender: TObject);
{*************************************************************}
begin
  with Main.ResourceExtractDialog do
  begin
    Filter := 'Rich Text Format (*.rtf,*.doc,*.wri)|*.rtf,*.doc,*.wri';
    DefaultExt := 'rtf';
    Title := 'Export as RTF';
    Filename := '';
  end;
  if Main.ResourceExtractDialog.Execute then
    EditMemo.ExportToRtfFile(Main.ResourceExtractDialog.Filename);
end;

end.
