;=======================================
;
; Music Quest Programmer's ToolKit
;
; Clock and Timer Services
;
; Written specifically for:
;   Microsoft Quick Basic (medium) model.
;
; Copyright (c) 1988
; Music Quest, Inc.
;
;=======================================
DATA    SEGMENT WORD PUBLIC
;=======================================
; Medium model values
;=======================================
p1of1           equ     6               ;offset to parameter 1
p1of2           equ     8               ;offset to parameter 1
p2of2           equ     6               ;offset to parameter 2
p1of3           equ     10              ;offset to parameter 1
p2of3           equ     8               ;offset to parameter 2
p3of3           equ     6               ;offset to parameter 3
;=======================================
; Timer Request Block
;=======================================
trqblok struc
trquse  db      0                       ;in use flag
trqcnt  dw      0                       ;tic count for event
trqefws dw      0                       ;seg(EFW)
trqefwo dw      0                       ;ofs(EFW)
trqblok ends
;=======================================
; Table of available TRQ blocks
;=======================================
trqdim  =       16
trq     trqblok trqdim dup(<0,0,0>)     ;array of trq bloks
;=======================================
; Clock data
;=======================================
clk_starts db   ?                       ;count of number of starts
clocktics dw    ?                       ;count of tics since last time
clockrate dw    ?                       ;clock tic decrement value
DATA    ENDS
;=======================================
CODE    SEGMENT BYTE PUBLIC
        ASSUME  CS:CODE,DS:DATA
trqiflag db     0                       ;TRQ table init flag
;=======================================
; External references
;=======================================
        extrn   mcccommand:far
;=======================================
; _mclk_init
;
; Initialize clock service
;
; Model:
; procedure _mclk_init(ticvalue: integer);
;   ticvalue = number of ticks to be counted
;     for each clock-to-PC interrupt
;
; Notes:
;   The tic value is the number of time units to
;   be counted on every timer tic.  It MUST match
;   the current settings of the MCC.  For example.
;   if the MCC clock resolution is 96 PPB and the
;   clock-to-PC rate is set to 48, then every
;   clock-to-PC interrupt counts as 12 clock tics.
;
;   The clock-to-PC interrupt occurs every M tics of
;   the MCC's clock, where M is the clock-to-PC rate
;   divided by 4.  The value of M is the number of
;   tics that the local clock will get for each
;   clock-to-PC interrupt.
;
;   Thus, the granularity of the local clock is
;   totally dependent on the frequency of clock-to-PC
;   interrupts and the number of tics each interrupt
;   represents.
;
;=======================================
        public  _mclk_init
_mclk_init proc far
        push    bp
        mov     bp,sp
;
        mov     ax,[bp+p1of1]           ;new tic rate value
        mov     clockrate,ax
        mov     clocktics,0             ;clear local clock
        mov     clk_starts,0            ;reset start flag
;
        pop     bp
        ret     2
_mclk_init endp
;=======================================
; _mclk_start
;
; Start clock service.  When the clock
; starts, clokc-to-PC interrupts are
; enabled and the local clock begins
; to "tic".
;
; Note: This routine uses mcccomand to
; start clock-to-PC interrupts.  The
; current interrupt handler must be
; able to handle any data captured by
; the mcccommand function.  If the interrupt
; handler can not handle all conditions,
; unpredicatble results may occur.
;
; Model:
; procedure _mclk_start;
;
;=======================================
        public  _mclk_start
_mclk_start proc far
        push    bp
        mov     bp,sp
;
        mov     clocktics,0             ;reset local clock
        inc     clk_starts              ;bump count of starts
        mov     ah,95H                  ;enable clock to PC
        call    mcccommand
;=======================================
; Sync local clock to MCC clock
;=======================================
        mov     cx,-1
mcs1:
        cmp     clocktics,0             ;wait for first tic
        jne     mcs2
        loop    mcs1                    ;avoid hang
mcs2:
        cli
        mov     clocktics,0             ;clock now synched
        sti
;=======================================
; Exit with local clock started and synced
;=======================================
        pop     bp
        ret
_mclk_start endp
;=======================================
; _mclk_stop
;
; Stop clock service.  The clock-to-PC is
; disabled, so that the local clock no
; longer "tics".
;
; Note: This routine uses mcccomand to
; start clock-to-PC interrupts.  The
; current interrupt handler must be
; able to handle any data captured by
; the mcccommand function.  If the interrupt
; handler can not handle all conditions,
; unpredicatble results may occur.
;
; Model:
; procedure _mclk_stop;
;
;=======================================
        public  _mclk_stop
_mclk_stop proc  far
        mov     ah,94H                  ;disable clock-to-PC interrupts
        call    mcccommand
        dec     clk_starts              ;decrement count of starts
        ret
_mclk_stop endp
;=======================================
; _midi_clock
;
; Read the current value of the local clock.
; After the clock is read, it is reset to 0.
;
; Model:
; function _midi_clock: integer;
;
;=======================================
        public  _midi_clock
_midi_clock proc far
        cli
        mov     ax,clocktics            ;get current clock value
        mov     clocktics,0             ;reset clock
        sti
        ret
_midi_clock endp
;=======================================
; clock_tic
;
; Clock tic interrupt handler
; Call by MCC co-processor SLIH
;
; Notes:
;   Assumes machine disabled at entry.
;   NOT callable from Pascal
;
;=======================================
        public  clock_tic
clock_tic proc far
;
        mov     dx,clockrate            ;tics/interrupt value
        add     clocktics,dx            ;update running timer
;=======================================
; Run through TRQ table, decrementing timers
;=======================================
        sub     bx,bx
        mov     cx,trqdim
;
tbtop:
        test    trq.trquse[bx],1
        jz      tbnext
        mov     ax,trq.trqcnt[bx]       ;decrement this timer
        sub     ax,dx
        mov     trq.trqcnt[bx],ax
        jz      tbexpire                ;timer has expired
        jnc     tbnext                  ;still > 0
tbexpire:
        mov     es,trq.trqefws[bx]      ;post the event
        mov     si,trq.trqefwo[bx]
        mov     word ptr es:[si],0FFh
;
tbnext:
        add     bx,SIZE trqblok
        loop    tbtop
;=======================================
; Exit after all timers decremented
;=======================================
        ret
clock_tic endp
;=======================================
; _set_trq
;
; Set up a Timer Request.  A TRQ block is
; allocated and an initial count down time
; is set. The function returns the "handle"
; for the TRQ.
;
; Model
; function _set_trq(var efw: integer; time: integer);
;                   seg(efw),ofs(efw),time
;                       +10      +8    +6
; Returned value
;   <0 = no TRQ available
;   >=0 = TRQ set
;
; efw - event flag word marked when TRQ expires
; time - number of ticks till TRQ expires
;
;=======================================
        public  _set_trq
_set_trq proc   far
        push    bp
        mov     bp,sp
        push    es
        push    si
;=======================================
; Initialize TRQ blocks
;=======================================
        cmp     trqiflag,0              ;table initialized?
        jne     trqi9
;
        sub     bx,bx
        mov     cx,trqdim
;
trqi:
        mov     trq.trquse[bx],0        ;clear in use flag
        add     bx,SIZE trqblok
        loop    trqi
        mov     trqiflag,1              ;only initialize once
;=======================================
; Look for a free TRQ block
;=======================================
trqi9:
        sub     bx,bx
        mov     cx,trqdim
;
trqtop:
        test    trq.trquse[bx],1        ;find unused trq blok
        jnz     trqnext
;=======================================
; Free TRQ block found
;=======================================
        cli
        mov     trq.trquse[bx],1        ;mark it used
        mov     si,[bp+p1of3]
        mov     trq.trqefws[bx],si      ;copy efw seg
        mov     es,si
        mov     si,[bp+p2of3]
        mov     trq.trqefwo[bx],si      ;copy efw ofs
        mov     word ptr es:[si],0      ;clear efw
        mov     ax,[bp+p3of3]
        mov     trq.trqcnt[bx],ax       ;copy starting time
        or      ax,ax                   ;if time==0, set efw
        jnz     trqexit
        mov     word ptr es:[si],0FFh
        jmp     trqexit
;
trqnext:
        add     bx,SIZE trqblok
        loop    trqtop
;=======================================
; No TRQ blocks free
;=======================================
        mov     bx,-1                   ;all trqs in use
;=======================================
; Exit with return code in AX
;=======================================
trqexit:
        sti
        mov     ax,bx
        pop     si
        pop     es
        pop     bp
        ret     6
_set_trq endp
;=======================================
; _set_trq_time
;
; Set TRQ timer value for an existing TRQ
; block.
;
; Model:
; procedure _set_trq_time(hx,time: integer);
;
; hx - TRQ block handle
; time - Number of tmer ticks until the TRQ expires.
;
;=======================================
        public  _set_trq_time
_set_trq_time proc far
        push    bp
        mov     bp,sp
        push    es
        mov     bx,[bp+p1of2]           ;the handle points to TRQ
        mov     ax,[bp+p2of2]           ;new time value
;=======================================
; Update TRQ block
;=======================================
        cli
        mov     trq.trqcnt[bx],ax
        or      ax,ax                   ;time==0?
        jnz     stt1
        mov     es,trq.trqefws[bx]
        mov     bx,trq.trqefwo[bx]
        mov     word ptr es:[bx],0FFh   ;set efw
stt1:
        sti
        pop     es
        pop     bp
        ret     4
_set_trq_time endp
;=======================================
; _add_trq_time
;
; Add to TRQ time value.  This is the preferred
; way of setting a new timer value, as it
; does not loose an tics.  The new value is added
; to the current TRQ timer.
;
; Model:
; procedure _add_trq_time(hx,time: integer);
;
; hx - TRQ block handle
; time - Number of timer ticks to be added to the TRQ.
;
;=======================================
        public  _add_trq_time
_add_trq_time proc far
        push    bp
        mov     bp,sp
        mov     bx,[bp+p1of2]           ;the handle points to the TRQ
        mov     ax,[bp+p2of2]           ;timer value adder
;=======================================
; Update the TRQ block
;=======================================
        cli
        add     trq.trqcnt[bx],ax
        js      short att1              ;already expired if negative
        jnz     short att2              ;not expired
att1:
        mov     es,trq.trqefws[bx]
        mov     bx,trq.trqefwo[bx]
        mov     word ptr es:[bx],0FFh   ;set efw
att2:
        sti
        pop     bp
        ret     4
_add_trq_time endp
;=======================================
; _end_trq
;
; Terminate a TRQ timer.  The designated timer
; is freed.
;
; Model:
; procedure _end_trq(hx: integer);
;
; hx - The TRQ handle.
;
;=======================================
        public  _end_trq
_end_trq proc   far
        push    bp
        mov     bp,sp
;=======================================
; Free TRQ block for re-use
;=======================================
        cli
        mov     bx,[bp+p1of1]           ;the handle points to the TRQ
        mov     trq.trquse[bx],0
        sti
        pop     bp
        ret     2
_end_trq endp
;=======================================
; _clear_efw
;
; Clear an EFW.  This provides a "safe"
; way to reset an event flag word.
;
; Model:
; procedure _clear_efw(var efw: integer);
;                      seg(efw),ofs(efw)
;                           +8       +6
; efw - flag word to be cleared
;
;=======================================
        public  _clear_efw
_clear_efw proc far
        push    bp
        mov     bp,sp
        push    es
;=======================================
; EFWs must be cleared in a disabled state
;=======================================
        mov     es,[bp+p1of2]           ;get address of EFW to be cleared
        mov     bx,[bp+p2of2]
        cli
        mov     word ptr es:[bx],0
        sti
        pop     es
        pop     bp
        ret     4
_clear_efw endp
;=======================================
CODE    ENDS
        END
