;=======================================
;
; Music Quest Programmer's ToolKit
;
; Co-processor mode second level interrupt handler
; This routine is suitable for use as
; a SLIH for the ToolKit FLIH.
;
; It provides the following functions:
;   - In memmory recording of event data
;   - Event flag words for MCC events
;   - Clock tics for the ToolKit clock services
;
; Written for use with small, medium, compacct, and large memory models
;
; Requires Turbo Assembler or Microsoft Assember 5.0 or later
;
; Supports
;   Turbo Assembler
;   Microsoft Assembler
;   Turbo C
;   Quick C and Microsoft C
;   Turbo Pascal
;   Quick Basic
;
; Copyright 1987, 1990 by Music Quest, Inc.
; All rights reserved
;
;=======================================
;
; Check for valid language OR model specification
;
        DOSSEG
        IFDEF   _SMALL_
        .model  small
        ENDIF
        IFDEF   _MEDIUM_
        .model  medium
        ENDIF
        IFDEF   _COMPACT_
        .model  compact
        ENDIF
        IFDEF   _LARGE_
        .model  large
        ENDIF
        IFDEF   _PASCAL_
        .model  medium
        ENDIF
        IFDEF   _QBASIC_
        .model  medium
        ENDIF

        IFNDEF  @codesize
        %out    ***ERROR***  Valid language or model name not specified
        .ERR
        END
        ENDIF

;=======================================
; External references
;=======================================
        IF      @codesize
        extrn   clock_tic:far           ;clock manager (mcctkclk.asm)
        extrn   mcccommand:far          ;execute MCC command (mcctkf.asm)
        ELSE
        extrn   clock_tic:near          ;clock manager (mcctkclk.asm)
        extrn   mcccommand:near         ;execute MCC command (mcctkf.asm)
        ENDIF

        .data
;=======================================
; Call parameter definitions
;=======================================
        if      @codesize GE 1
p1      equ     6                       ;parameter 1 medium, large, huge
p2      equ     8                       ;parameter 2 medium, large, huge
p3      equ     10                      ;parameter 3 medium, large, huge
        else
p1      equ     4                       ;parameter 1 small model
p2      equ     6                       ;parameter 2 small model
p3      equ     8                       ;parameter 3 small model
        endif
;=======================================
; SLIH state data
;=======================================
mqstatew        dw      new_event       ;current, state of recording
rstatec         db      2               ;current running status data count
data_cnt        db      2,2,2,2,1,1,2,0 ;data count for status codes
res_cnt         db      0               ;residual data count expected
data_in         db      0               ;midi-in data
;
eventp          dw      DGROUP:event_buff ;event assembly area
event_buff      db      0,0,0,0,0,0,0,0 ;the assembly area
                db      0,0,0,0,0,0,0,0
spp_byte1       db      0               ;first byte of SPP
eot_mark        db      0,0FCh          ;MCC end of track marker
;
rbuff           dw      0               ;recording buffer
rbuffseg        dw      0               ;
rbx             dw      0               ;recording index
rbsize          dw      0               ;buffer size (should be 1K multiple)
pagerdy         db      0               ;1K page ready flag
                IF      @codesize
tdr_rtn         dd      0               ;tdr processing routine
cdr_rtn         dd      0               ;cdr processing routine
                ELSE
tdr_rtn         dw      0               ;tdr processing routine
cdr_rtn         dw      0               ;cdr processing routine
                ENDIF
eventcases      dw      toflow_event    ;jump table for mcc msgs f8-ff
                dw      cdr_event
                dw      $fa_event
                dw      $fb_event
                dw      $fc_event
                dw      $fd_event
                dw      $fe_event
                dw      $ff_event
;=======================================
; Event Flag words
;=======================================
track_efw       dw      0,0,0,0,0,0,0,0 ;EFWs for each track
conductor_efw   dw      0               ;conductor EFW
ackn_efw        dw      0               ;command acknowledge efw
playend_efw     dw      0               ;all play tracks ended
recordend_efw   dw      0               ;recording completed
me_efw          dw      0               ;measure end
clk_efw         dw      0               ;clock-to-PC event
cpt_efw         dw      0               ;cue point event
spp_efw         dw      0               ;SPP event
smpte_efw       dw      0               ;SMPTE frame address received
realtime_efw    dw      0               ;MIDI realtime EFW
strk_end_efw    dw      0               ;SMPTE track end EFW
rbfull_flag     dw      1               ;buffer full flag (=1 since no rec buff)
;=======================================
; Decoded values
;=======================================
last_spp        dw      0               ;decoded value from last SPP
last_smpte      db      0,0,0,0         ;last SMPTE frame address
;=======================================
        .code
;=======================================
; _coproc_slih
;
; Co-processor mode SLIH entry point
;
; Inputs:
;   al = received character
; Outputs:
;   None.
; Notes:
;   Clock tic interrupts are passed to
;       the clock services manager.
;   Event flag words are posted when
;       event messages arrive.
;=======================================
        public  _coproc_slih
_coproc_slih proc
        sub     ah,ah
        mov     data_in,al              ;save incoming char
        mov     dx,ax
        jmp     word ptr mqstatew       ;branch directly to current state routine
;=======================================
; Start of a new event
;=======================================
new_event:
        cmp     al,0F0h                 ;midi data or mark?
        jb      midi_event              ;yes
        cmp     al,0F8h                 ;track data request
        jb      tdr_event               ;yes
;
        sub     ax,00F8h                ;compute index to jump table
        shl     ax,1
        mov     bx,ax
        jmp     word ptr eventcases[bx] ;jump to case handler
;=======================================
; Track data request event
;=======================================
tdr_event:
;=======================================
; If a TDR handler has been installed, it
; is called as an extension of the SLIH.
;=======================================
        and     ax,0007h                ;isolate track number
        mov     bx,ax
        add     bx,bx
        mov     track_efw[bx],1         ;post efw
        IF      @codesize
        cmp     word ptr [tdr_rtn+2],0  ;TDR handler installed?
        je      tdr_nxt                 ;no
        call    dword ptr [tdr_rtn]     ;yes, call TDR handler
        ELSE
        cmp     tdr_rtn,0               ;TDR handler installed?
        je      tdr_nxt                 ;no
        call    word ptr [tdr_rtn]      ;yes, call TDR handler
        ENDIF
tdr_nxt:
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; Midi event
;=======================================
midi_event:
        call    rec_data                ;record timing byte
        mov     mqstatew,offset midi1   ;next state
        jmp     rpihxt
;=======================================
; MIDI event or mark
;=======================================
midi1:
        test    al,80h                  ;new running status?
        jz      same_event              ;no, use running status
rih_new:
        push    ax
        call    rec_data                ;record first (status) byte
        pop     bx
;
        cmp     bl,0F0h                 ;normal midi event?
        jl      rih_midi                ;yes
        cmp     bl,0F9h                 ;measure end?
        je      rih_new2                ;yes
        cmp     bl,0FCh                 ;end of recording mark?
        je      rih_new1                ;yes
;
;=======================================
; First byte (possibly new status)
;=======================================
rih_midi:
        and     bx,0070h                ;isolate state
        mov     cl,4
        shr     bx,cl                   ;new running status
        mov     ah,data_cnt[bx]         ;get count for this code
;
        mov     res_cnt,ah              ;save new count
        mov     rstatec,ah
        mov     mqstatew,offset rcvdata ;establish new state to receive data
        jmp     rpihxt
;
rih_new1:
        mov     recordend_efw,1         ;post end efw
        mov     mqstatew,offset new_event ;establish new state for next event
        jmp     rpirec                  ;go record this event
rih_new2:
        mov     me_efw,1                ;post end efw
        mov     mqstatew,offset new_event ;establish new state for next event
        jmp     rpirec                  ;go record this event
;
; receive data after status byte
;
rcvdata:                                ;receive event data 80-e0
        call    rec_data                ;record data byte
        dec     res_cnt                 ;decrement
        jz      rcvd1                   ;all data rec'd
        jmp     rpihxt
rcvd1:
        mov     mqstatew,offset new_event ;new event, with timing byte is next
        jmp     rpirec                  ;go record this event
;
; record data for same event (running status)
;
same_event:
        mov     al,data_in              ;record first byte of running status
        call    rec_data
        mov     ah,rstatec
        dec     ah
        jz      rpi_rs                  ;single byte event
        mov     res_cnt,ah              ;remaining count of data to be rec'd
        mov     mqstatew,offset rcvdata ;next state
        jmp     rpihxt
rpi_rs:
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpirec                  ;go record this event
;=======================================
; Timer overflow event
;=======================================
toflow_event:
        mov     al,data_in
        call    rec_data                ;record event
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpirec                  ;go record this event
;=======================================
; Conductor data request event
;=======================================
cdr_event:
        mov     conductor_efw,1         ;post efw
        mov     mqstatew,offset new_event ;new event is next state
        IF      @codesize
        cmp     word ptr [cdr_rtn+2],0  ;CDR handler installed?
        je      cdr_nxt                 ;no
        call    dword ptr [cdr_rtn]     ;yes, call CDR handler
        ELSE
        cmp     cdr_rtn,0               ;CDR handler installed?
        je      cdr_nxt                 ;no
        call    word ptr [cdr_rtn]      ;yes, call CDR handler
        ENDIF
cdr_nxt:
        jmp     rpihxt
;=======================================
; Command acknowledge event
;=======================================
$fe_event:
        inc     ackn_efw                ;post efw
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; Host clock event
;=======================================
$fd_event:
        inc     clk_efw                 ;post efw
        call    clock_tic               ;notify clock manager
        mov     mqstatew,offset new_event ;new event is next state
        jmp     rpihxt
;=======================================
; System message event
;=======================================
$ff_event:
        mov     al,data_in
        call    rec_data                ;record sysex data
        mov     mqstatew,offset system_event ;next state
        jmp     rpihxt
;=======================================
; First byte of system event
;
; Note: Beware, system events can be recorded
; but they can not be played back through
; a play track.
;=======================================
system_event:
        call    rec_data                ;record system msg marker
        cmp     al,0FFh                 ;extended system message?
        jne     sys_ev                  ;no
        mov     mqstatew,offset sys_eventx ;next state for extended msg
        jmp     rpihxt
sys_ev:
        cmp     al,0F0h                 ;sysex?
        jne     sys_ev1                 ;no
        mov     mqstatew,offset sysex_event ;next state
        jmp     rpihxt
sys_ev1:
        cmp     al,0F2h                 ;SPP?
        jne     sys_f3                  ;no
        mov     mqstatew,offset rcvspp1 ;set up to receive next SPP byte
        jmp     rpihxt
sys_f3:
        mov     ah,1                    ;try F3=song select
        cmp     al,0F3h
        jne     sys_rt                  ;maybe real time
;
        mov     res_cnt,ah              ;expected data length
        mov     mqstatew,offset rcvdata ;set up to receive the rest of data
        jmp     rpihxt
;
sys_rt:
        cmp     al,0FAh                 ;real time?
        jb      sys_ignore              ;no, ignore
        cmp     al,0FDh
        jnb     sys_ignore              ;nope
        sub     ah,ah                   ;yes, set the EFW
        mov     realtime_efw,ax
sys_ignore:
        mov     mqstatew,offset new_event ;next state
        jmp     rpihrsb
;=======================================
; Continuation of system exclusive
;=======================================
sysex_event:                                    ;sys ex
        call    rec_data                ;record sysex data
        test    data_in,80h             ;EOX or new status?
        jnz     short sysex_end         ;yes
        jmp     rpihxt                  ;no, continue this state
sysex_end:
        cmp     byte ptr event_buff+2,7Fh ;possible SMPTE msg?
        jne     sysex_end9              ;no
        mov     ax,word ptr event_buff+4 ;get sub id
        cmp     ax,0101h                ;SMPTE msg?
        jne     sysex_end9              ;no
        mov     ax,word ptr event_buff+6 ;save the frame address data
        mov     word ptr last_smpte,ax
        mov     ax,word ptr event_buff+8
        mov     word ptr last_smpte+2,ax
        mov     smpte_efw,1             ;signal frame received
        mov     mqstatew,offset new_event ;next state
        jmp     short rpihrsb
;
sysex_end9:
        mov     mqstatew,offset new_event ;next state
        jmp     short rpirec
;=======================================
; Continuation of SPP
;=======================================
rcvspp1:
        and     al,7Fh                  ;isolate low order 7 bits
        mov     spp_byte1,al            ;save data
        mov     mqstatew,offset rcvspp2 ;set up to receive next SPP byte
        jmp     short rpihxt
rcvspp2:
        mov     cl,7                    ;left adjust high order bits
        sub     ah,ah
        shl     ax,cl
        or      al,spp_byte1
        mov     last_spp,ax             ;save calculated SPP
        mov     spp_efw,1               ;set EFW to signal SPP received
        mov     mqstatew,offset new_event ;next state
        jmp     short rpihrsb
;=======================================
; Continuation extended system message
;=======================================
sys_eventx:
        or      al,al                   ;cue point hit?
        jnz     sys_xm1                 ;no
        mov     cpt_efw,1               ;set cue point EFW
        jmp     short sys_xmx
sys_xm1:
        cmp     al,1                    ;end of SMPTE track?
        jne     sys_xm2                 ;no
        mov     strk_end_efw,1          ;set SMPTE track end EFW
sys_xm2:
sys_xmx:
        mov     mqstatew,offset new_event ;next state
        jmp     short rpihrsb
;=======================================
; All end event
;=======================================
$fc_event:
        mov     playend_efw,1
        jmp     short rpihxt
;=======================================
; Invalid/unexpected events
;=======================================
$fa_event:
$fb_event:
        jmp     short rpihxt
;=======================================
; Record the assembled event
;=======================================
rpirec:
        call    rec_asmdata
rpihrsb:
        mov     eventp,offset DGROUP:event_buff
;=======================================
; The grand exit from interrupt handling!
;=======================================
rpihxt:
        ret
_coproc_slih endp
;=======================================
; rec_data
;
; Assemble event data into an event
;
; Inputs:
;   al = data to be recorded
; Outputs:
;   None.
;
; Note: assumes interrupts disabled
;
;=======================================
rec_data proc   near
        mov     bx,eventp               ;get ptr to assembly area
        mov     [bx],al                 ;assemble byte
        inc     eventp                  ;bump assembly ptr
        ret
rec_data endp
;=======================================
; rec_asmdata
;
; Record assembled event in the recording buffer
;
; Inputs:
;   al = data to be recorded
; Outputs:
;   None.
;
; Note: assumes interrupts disabled
;
;=======================================
rec_asmdata proc   near
        push    es
;
        cmp     rbfull_flag,0           ;any room?
        jne     rec_1                   ;no
;
        mov     es,rbuffseg             ;set up ds:si and es:di for move
        mov     di,rbuff
        add     di,rbx
        mov     si,offset DGROUP:event_buff
;
        mov     cx,eventp
        sub     cx,si                   ;length of event
        add     rbx,cx                  ;bump record offset
        sub     rbsize,cx               ;adjust remaining space
        rep     movsb                   ;put event in record buffer
;
        cmp     rbsize,12               ;record buffer filled?
        jnb     rec_1                   ;no
;
        mov     si,offset DGROUP:eot_mark ;force an end mark on the track
        mov     cx,2
        add     rbx,cx
        rep     movsb
        mov     rbfull_flag,1           ;buffer full, stop recording
rec_1:
        pop     es
        ret
rec_asmdata endp
;=======================================
; _rec_overflow
;
; Test for recording buffer overflow
;
; C Model:
;       rc=rec_overflow();
;               int rc;
;               1 = buffer overflow/full
;               0 = buffer not full
;
;=======================================
        public  _rec_overflow
_rec_overflow proc
        cli
        mov     ax,rbfull_flag
        mov     rbfull_flag,0
        sti
        ret
_rec_overflow endp
;=======================================
; _track_efw
;
; Track Event Flag Word state
;
; C Model:
;       rc=track_efw(track);
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _track_efw
_track_efw proc
        push    bp
        mov     bp,sp
        mov     bx,[bp+p1]
        add     bx,bx
        cli
        mov     ax,track_efw[bx]
        mov     track_efw[bx],0
        sti
        pop     bp
        ret
_track_efw endp
;=======================================
; _conductor_efw
;
; Conductor EFW status
;
; C Model:
;       rc=conductor_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _conductor_efw
_conductor_efw proc
        cli
        mov     ax,conductor_efw
        mov     conductor_efw,0
        sti
        ret
_conductor_efw endp
;=======================================
; _clock_efw
;
; Clock-to-PC EFW status
;
; C Model:
;       rc=clock_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _clock_efw
_clock_efw proc
        cli
        mov     ax,clk_efw
        mov     clk_efw,0
        sti
        ret
_clock_efw endp
;=======================================
; _ackn_efw
;
; Command acknowledge EFW status
;
; C Model:
;       rc=ackn_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _ackn_efw
_ackn_efw proc
        cli
        mov     ax,ackn_efw
        or      ax,ax
        jz      ackx
        dec     ackn_efw
ackx:
        sti
        ret
_ackn_efw endp
;=======================================
; _playend_efw
;
; All playback tracks ended EFW status
;
; C Model:
;       rc=playend_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _playend_efw
_playend_efw proc
        cli
        mov     ax,playend_efw
        mov     playend_efw,0
        sti
        ret
_playend_efw endp
;=======================================
; _recordend_efw
;
; Recording ended EFW status
;
; C Model:
;       rc=recordend_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _recordend_efw
_recordend_efw proc
        cli
        mov     ax,recordend_efw
        mov     recordend_efw,0
        sti
        ret
_recordend_efw endp
;=======================================
; _measurend_efw
;
; Measure end EFW status
;
; C Model:
;       rc=measurend_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _measurend_efw
_measurend_efw proc
        cli
        mov     ax,me_efw
        mov     me_efw,0
        sti
        ret
_measurend_efw endp
        IFDEF   _QBASIC_
;=======================================
; _spp_efw
;
; SPP EFW status
;
; Basic Model:
;       spp_efw(VARPTR(efw),VARPTR(spp))
;               efw%
;               0 = event not posted
;               1 = event posted
;               spp%
;
;=======================================
        public  _spp_efw
_spp_efw proc far
        push    bp
        mov     bp,sp
        cli
        mov     ax,spp_efw              ;return EFW status
        mov     spp_efw,0
        or      ax,ax                   ;EFW set?
        jz      sppefwx                 ;no
        mov     bx,[bp+p2]              ;get ptr to result
        mov     cx,last_spp
        mov     [bx],cx
sppefwx:
        sti
        mov     bx,[bp+p1]              ;get ptr to result
        mov     [bx],ax
        pop     bp
        ret
_spp_efw endp
;=======================================
; _smpte_efw
;
; SMPTE frame message EFW status
;
; Basic Model:
;       smpte_efw(VARPTR(efw),VARPTR(fid) or SADD(fid))
;               efw%
;               0 = event not posted
;               1 = event posted
;               VARPTR(fid) if fid is fixed string
;               SADD(fid) if fid is variable string
;
;=======================================
        public  _smpte_efw
_smpte_efw proc far
        push    bp
        mov     bp,sp
        cli
        mov     ax,smpte_efw            ;return EFW stae
        mov     smpte_efw,0
        or      ax,ax                   ;if EFW set, return address
        jz      smpte_efwx
        mov     bx,[bp+p2]              ;get ptr to result
        mov     cx,word ptr last_smpte  ;move last frame address to result
        mov     word ptr [bx],cx
        mov     cx,word ptr last_smpte+2
        mov     word ptr [bx+2],cx
smpte_efwx:
        sti
        mov     bx,[bp+p1]              ;get ptr to result
        mov     [bx],ax
        pop     bp
        ret
_smpte_efw endp
        ELSE
;=======================================
; _spp_efw
;
; SPP EFW status
;
; C Model:
;       rc=spp_efw(&spp);
;               int rc;
;               0 = event not posted
;               1 = event posted
;               int *spp;
;
;=======================================
        public  _spp_efw
_spp_efw proc
        push    bp
        mov     bp,sp
        cli
        mov     ax,spp_efw              ;return EFW status
        mov     spp_efw,0
        or      ax,ax                   ;EFW set?
        jz      sppefwx                 ;no
        mov     cx,last_spp
        IF      @datasize
        les     bx,[bp+p1]              ;get ptr to result
        mov     es:[bx],cx
        ELSE
        mov     bx,[bp+p1]              ;get ptr to result
        mov     [bx],cx
        ENDIF
sppefwx:
        sti
        pop     bp
        ret
_spp_efw endp
;=======================================
; _smpte_efw
;
; SMPTE frame message EFW status
;
; C Model:
;       rc=smpte_efw(fid);
;               int rc;
;               0 = event not posted
;               1 = event posted
;               char fid[4];
;
;=======================================
        public  _smpte_efw
_smpte_efw proc
        push    bp
        mov     bp,sp
        cli
        mov     ax,smpte_efw            ;return EFW stae
        mov     smpte_efw,0
        or      ax,ax                   ;if EFW set, return address
        jz      smpte_efwx
        IF      @datasize
        les     bx,[bp+p1]              ;get ptr to result
        mov     cx,word ptr last_smpte  ;move last frame address to result
        mov     word ptr es:[bx],cx
        mov     cx,word ptr last_smpte+2
        mov     word ptr es:[bx+2],cx
        ELSE
        mov     bx,[bp+p1]              ;get ptr to result
        mov     cx,word ptr last_smpte  ;move last frame address to result
        mov     word ptr [bx],cx
        mov     cx,word ptr last_smpte+2
        mov     word ptr [bx+2],cx
        ENDIF
smpte_efwx:
        sti
        pop     bp
        ret
_smpte_efw endp
        ENDIF
;=======================================
; _cuepoint_efw
;
; Cue Point EFW status
;
; C Model:
;       rc=cuepoint_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _cuepoint_efw
_cuepoint_efw proc
        cli
        mov     ax,cpt_efw              ;return EFW status
        mov     cpt_efw,0
        sti
        ret
_cuepoint_efw endp
;=======================================
; _realtime_efw
;
; MIDI realtime command reception status
;
; C Model:
;       rc=realtime_efw();
;               int rc;
;               0 = event not posted
;               0xFA = start
;               0xFB = continue
;               0xFC = stop
;
;=======================================
        public  _realtime_efw
_realtime_efw proc
        cli
        mov     ax,realtime_efw         ;return EFW status
        mov     realtime_efw,0
        sti
        ret
_realtime_efw endp
;=======================================
; _strk_end_efw
;
; SMPTE track end status
;
; C Model:
;       rc=strk_end_efw();
;               int rc;
;               0 = event not posted
;               1 = event posted
;
;=======================================
        public  _strk_end_efw
_strk_end_efw proc
        cli
        mov     ax,strk_end_efw         ;return EFW status
        mov     strk_end_efw,0
        sti
        ret
_strk_end_efw endp
;=======================================
; _rec_bytes
;
; Return amount of data recorded in current
; recording buffer.
;
; C Model:
;       count=rec_bytes();
;               int count;
;=======================================
        public  _rec_bytes
_rec_bytes proc
        mov     ax,rbx
        ret
_rec_bytes endp
;=======================================
; _rec_init
;
; Intialize MIDI event recording
;
; Note: Buffer size is expected to be
; at least 16 bytes smaller than the record
; buffer, so that recording end can be
; correctly stored in track.
;
; C Model:
;       rec_init(buffseg,buffoff,size);
;               unsigned buffseg;
;                       buffer segment address
;               unsigned buffoff;
;                       buffer offset address
;               unsigned size;
;                       size of buffer
;
;=======================================
        public  _rec_init
_rec_init proc
        push    bp
        mov     bp,sp
        cli
        mov     ax,[bp+p1]              ;buffer segment address
        mov     rbuffseg,ax
        mov     ax,[bp+p2]              ;buffer address/size
        mov     rbuff,ax
        mov     ax,[bp+p3]
        mov     rbsize,ax
        sub     ax,ax
        mov     rbx,ax                  ;reset buffer index
        mov     pagerdy,al              ;reset page ready
; clear all efws
        mov     rbfull_flag,ax
        mov     ackn_efw,ax
        mov     playend_efw,ax
        mov     recordend_efw,ax
        mov     conductor_efw,ax
        mov     me_efw,ax
;
        mov     cx,8
        mov     bx,offset DGROUP:track_efw
ri1:
        mov     word ptr [bx],ax
        add     bx,2
        loop    ri1
;
        sti
        pop     bp
        ret
_rec_init endp
;=======================================
; _set_tdr_handler
;
; Install a TDR handling routine to be used
; during playback.
;
; C Model:
;       void set_tdr_handler(routine);
;               int (*routine)();
;=======================================
        public  _set_tdr_routine
_set_tdr_routine proc
        push    bp
        mov     bp,sp
        IF      @codesize
        les     bx,[bp+p1]              ;get address of routine
        mov     word ptr [tdr_rtn],bx
        mov     word ptr [tdr_rtn+2],es
        ELSE
        mov     ax,[bp+p1]              ;get addres of routine
        mov     tdr_rtn,ax
        ENDIF
        pop     bp
        ret
_set_tdr_routine endp
;=======================================
; _set_cdr_handler
;
; Install a CDR handling routine to be used
; during playback.
;
; C Model:
;       void set_cdr_handler(routine);
;               int (*routine)();
;=======================================
        public  _set_cdr_routine
_set_cdr_routine proc
        push    bp
        mov     bp,sp
        IF      @codesize
        les     bx,[bp+p1]              ;get address of routine
        mov     word ptr [cdr_rtn],bx
        mov     word ptr [cdr_rtn+2],es
        ELSE
        mov     ax,[bp+p1]              ;get addres of routine
        mov     cdr_rtn,ax
        ENDIF
        pop     bp
        ret
_set_cdr_routine endp
;
        END
