/*   ARIAINIT.C    --   C Program to initialize the ARIA sound card.
                        Just like the original, this program is very
                        picky about the format of the config file.

     This program may be compiled with Microsoft Quick-C (2.?) or MSVC++

     11/4/94  Initial Creation
    11/10/94  1.00 Released
    12/15/94  Removed copyright
*/

#include <stdio.h>
#include <string.h>

void      initConfig (int label, int index);
void      writeConfig(unsigned aReg, unsigned value);
unsigned  readBack(unsigned rba);

/*
   These are the config labels in the Aria config file:
*/
char aBas[]= "ARIA_BASE";
char aIrq[]= "ARIA_IRQ";
char aDma[]= "ARIA_DMA";
char bBas[]= "SB_BASE";
char bIrq[]= "SB_IRQ";
char bDma[]= "SB_DMA";
char mBas[]= "MIDI_BASE";
char mIrq[]= "MIDI_IRQ";
char sBas[]= "SCSI_BASE";
char sIrq[]= "SCSI_IRQ";
char lBas[]= "ADLIB_BASE";
char jBas[]= "JOYSTICK_BASE";
char jSpd[]= "JOYSTICK_SPEED";
char rSiz[]= "ROM_SIZE";

char *cfgLabl[]= { aBas, aIrq, aDma, bBas, bIrq, bDma, mBas,
                   mIrq, sBas, sIrq, lBas, jBas, jSpd, rSiz };

int labelCnt= (sizeof(cfgLabl) / sizeof(cfgLabl[0]) );

/*
   These are the config values for each label in the Aria config file:
*/
char *aBv[]= { "290",    "280",   "2A0",   "2B0",    "NONE",   NULL };
char *aIv[]= { "10",     "11",    "12",    "NONE",    NULL };
char *aDv[]= { "5",      "6",     "NONE",   NULL };
char *bBv[]= { "220",    "240",   "NONE",   NULL };
char *bIv[]= { "7",      "5",     "9",     "2",      "NONE",   NULL };
char *bDv[]= { "1",      "0",     "NONE",   NULL };
char *mBv[]= { "330",    "320",   "NONE",   NULL };
char *mIv[]= { "9",      "2",     "5",     "7",      "NONE",   NULL };
char *sBv[]= { "CA000",  "C8000", "CE000", "DE000",  "NONE",   NULL };
char *sIv[]= { "NONE",   "11",    "10",    "12",      NULL };
char *lBv[]= { "388",    "NONE",  NULL };
char *jBv[]= { "200",    "NONE",  NULL };
char *jSv[]= { "NORMAL", "FAST",  NULL };
char *rSv[]= { "512",    "1024",  NULL };

char * *cfgValu[]= { aBv, aIv, aDv, bBv, bIv, bDv, mBv, mIv,
                     sBv, sIv, lBv, jBv, jSv, rSv };
/*
   These are bit constants that corespond to labels in the config file.
*/
unsigned char aBb[]= { 0xA4, 0xA0, 0xA8, 0xAC, 0x00 };
unsigned char aIb[]= { 0x04, 0x05, 0x06, 0x00  };
unsigned char aDb[]= {    0,    0,    0  };
unsigned char bBb[]= { 0x88, 0x90, 0x00  };
unsigned char bIb[]= { 0x03, 0x02, 0x01, 0x01, 0x00 };
unsigned char bDb[]= {    0,    0,    0  };
unsigned char mBb[]= { 0xCC, 0xC8, 0x00  };
unsigned char mIb[]= { 0x01, 0x01, 0x02, 0x03, 0x00 };
unsigned char sBb[]= { 0x34, 0x24, 0x14, 0x04, 0x00 };
unsigned char sIb[]= { 0x00, 0x05, 0x04, 0x06  };
unsigned char lBb[]= { 0xE2, 0x00  };
unsigned char jBb[]= { 0x80, 0x00  };
unsigned char jSb[]= {    0,    0  };
unsigned char rSb[]= {    0,    0  };

unsigned char *cfgBits[]= { aBb, aIb, aDb, bBb, bIb, bDb, mBb, mIb,
                            sBb, sIb, lBb, jBb, jSb, rSb };
/*
   These are and/or mask constants coresponding to labels in the config file:
*/
unsigned char aBm[]= { 0xBF,0xA0, 0xBF,0xA0, 0xBF,0xA0, 0xBF,0xA0, 0x88,0x00 };
unsigned char aIm[]= { 0xF8,0x00, 0xF8,0x00, 0xF8,0x00, 0xF8,0x00  };
unsigned char aDm[]= { 0xF7,0x10, 0xFF,0x18, 0xEF,0x00  };
unsigned char bBm[]= { 0xDF,0xC0, 0xDF,0xC0, 0x88,0x00  };
unsigned char bIm[]= { 0xF8,0x00, 0xF8,0x00, 0xF8,0x00, 0xF8,0x00, 0xF8,0x00 };
unsigned char bDm[]= { 0xFF,0x18, 0xF7,0x10, 0xEF,0x00  };
unsigned char mBm[]= { 0x3F,0x20, 0x3F,0x20, 0x88,0x00  };
unsigned char mIm[]= { 0xF8,0x00, 0xF8,0x00, 0xF8,0x00, 0xF8,0x00, 0xFF,0x00 };
unsigned char sBm[]= { 0xCF,0x00, 0xCF,0x00, 0xCF,0x00, 0xFB,0x00  };
unsigned char sIm[]= { 0xF8,0x00, 0xF8,0x00, 0xF8,0x00, 0xF8,0x00  };
unsigned char lBm[]= { 0xBF,0xA0, 0x9F,0x00  };
unsigned char jBm[]= { 0x3F,0x20, 0x9F,0x00  };
unsigned char jSm[]= { 0xFF,0x08, 0xF7,0x00  };
unsigned char rSm[]= { 0xBF,0x00, 0xFF,0x40  };

unsigned char *cfgMask[]= { aBm, aIm, aDm, bBm, bIm, bDm, mBm, mIm,
                            sBm, sIm, lBm, jBm, jSm, rSm };

int cfgFunc[]= { 0x111, 0x11, 0x11, 0x113, 0x13, 0x13, 0x117,
                 0x17, 0x0E, 0x19, 0x115, 0x19, 0x0E, 0x0E };

#define bufSize  32

unsigned char buf1[bufSize];
unsigned rba= 0x300;

void main(argc, argv)
int argc;
char *argv[];
{
FILE *fp1;
unsigned end;
char *filVal, **ptr;
int err, label, index;

 printf("\n  *** Aria port configuration program ***");
 printf("\nVersion 1.00, Thomas K. Myers 1994\n");
 err= 0;
 if (argc >= 2) {                                     /* Validate cmd parms */
    if ((fp1= fopen(argv[1], "rt")) == (FILE *)0) {
       printf("\nERROR: Opening ARIA config file!");
       err= 1;
       }
    if (argc == 3) {                                /* Process optional rba */
       sscanf(argv[2], "%X", &rba);
       if ((rba < 0x280) || (rba > 0x330)) {
          printf("\nERROR: Invalid Read Back Address!");
          err=2;
          }
       else printf("\nUsing rba= %X", rba);
       }
    outp(0x204, 0x4C);                             /* Begin config sequence */
    outp(0x205, 0x42);
    outp(0x206, 0x00);
    writeConfig(0x000F, 0x00);
    writeConfig(0x0002, rba >> 2);
    /*
      For each line of the file,  figure out which label it is,
      and the index of it's parameter value.  Thus the ordering
      of say,  ARIA_BASE and ROM_SIZE  may be rearranged in the
      file and still send the same info to the  Aria ports,  it
      may well be that the sequence is also important. I do not
      know if it is or not. 11/6/94
    */
    while (! err && fgets(buf1, sizeof(buf1) - 2, fp1)) {
       for (label= 0; (label < labelCnt) &&                       /* label? */
                      ! strstr(buf1, cfgLabl[ label ]); label++);
       if (label > labelCnt) {
          printf("\nERROR: Illegal parameter specified!");
          err= 3;
          }
       else {
          buf1[ strlen(buf1) - 1 ] = '\0';          /* Find parameter index */
          filVal= strchr(buf1, '=') + 1;
          ptr= cfgValu[ label ];
          for (index= 0; ptr[ index ] &&
                         strcmpi(filVal, ptr[ index ]); index++);
          if (ptr[ index ] == NULL) {
             printf("\nERROR: Invalid parameter value!");
             err= 4;
             }
          initConfig(label, index);                   /* Do some port magic */
          }
       }
    fclose(fp1);
    end= readBack(0x000F);                           /* End config sequence */
    writeConfig(0x000F, end | 0x80);
    if (! err) printf("\n\nAria board configured.");
    inp(0x200);
    }
  else printf("\nUSAGE: ARIAinit <cfgFile> [readBack port, 280H - 330H ]");
}

/*
  This is where things happen. It is all table driven, so the purpose of
  the values is not known. My guess is that the "func" value is the Aria
  chip's register that establishes a functional port configuration. Thus
  there may be a corespondence between func=15 and ADLIB_BASE.  The pur-
  pose of the rba is also unclear. Since the Aria offers a wide range of
  port config options which could be conflicting,  perhaps the rba value
  is used to prohibit/correct conflicting settings.  I can see a corela-
  tion of how the base port address for ARIA, SB, MIDI and ADLIB are set
  vs. the other parameters. The base is set using the code within the IF
  block, while the other parameters are set using the ELSE code.
*/
void initConfig (label, index)
int label, index;
{
unsigned char *ptr;
unsigned int  and, or, bits, func, rbaIn;

 func= cfgFunc[ label ];                          /* Get values from tables */

 ptr= cfgBits[ label ];
 bits= ptr[ index ];

 ptr= cfgMask[ label ];
 and= ptr[ index * 2 ];
 or= ptr[ (index * 2) + 1 ];

 if (func & 0x100) {                         /* "BASE" functions output the */
    func &= ~0x100;                         /* bits first, then do the rba. */
    if (bits) writeConfig(func - 1, bits);     /* And, bits go out func - 1 */
    }
 else or |= bits;                            /* The others do the rba thing */
                                         /* first, then add the bits to it. */
 rbaIn= readBack(func);
 writeConfig(func, (rbaIn & and) | or);

 printf("\n%14.14s: Index=%d func=%0.4X Bits=%0.2X rbaIn=%0.2X and=%0.2X Or=%0.2X",
    cfgLabl[ label ], index,      func,      bits,      rbaIn,      and,       or);
}

/*
  Function to write config port values to config port addresses
*/
void writeConfig(aReg, value)
unsigned aReg, value;
{
 outpw(0x200, aReg);
 outp(0x201, value);
}

/*
  Function to read back config port values
*/
unsigned readBack(aReg)
unsigned aReg;
{
 outp(0x200, aReg);
 return inp(rba);
}
