// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : t-show.cpp
//
//  DESCRIPTION :
//         Class TuringMachine : show methods (Implementation)
//
// ##############################################################


// =================
#include "turing-s.h"
// =================

#define	SETW_RULE_NUMBER	4
#define	RESULTING_text		"Resulting"

// =========
class less1_string
{
  public:
    bool operator()(const string& inst1_i, const string& inst2_i);
};

bool less1_string:: operator()(const string& inst1_i, const string& inst2_i)
{
  if (inst1_i.size() < inst2_i.size()) return true;
  if (inst2_i.size() < inst1_i.size()) return false;

  return (inst1_i < inst2_i);
}


// =========
void TuringMachine::show_detail_description_of_program_states (const string& msg_i) const
{
  cout << "\t   ====== Detailed Description of Program States ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');

  cout << endl;

map<state_t, string, less1_string> tmp_map;
typedef map<state_t, string, less1_string>::value_type value_type;
pair<map<state_t, string, less1_string>::iterator, bool> couple;

const vector<state_t>	tmp1_vector (get_all_user_defined_program_states ()); 
const vector<state_t>	tmp2_vector (get_all_extra_program_states());

  for (size_t i = 0; i < tmp1_vector.size(); i++)
  {
    couple = tmp_map.insert (value_type(tmp1_vector[i], get_descr_of_program_state(tmp1_vector[i], true)));
    assert (couple.second);
  }

  for (size_t i = 0; i < tmp2_vector.size(); i++)
  {
    couple = tmp_map.insert (value_type(tmp2_vector[i], get_descr_of_program_state(tmp2_vector[i], true)));
    assert (couple.second);
  }

map<state_t, string>::const_iterator	pos_iter;

  for (pos_iter = tmp_map.begin(); pos_iter != tmp_map.end(); pos_iter++)
  {
    cout << "\t"
         << setw (max_size_of_program_states_)
         << pos_iter->first.c_str() 
         << "  " 
         << pos_iter->second.c_str() 
         << endl;
  }

  cout << endl;
}

// =========
void TuringMachine::show_descr (const string& msg_i) const
{
  cout << "\t   ====== Description ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');

  if (descr_.empty())
  {
    cout << "No description" << endl;
  }

  for (size_t i = 0; i < descr_.size(); i++)
  {
    copy (descr_[i].begin(), descr_[i].end(), ostream_iterator<string> (cout, " "));
    cout << endl;
  }

}

// =========
void TuringMachine::show_user_defined_program_states (const string& msg_i) const
{
  cout << "\t   ====== User-Defined Program States Definition ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');

const string str_tail(" program states");
string text_user_defined_initial_program_states (get_descr_of_user_defined_program_state_kind_s (USER_DEFINED_PROGRAM_STATE_KIND__INITIAL) + str_tail);
string text_user_defined_halting_program_states (get_descr_of_user_defined_program_state_kind_s (USER_DEFINED_PROGRAM_STATE_KIND__HALTING) + str_tail);
string text_user_defined_internal_program_states (get_descr_of_user_defined_program_state_kind_s (USER_DEFINED_PROGRAM_STATE_KIND__INTERNAL) + str_tail);
size_t text_max_size = 0;

  text_max_size	= MAX_VALUE(text_max_size, text_user_defined_initial_program_states.size());
  text_max_size	= MAX_VALUE(text_max_size, text_user_defined_halting_program_states.size());
  text_max_size	= MAX_VALUE(text_max_size, text_user_defined_internal_program_states.size());


  cout << setw(text_max_size) << left << text_user_defined_initial_program_states.c_str() << " : ";
  for (size_t i = 0; i < user_defined_initial_program_states_.size(); i++)
  {
    cout << setw (max_size_of_user_defined_program_states_) << user_defined_initial_program_states_[i].c_str() << " ";
  }
  cout << endl;

  cout << setw(text_max_size) << left << text_user_defined_halting_program_states.c_str() << " : ";
  for (size_t i = 0; i < user_defined_halting_program_states_.size(); i++)
  {
    cout << setw (max_size_of_user_defined_program_states_) << user_defined_halting_program_states_[i].c_str() << " ";
  }
  cout << endl;

  cout << setw(text_max_size) << left << text_user_defined_internal_program_states.c_str() << " : ";
  for (size_t i = 0; i < user_defined_internal_program_states_.size(); i++)
  {
    cout << setw (max_size_of_user_defined_program_states_) << user_defined_internal_program_states_[i].c_str() << " ";
  }
  cout << endl;


} // show_user_defined_program_states


// =========
void TuringMachine::show_extra_program_states (const string& msg_i) const
{
  cout << "\t   ====== Extra Program States Definition ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');

const string str_tail (" states"); 
// const string text_neutral_program_states (get_descr_of_neutral_program_state_kind_s (NEUTRAL_PROGRAM_STATE_KIND__FIRST));
const string text_neutral_program_states (get_descr_of_program_state_top_kind_s (PROGRAM_STATE_TOP_KIND__NEUTRAL));

const string text_pre_initial_program_states (get_descr_of_pre_initial_program_state_kind_s (PRE_INITIAL_PROGRAM_STATE_KIND__INITIAL));
const string text_post_halting_program_states (get_descr_of_post_halting_program_state_kind_s (POST_HALTING_PROGRAM_STATE_KIND__HALTING));
const string text_user_required_check_point_program_states (get_descr_of_user_required_program_state_kind_s (USER_REQUIRED_PROGRAM_STATE_KIND__CHECK_POINT));

size_t text_max_size = max_size_of_descr_of_embedded_program_state_kinds_s;
  assert (text_max_size);
  text_max_size = MAX_VALUE (text_max_size, text_neutral_program_states.size());
  text_max_size = MAX_VALUE (text_max_size, text_pre_initial_program_states.size());
  text_max_size = MAX_VALUE (text_max_size, text_post_halting_program_states.size());
  text_max_size = MAX_VALUE (text_max_size, text_user_required_check_point_program_states.size());
  text_max_size += str_tail.size();  


  cout << setw(text_max_size) << left << string (text_neutral_program_states + str_tail).c_str() << " : ";
  to_out (neutral_program_states_);


  cout << setw(text_max_size) << left << string (text_pre_initial_program_states + str_tail).c_str() << " : ";
  to_out (pre_initial_program_states_);


  cout << setw(text_max_size) << left << string (text_post_halting_program_states + str_tail).c_str() << " : ";
  to_out (post_halting_program_states_);

  cout << setw(text_max_size) << left << string (text_user_required_check_point_program_states + str_tail).c_str() << " : ";
map<CurSituation, state_t>::const_iterator pos_iter;
  for (pos_iter = user_required_check_point_program_states_.begin();
       pos_iter != user_required_check_point_program_states_.end();
       pos_iter++
       )
  {
    cout << setw (max_size_of_extra_program_states_) << left << pos_iter->second.c_str() << " ";
  }
  cout << endl;

  assert (embedded_program_states_.size() == NUMBER_OF_EMBEDDED_PROGRAM_STATE_KINDS);
  for (size_t i = 0; i < embedded_program_states_.size(); i++)
  {
    cout << setw(text_max_size) << left << string (get_descr_of_embedded_program_state_kinds_s (static_cast<EmbeddedProgramStateKinds>(i)) + str_tail).c_str() << " : ";
    for (size_t j = 0; j < embedded_program_states_[i].size(); j++)
    {
      cout << setw (max_size_of_extra_program_states_) << left << embedded_program_states_[i][j].c_str() << " ";
    }
    cout << endl;

  }

} // show_extra_program_states


// =========
void TuringMachine::show_daemon_states_s (const string& msg_i)
{
  cout << "\t   ====== Daemon States Definition ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');


string text_daemon_passive_states ("Passive daemon states");
string text_daemon_active_states ("Active daemon states");
string text_daemon_aggressive_states ("Aggressive daemon states");
size_t text_max_size = 0;

  text_max_size	= MAX_VALUE(text_max_size, text_daemon_passive_states.size());
  text_max_size	= MAX_VALUE(text_max_size, text_daemon_active_states.size());
  text_max_size	= MAX_VALUE(text_max_size, text_daemon_aggressive_states.size());


  cout << setw(text_max_size) << left << text_daemon_passive_states.c_str() << " : ";
  for (size_t i = 0; i < daemon_passive_states_s.size(); i++)
  {
    cout << setw (max_size_of_daemon_states_s) << daemon_passive_states_s[i].c_str() << " ";
  }
  cout << endl;

  cout << setw(text_max_size) << left << text_daemon_active_states.c_str() << " : ";
  for (size_t i = 0; i < daemon_active_states_s.size(); i++)
  {
    cout << setw (max_size_of_daemon_states_s) << daemon_active_states_s[i].c_str() << " ";
  }
  cout << endl;

  cout << setw(text_max_size) << left << text_daemon_aggressive_states.c_str() << " : ";
  for (size_t i = 0; i < daemon_aggressive_states_s.size(); i++)
  {
    cout << setw (max_size_of_daemon_states_s) << daemon_aggressive_states_s[i].c_str() << " ";
  }
  cout << endl;


} // show_daemon_states_S

			   

// =========
void TuringMachine::show_tester_states_s (const string& msg_i)
{
  cout << "\t   ====== Tester States Definition ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');


string text_tester_tracking_states ("Tracking tester states");
string text_tester_stabilizing_states ("Stabilizing tester states");
size_t text_max_size = 0;

  text_max_size	= MAX_VALUE(text_max_size, text_tester_tracking_states.size());
  text_max_size	= MAX_VALUE(text_max_size, text_tester_stabilizing_states.size());


  cout << setw(text_max_size) << left << text_tester_tracking_states.c_str() << " : ";
  for (size_t i = 0; i < tester_tracking_states_s.size(); i++)
  {
    cout << setw (max_size_of_tester_states_s) << tester_tracking_states_s[i].c_str() << " ";
  }
  cout << endl;

  cout << setw(text_max_size) << left << text_tester_stabilizing_states.c_str() << " : ";
  for (size_t i = 0; i < tester_stabilizing_states_s.size(); i++)
  {
    cout << setw (max_size_of_tester_states_s) << tester_stabilizing_states_s[i].c_str() << " ";
  }
  cout << endl;


} // show_tester_states_S



// =========
void TuringMachine::show_apparatus_states_s (const string& msg_i)
{
  cout << "\t   ====== Apparatus States Definition ======" << endl;

  IF_NOT_EMPTY (msg_i, 5, '=');


string text_apparatus_normal_states ("Normal apparatus states");
string text_apparatus_emergency_states ("Emergency apparatus states");
size_t text_max_size = 0;

  text_max_size	= MAX_VALUE(text_max_size, text_apparatus_normal_states.size());
  text_max_size	= MAX_VALUE(text_max_size, text_apparatus_emergency_states.size());


  cout << setw(text_max_size) << left << text_apparatus_normal_states.c_str() << " : ";
  for (size_t i = 0; i < apparatus_normal_states_s.size(); i++)
  {
    cout << setw (max_size_of_apparatus_states_s) << apparatus_normal_states_s[i].c_str() << " ";
  }
  cout << endl;

  cout << setw(text_max_size) << left << text_apparatus_emergency_states.c_str() << " : ";
  for (size_t i = 0; i < apparatus_emergency_states_s.size(); i++)
  {
    cout << setw (max_size_of_apparatus_states_s) << apparatus_emergency_states_s[i].c_str() << " ";
  }
  cout << endl;


} // show_apparatus_states_S



// =========
void TuringMachine::show_alphabet (const Tapes_t::const_iterator& iter) const
{
  iter->second.show_alphabet();
}



// =========
void TuringMachine::show_pseudo_alphabets () const
{
const size_t setw3 = 3;
const size_t setw4 = 3;
const string text1 ("Never-mind-pseudo-symbol");
const string text2 ("Not-marker-pseudo-symbol");
const string text3 ("Not-marker-and-not-blank-pseudo-symbol");
const string text4 ("Write-nothing-pseudo-symbol");

size_t max_size = 0;
  max_size = MAX_VALUE (max_size, text1.size());
  max_size = MAX_VALUE (max_size, text2.size());
  max_size = MAX_VALUE (max_size, text3.size());
  max_size = MAX_VALUE (max_size, text4.size());

  cout << "\t" 
       << string (setw4, ' ')
       << string (setw3, '-')
       << "> "
       << "Special Alphabets"
       << " <"
       << string (setw3, '-') 
       << endl;

  cout << setw (max_size) << left << text1.c_str() << " : ";
  to_out (pseudo_never_mind_symbols_alphabet_);

  cout << setw (max_size) << left << text2.c_str() << " : ";
  to_out (pseudo_not_marker_symbols_alphabet_);

  cout << setw (max_size) << left << text3.c_str() << " : ";
  to_out (pseudo_not_marker_and_not_blank_symbols_alphabet_);

  cout << setw (max_size) << left << text4.c_str() << " : ";
  to_out (pseudo_write_nothing_symbols_alphabet_);
  cout << endl;

}

// =========
void TuringMachine::show_alphabets () const
{
Tapes_t::const_iterator pos_iter;
const size_t setw1 = 8;
const size_t setw2 = 6;
const size_t setw3 = 3;
const size_t setw4 = 3;
const string text1 ("Tape(s)");

  cout << endl;
  cout << endl;
  cout << "\t   ====== Alphabets Definition ======" << endl;

  show_pseudo_alphabets ();
  cout << endl;

  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    cout << "\t" 
         << string (setw4, ' ')
         << string (setw3, '-')
         << "> "
         << (i + 1)
         << ". " 
         << get_descr_of_tape_kind_s (static_cast<TapeKinds> (i))
         << " "
         << text1
         << " <"
         << string (setw3, '-') 
         << endl;

    for (pos_iter = tapes_[i].begin(); pos_iter != tapes_[i].end(); pos_iter++)
    {
      cout << "\t"
           << string (setw1, ' ')
           << string (setw2, '-')
           << " "
           << "Tape# " 
           << distance (tapes_[i].begin(), pos_iter) 
           << " "
           << string (setw2, '-') 
           << endl;
      show_alphabet (pos_iter);
      cout << endl;
    }
    cout << endl;
  }
}		    

// =========
void TuringMachine::show_transitions (
			bool flag_control_via_i,   
			bool flag_from_to_i,  
			const string& msg1_i,
			const string& msg2_i
			) const
{
  // IF_NOT_EMPTY (msg_i, 5, '=');

  assert (!built_transitions_.empty());

Transitions_t::const_iterator pos_iter;

const size_t size1 = 6;
  // cout << endl;
  cout << endl;
  cout << "\t   "
       << string (size1, '=')
       << " "
       << msg1_i
       << " Transition Rules "
       << string (size1, '=')
       << endl;

  for (pos_iter = built_transitions_.begin(); 
       pos_iter != built_transitions_.end(); 
       pos_iter++)
  {
    vector<string>  tmp_vect (getstr_transition_rule (
			built_transitions_.begin(), 
			pos_iter, 
			flag_control_via_i, 
			flag_from_to_i, 
			((msg2_i.empty()) ? msg1_i : msg2_i)
			)
			);
    for (size_t i = 0; i < tmp_vect.size(); i++)
    {
      if (i > 0) cout << " :   ";
      cout << tmp_vect[i];
    } 
    cout << endl;
  }
  cout << endl;
}


// =========
void TuringMachine::show_outside_tact1_transitions (
			const string& msg1_i
			) const
{
  if (!msg1_i.empty())  cout << msg1_i << endl;

map<CurOutsideSituation, NextOutsideSituation>::const_iterator pos_iter;

const string prefix1 ("Outside (Tact1) Rule#");

  cout << endl;
  cout << "\t   ====== [Tact1] Outside Non-Deterministic Transition Rules ======" << endl; 

size_t counter = 0;

  for (pos_iter = outside_tact1_transitions_.begin(); 
       pos_iter != outside_tact1_transitions_.end(); 
       pos_iter++
       )
  {
    assert (pos_iter->first.get_non_deterministic_rule_no() > 0);

    cout << ""
         << prefix1
         << setw (2)
         << right
         << counter
         << " : "
         << "( " 
         << left
         << setw (max_size_of_daemon_states_s)
         << pos_iter->first.get_cur_daemon_state() 
         << " " 
         << setw (max_size_of_tester_states_s)
         << pos_iter->first.get_cur_tester_state()
         << " " 
         << setw (max_size_of_apparatus_states_s)
         << pos_iter->first.get_cur_apparatus_state()
         << " ) " 
         << " ---> " 
         << "( " 
         << setw (max_size_of_daemon_states_s)
         << pos_iter->second.get_next_daemon_state()
         << " " 
         << setw (max_size_of_tester_states_s)
         << pos_iter->second.get_next_tester_state() 
         << " " 
         << setw (max_size_of_apparatus_states_s)
         << pos_iter->second.get_next_apparatus_state()
         // << " " 
         // << pos_iter->second.get_program_state_allowed_flag()
         << " ) " 
         << endl;

    counter++;
  }

  cout << endl;

} // show_outside_tact1_transitions




// =========
void TuringMachine::show_outside_tact2_transitions (
			const string& msg1_i
			) const
{
  if (!msg1_i.empty())  cout << msg1_i << endl;

map<CurOutsideSituation, NextOutsideSituation>::const_iterator pos_iter;

const string prefix1 ("Outside (Tact2) Rule#");

  cout << endl;
  cout << "\t   ====== [Tact2] Outside (Deterministic) Transition Rules ======" << endl; 

size_t counter = 0;

  for (pos_iter = outside_tact2_transitions_.begin(); 
       pos_iter != outside_tact2_transitions_.end(); 
       pos_iter++
       )
  {
    assert (pos_iter->first.get_non_deterministic_rule_no() == 0);

    cout << ""
         << prefix1
         << setw (2)
         << right
         << counter
         << " : "
         << "( " 
         << left
         << setw (max_size_of_daemon_states_s)
         << pos_iter->first.get_cur_daemon_state() 
         << " " 
         << setw (max_size_of_tester_states_s)
         << pos_iter->first.get_cur_tester_state()
         << " " 
         << setw (max_size_of_apparatus_states_s)
         << pos_iter->first.get_cur_apparatus_state()
         << " ) " 
         << " ---> " 
         << "( " 
         << setw (max_size_of_daemon_states_s)
         << pos_iter->second.get_next_daemon_state()
         << " " 
         << setw (max_size_of_tester_states_s)
         << pos_iter->second.get_next_tester_state() 
         << " " 
         << setw (max_size_of_apparatus_states_s)
         << pos_iter->second.get_next_apparatus_state()
         << " " 
         << " " 
         << pos_iter->second.get_program_state_allowed_flag()
         << " ) " 
         << endl;

    counter++;
  }

  cout << endl;

} // show_outside_tact2_transitions


// =========
void TuringMachine::show_daemon_transitions (
			bool flag_control_via_i,   
			bool flag_from_to_i,  
			const string& msg1_i,
			const string& msg2_i
			) const
{
  // IF_NOT_EMPTY (msg_i, 5, '=');

const size_t size1 = 6;

  if (daemon_defined_fault_transitions_.empty())
  {
    ostringstream oss;
    oss << "Daemon has no transition rules";

    cout << endl;
    cout << "\t   " << string (oss.str().size(), '=') << endl;
    cout << "\t   " << oss.str() << endl;
    cout << "\t   " << string (oss.str().size(), '=') << endl;
    cout << endl;

    return;
  }

  assert (!daemon_defined_fault_transitions_.empty());

Transitions_t::const_iterator pos_iter;

  // cout << endl;
  cout << endl;
  cout << "\t   "
       << string (size1, '=')
       << " "
       << msg1_i
       << " Daemon Transition Rules "
       << string (size1, '=')
       << endl;

  for (pos_iter = daemon_defined_fault_transitions_.begin(); 
       pos_iter != daemon_defined_fault_transitions_.end(); 
       pos_iter++)
  {
    vector<string>  tmp_vect (getstr_transition_rule (
			daemon_defined_fault_transitions_.begin(), 
			pos_iter, 
			flag_control_via_i, 
			flag_from_to_i, 
			((msg2_i.empty()) ? msg1_i : msg2_i)
			)
			);
    for (size_t i = 0; i < tmp_vect.size(); i++)
    {
      if (i > 0) cout << " :   ";
      cout << tmp_vect[i];
    } 
    cout << endl;
  }

  cout << endl;
} // show_daemon_transitions


// =========
vector<string> TuringMachine::getstr_transition_rule (
		const Transitions_t::const_iterator& iter_begin_i,
		const Transitions_t::const_iterator& iter_i,
		bool		flag_control_via_i,
		bool		flag_from_to_i,
		const string&	msg_i
		) const
{

  assert (
	   (iter_begin_i == user_defined_transitions_.begin())
	   ||
	   (iter_begin_i == built_transitions_.begin())
	   ||
	   (iter_begin_i == daemon_defined_fault_transitions_.begin())
         );

const size_t dist = distance (iter_begin_i, iter_i);

vector<string>	ret_vector;
ostringstream oss;
  oss	<< ""
	<< msg_i
	<< " Rule#"
	<< setw (SETW_RULE_NUMBER)
	<< right
	<< dist;

  if (iter_begin_i == built_transitions_.begin())
  {
  if (!(msg_i == RESULTING_text))
  {
    oss	<< " ["
        // << setw (3)
	<< info_user_transitions_[iter_i->first].c_str()
	<< "]";
    }
  }
  ret_vector.push_back (oss.str());
  oss.str("");


const state_t cur_substate1 (iter_i->first.get_substate1());
const state_t cur_substate2 (iter_i->first.get_substate2());
const state_t next_substate1 (iter_i->second.get_substate1());
const state_t next_substate2 (iter_i->second.get_substate2());

const string dummy0 ("-");

  oss	<< ""

	<< "( "

	<< setw (max_size_of_user_defined_program_states_)
	<< left
	<< iter_i->first.get_state().c_str()
	<< " "

	<< setw (max_size_of_user_defined_program_states_)
	<< left
        << ((cur_substate1 == state_t()) ? dummy0 : cur_substate1).c_str()
	<< " "

	<< setw (max_size_of_user_defined_program_states_)
	<< left
        << ((cur_substate2 == state_t()) ? dummy0 : cur_substate2).c_str()
	<< " ) "

	<< " [ " 
	<< iter_i->first.getstr_symbols(max_size_of_user_defined_alphabet_symbol_)
	<< "]  --->  "

        << "( "

	<< setw (max_size_of_user_defined_program_states_)
	<< left
	<< iter_i->second.get_state().c_str()
	<< " "

	<< setw (max_size_of_user_defined_program_states_)
	<< left
        << ((next_substate1 == state_t()) ? dummy0 : next_substate1).c_str()
	<< " "

	<< setw (max_size_of_user_defined_program_states_)
	<< left
        << ((next_substate2 == state_t()) ? dummy0 : next_substate2).c_str()
	<< " ) "

	<< " [ " 
	<< iter_i->second.getstr_symbols_and_shifts(max_size_of_user_defined_alphabet_symbol_)
	<< "]";
  ret_vector.push_back (oss.str());
  oss.str("");

  if (flag_control_via_i)
  {
    if (!user_required_check_point_program_states_.empty())
    {
      if (iter_i->first.get_controlling_flag ())
      {
        oss << "  ( " << CHECK_POINT_FIELD_IN_TRANSITION_FILE << " : via control extra program state ";
        assert (user_required_check_point_program_states_.count (iter_i->first) == 1);
        oss << user_required_check_point_program_states_.find (iter_i->first)->second;
        oss << " )";
        ret_vector.push_back (oss.str());
        oss.str("");
      }
    }
  }

  // --------------------------
  if (flag_from_to_i)
  {
    oss << ""
        << "\""
        << get_descr_of_program_state (iter_i->first.get_state(), true)
        << "\""
        << "  --->  "
        << "\""
        << get_descr_of_program_state (iter_i->second.get_state(), true)
        << "\"";
        ret_vector.push_back (oss.str());
        oss.str("");
  }

  return ret_vector;
}
   
// =========
vector<string> TuringMachine::getstr_machine_applied_rule (
			const Transitions_t::const_iterator& iter_i, 
			const string&	msg_i
			) const
{
  return getstr_transition_rule (built_transitions_.begin(), iter_i, true, true, msg_i);
}
   
// =========
vector<string> TuringMachine::getstr_tester_applied_rule (
			const Transitions_t::const_iterator& iter_i, 
			const string&	msg_i
			) const
{
  return getstr_transition_rule (user_defined_transitions_.begin(), iter_i, false, false, msg_i);
}


   
// =========
vector<string> TuringMachine::getstr_daemon_applied_rule (
			const Transitions_t::const_iterator& iter_i, 
			const string&	msg_i
			) const
{
  return getstr_transition_rule (daemon_defined_fault_transitions_.begin(), iter_i, false, false, msg_i);
}

// =========
void TuringMachine::show_tape (const Tapes_t::const_iterator& iter) const
{
  iter->second.show_tape();
}

// =========
void TuringMachine::show_situation (
			const string& info_i, 
			size_t interval_no_i, 
			size_t local_no_i, 
			size_t serial_no_i, 
			const string& msg_i
			) const

{
Tapes_t::const_iterator pos_iter;
const size_t size1 = 5;
const string prefixt ("\t");
const string preprefix ("==M==| ");

  if (!process_transitions_.empty())
  {  
const Transitions_t::const_iterator find_iter = built_transitions_.find ((*process_transitions_.rbegin())->first);
    assert (find_iter != built_transitions_.end());
    cout << prefixt
         << preprefix 
         << getstr_id_info ()
         << "< Machine-Step "
         << setw (3)
         << right
         << interval_no_i
         << "."
         << setw (2)
         << setfill ('0')
         << right
         << local_no_i
         << " /"
         << setw (4)
         << setfill (' ')
         << right
         << serial_no_i
         << ") >   "         
         << "Applied ";

vector<string> tmp_vect (getstr_machine_applied_rule (find_iter, RESULTING_text));
    for (size_t i = 0; i < tmp_vect.size(); i++)
    {
      if (i > 0) cout << prefixt << preprefix;
      cout << tmp_vect[i] << endl;
    }
    cout << endl;
  }

  cout << endl;

  IF_NOT_EMPTY (msg_i, size1, '-');

  cout << "\t"
       << string (size1, '-')
       << " ";
  
  if (info_i.empty())
  {
    cout << getstr_id_info ()
         << "Machine-Configuration #"
         << interval_no_i
         << "."
         << local_no_i
         << " ["
         << serial_no_i
         << "]";
  }
  else
  {
    cout << getstr_id_info () 
         << info_i
         << " Machine-Configuration";

  }

  cout << " "
       << string (size1, '-')
       << endl;

  process_extended_states_.rbegin()->show_extended_state (
	get_descr_of_program_state (
		process_extended_states_.rbegin()->program_state_
		), 
		"\t"
	); 

  // cout << "\t" << string (size2, '-') << endl;

const size_t size2 = max_size_of_descr_of_tape_kinds_s;
  for (size_t i = 0; i < NUMBER_OF_TAPE_KINDS; i++)
  {
    cout << endl;
    for (pos_iter = tapes_[i].begin(); pos_iter != tapes_[i].end(); pos_iter++)
    {
      cout << "" 
           << setw (size2)
           << left
           << get_descr_of_tape_kind_s (static_cast<TapeKinds>(i)).c_str()
           << " "
           << "Tape#" 
           << distance (tapes_[i].begin(), pos_iter) 
           << " : ";
      show_tape (pos_iter);
    }
  }
  // cout << endl;

} // TuringMachine::show_situation 




// =========
void TuringMachine::show_tester_situation (
			bool start_flag_i, 
			size_t call_no_i, 
			size_t local_no_i, 
			size_t serial_no_i, 
			const string& msg_i
			) const

{
Tapes_t::const_iterator pos_iter;
const size_t size1 = 3;
const string prefix3 (3, ' ');

  if ((!start_flag_i) && (!process_tester_transitions_.empty()))
  {  
const Transitions_t::const_iterator find_iter = user_defined_transitions_.find ((*process_tester_transitions_.rbegin())->first);
    assert (find_iter != user_defined_transitions_.end());
    cout << prefix3
         << getstr_id_info ()
         << "< Tester-Step "
         << setw (3)
         << right
         << call_no_i
         << "."
         << setw (2)
         << setfill ('0')
         << right
         << local_no_i
         << " /"
         << setw (4)
         << setfill (' ')
         << right
         << serial_no_i
         << ">   "         
         << "Applied "; 
vector<string> tmp_vect(getstr_tester_applied_rule (find_iter, "User-Defined"));
    for (size_t i = 0; i < tmp_vect.size(); i++)
    {
      if (i > 0) cout << " :  ";
      cout << tmp_vect[i];
    }
    cout << endl;
  }

  cout << endl;

  IF_NOT_EMPTY (msg_i, size1, '-');


  cout << string (3, ' ')
       << string (size1, '-')
       << " ";
  
  if (start_flag_i)
  {
    cout << getstr_id_info () 
         << "Tester-Configuration (Start-" 
         << call_no_i 
         << ")";
  }
  else
  {
    cout << getstr_id_info ()
         << "Tester-Configuration #"
         << call_no_i
         << "."
         << local_no_i
         << " ["
         << serial_no_i
         << "]";
  }
  cout << " "
       << string (size1, '-')
       << endl;



  cout << prefix3 << "State  : " << *process_tester_states_.rbegin() << endl;

  // cout << "\t" << string (size2, '-') << endl;

  for (pos_iter = tapes_[TESTER_TAPE].begin(); pos_iter != tapes_[TESTER_TAPE].end(); pos_iter++)
  {
    cout << "Tape#" << distance (tapes_[TESTER_TAPE].begin(), pos_iter) << " : ";
    show_tape (pos_iter);
  }

  // cout << endl;


} // TuringMachine::show_tester_situation 


// =========
void TuringMachine::show_statistics (const string& msg_i) const
{
const size_t size1 = 5;

  IF_NOT_EMPTY (msg_i, size1, '-');

  assert (process_statistics_.size() == built_transitions_.size());

uint total = accumulate (process_statistics_.begin(), process_statistics_.end(), 0);

const string text_statistics ("Statistics");
const string text_transition ("Transitions");
const string text_rules ("Rule#");
const string text_times ("Times");
const string text_total ("Total");
const string delim (":");

const size_t max_size1 = MAX_VALUE (text_statistics.size(), text_transition.size());
const size_t the_size2 = 3;
const size_t the_size1 = the_size2 + 1 + max_size1 + 1 + the_size2;
const size_t the_size3 = (the_size1 - 1)/2;
const size_t the_size4 = (the_size1 - 1) - the_size3;
  assert (text_rules.size() < (the_size3 - 1));
  assert (text_times.size() < (the_size4 - 1));

const string l_border ("\t      |");		
const string r_border ("|");		

  cout << l_border << string (the_size1, '=') << r_border << endl;

  cout << l_border 
       << string (the_size2, '-') 
       << " "
       << setw (max_size1)
       << left
       << text_statistics.c_str()
       << " "
       << string (the_size2, '-') 
       << r_border 
       << endl;

  cout << l_border 
       << string (the_size2, '.') 
       << " "
       << setw (max_size1)
       << left
       << text_transition.c_str()
       << " "
       << string (the_size2, '.') 
       << r_border 
       << endl;

  cout << l_border << string (the_size1, '-') << r_border << endl;

  cout << l_border 
       << setw (the_size3 - 1)
       << right
       << text_rules.c_str()
       << " "
       << delim
       << setw (the_size4 - 1)
       << right
       << text_times.c_str()
       << " "
       << r_border 
       << endl;

  cout << l_border << string (the_size1, '-') << r_border << endl;

  for (size_t i = 0; i < process_statistics_.size(); i++)
  {
    if (process_statistics_[i] == 0) continue;
    cout << l_border 
         << setw (the_size3 - 1)
         << right
         << i
         << " "
         << delim
         << setw (the_size4 - 1)
         << right
         << process_statistics_[i]
         << " "
         << r_border 
         << endl;
  }
  cout << l_border << string (the_size1, '-') << r_border << endl;

  cout << l_border 
       << setw (the_size3 - 1)
       << right
       << text_total.c_str()
       << " "
       << delim
       << setw (the_size4 - 1)
       << right
       << total
       << " "
       << r_border 
       << endl;

  cout << l_border << string (the_size1, '=') << r_border << endl;

}


// =========
void TuringMachine::show_all_tapes_alphabets_combinations () const
{
vector<vector<symbol_mt> > tmp_vector (get_all_tapes_alphabets_combinations ());
  for (size_t i = 0; i < tmp_vector.size(); i++)
  {
    cout << "Symbol Combination# " << setw (4) << i << " : ";
    to_out (tmp_vector[i]);
  }

}


// =========
void TuringMachine::show_input (
	const vector<vector<symbol_mt> >& input_words_i,
	bool full_flag_i
	) const
{
const size_t size1 = 5;

  cout << endl;
  cout << "\t"
       << string (size1, '#')
       << " ";

  if (full_flag_i)
  {
    cout << ""
         << getstr_id_info ();
  }
  cout << "Input words on "
       << get_descr_of_tape_kind_s (MASTER_COMPUTATION_TAPE)
       << " Tape(s) "
       << string (size1, '#')
       << endl;

  for (size_t i = 0; i < input_words_i.size(); i++)
  {
    cout << "Tape#" << i << " : ";
    for (size_t j = 0; j < input_words_i[i].size(); j++)
    {
      cout << setw (max_size_of_user_defined_alphabet_symbol_) << to_string (input_words_i[i][j]).c_str() << " ";
    }
    cout << endl;
  }
  cout << endl;
}


// =========
void TuringMachine::show1_env () const
{
  cout << endl;
  cout << "\t###### Turing Machine Definition ######" << endl;
  show_descr ();
  cout << endl;
  show_user_defined_program_states ();
  cout << endl;
  show_extra_program_states ();
  cout << endl;
  show_daemon_states_s ();
  cout << endl;
  show_tester_states_s ();
  cout << endl;
  show_apparatus_states_s ();
  cout << endl;
  cout << endl;
  show_detail_description_of_program_states ();
  cout << endl;
  show_alphabets();

}


// =========
void TuringMachine::show2_env () const
{
  show_transitions (false, true, RESULTING_text);
  cout << endl;

}

