// ------------------------------------------------------------------------
// System         : WIN9X/NT
// Program        : idiff.c
// Description    : Rewritten version of an ancient diff...
//
//                  This program compares 2 text files and
//                  reports their differences.
//
//                  Options -
//                  -b ignore whitespace
//                  -i ignore case
//                  -! ignore characters after (Clarion) comments
//
// Written by     : Unknown
// ReWritten by   : Bill Buckels
//
// Date ReWritten : 1994
// Revision       : 1.1 First Release
//                  2.0 Added Support For Long FileNames - March 2000
// ------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <malloc.h>
#include <string.h>

/* local functions */

int checkargs(char *ptr);
int openfiles(char *nam1, char *nam2);
char *getl(int f, long n);
void movstr(register char *s, register char *t);
void clrl(int f, long n);
int easysynch(void);
int output(int a, int b);
void change(long a, int b, long c, int d, char *s);
void range(long a, int b);
int cmp(char *s, char *t);
void error(char *s, char *t);
void progerr(char *s);
int hardsynch(void);

#define C 2
#define MAX_RANGE 2048      // push our luck with malloc
#define LEN 255             // compare strings up to 255 characters long
#define BUF_SIZE (LEN+1)
#define INF 16384

char *text[2][MAX_RANGE];
long lineno[2] = {1, 1};    // no. of 1st stored line in each file
int ntext[2];               // number of stored lines in each
long n0,n1;                 // scan pointer in each
int bflag=0;                // ignore blanks
int iflag=0;                // ignore case
int clarion=0;              // ignore clarion comments

// check for command line arguments
int checkargs(char *ptr)
{
  int i,c;
  
  c = ptr[0];
  switch(c)
  {
    case '-':
    case '/': break;
    default : return - 1;
  }
  
  for(i = 1;ptr[i] != 0;i++)
  {
    c = tolower(ptr[i]);
    switch(c)
    {
      case 'i':    iflag++;   break;
      case '!':    clarion++; break;
      case 'b':    bflag++;   break;
    }
  }
  
  return 0;
}

FILE *file[2];

// open the files that were named on the command line
int openfiles(char *nam1, char *nam2)
{
  char buffer[_MAX_PATH];
  if((file[0] = fopen(nam1, "r")) == NULL)
  {
    sprintf(buffer, "file1 : %s", nam1);
    perror(buffer);
    return - 1;
  }
  if((file[1] = fopen(nam2, "r")) == NULL)
  {
    fclose(file[0]);
    sprintf(buffer, "file2 : %s", nam2);
    perror(buffer);
    return - 2;
  }
  return 0;
}

/* return pointer to line n of file f*/
char *getl(int f, long n)
{
  register char *t;
  register delta, nt;
  char *ptr;
  
  again:
  delta = n - lineno[f];
  nt = ntext[f];
  if(delta < 0)
    progerr("1");
  if(delta < nt)
    return(text[f][delta]);
  if(delta > nt)
    progerr("2");
  if(nt >= MAX_RANGE)
    progerr("3");
  if(feof(file[f]))
    return(NULL);
  t = text[f][nt];
  if(t == 0) 
  {
    t = text[f][nt] = malloc(LEN + 1);
    if(t == NULL)
      if(hardsynch())
        goto again;
    else
      progerr("5");
  }
  
  ptr = fgets((char *)&t[1], LEN, file[f]);
  /* pad all lines with a whitespace */
  if(ptr != NULL)
  {
    ntext[f]++;
    t[0] = 32;
  }
  else
  {
    t = NULL;
  }
  return(t);
}

void movstr(register char *s, register char *t)
{
  while(*t++ = *s++)
    continue;
  return;
}

/* remove thru line n of file f from storage */
void clrl(int f, long n)
{
  register i,j;
  j = n - lineno[f] + 1;
  for(i = 0;i + j < ntext[f];i++)
    movstr(text[f][i + j], text[f][i]);
  lineno[f] = n + 1;
  ntext[f] -= j;
}

int main(int argc, char **argv)
{
  char *s0,*s1;
  char nam1[_MAX_PATH], nam2[_MAX_PATH],num = 0,i;
  char buffer[_MAX_PATH];
  
  puts("IDIFF(C) CopyLeft Bill Buckels 1994-2000. All Rights Reversed.");
  puts("--------------------------------------------------------------");

  // perform all the usual command line argument checks
  // open the files, activate the switches, etc.
  if(argc > 2)
  {
    for(i = 1;i < argc;i++)
    {
      strcpy(buffer, argv[i]);
      if(checkargs(buffer))
      {
        switch(num)
        {
          case 0: num = 1;strlwr(buffer);strcpy(nam1, buffer);break;
          case 1: num = 2;strlwr(buffer);strcpy(nam2, buffer);break;
        }
      }
    }
    if(num > 1)
    {
      if(openfiles(nam1, nam2))num = 0;
    }
  }
  
  if(argc < 3 || num < 2)
  {
    puts("Usage is :\"idiff -i!b file1 file2\"");
    puts("Switches : i (ignore case)");
    puts("         : ! (ignore [Clarion] comments)");
    puts("         : b (ignore blanks [tabs and whitespace])");
    return - 1;
  }
  
  printf("%s <---> %s\n\n", nam1, nam2);
  
  for(;;) 
  {
    s0 = getl(0, ++n0);
    s1 = getl(1, ++n1);
    if(s0 == NULL || s1 == NULL)
      break;
    if(cmp(s0, s1) != 0) 
    {
      if(!easysynch() && !hardsynch())
        progerr("5");
    } else 
    {
    clrl(0, n0);
    clrl(1, n1);
  }
  }
  if(s0 == NULL && s1 == NULL)
  {
    fcloseall();
    return 0;
  }
  if(s0 == NULL)
    output(- 1, INF);
  if(s1 == NULL)
    output(INF, - 1);
  fcloseall();
  return 0;
}

/* synch on C successive matches */
int easysynch()
{
  int i,j;
  register k,m;
  char *s0,*s1;
  for(i = j = 1;i < MAX_RANGE && j < MAX_RANGE;i++,j++)
  {
    s0 = getl(0, n0 + i);
    if(s0 == NULL)
      return(output(INF, INF));
    for(k = C - 1;k < j;k++) 
    {
      for(m = 0;m < C;m++)
        if(cmp(getl(0, n0 + i - m), 
            getl(1, n1 + k - m)) != 0)
          goto cont1;
      return(output(i - C, k - C));
      cont1:;
    }
    s1 = getl(1, n1 + j);
    if(s1 == NULL)
      return(output(INF, INF));
    for(k = C - 1;k <= i;k++) 
    {
      for(m = 0;m < C;m++)
        if(cmp(getl(0, n0 + k - m), 
            getl(1, n1 + j - m)) != 0)
          goto cont2;
      return(output(k - C, j - C));
      cont2:;
    }
  }
  return(0);
}

int output(int a, int b)
{
  register i;
  char *s;
  if(a < 0)
    change(n0 - 1, 0, n1, b, " [ lines added at lines   ] ");
  else if(b < 0)
    change(n0, a, n1 - 1, 0, " [ lines deleted at lines ] ");
  else
    change(n0, a, n1, b, " [ lines changed at lines ] ");
  for(i = 0;i <= a;i++) 
  {
    s = getl(0, n0 + i);
    if(s == NULL)
      break;
    printf("< %s", s);
    clrl(0, n0 + i);
  }
  n0 += i - 1;
  if(a >= 0 && b >= 0)
    printf("---\n");
  for(i = 0;i <= b;i++) 
  {
    s = getl(1, n1 + i);
    if(s == NULL)
      break;
    printf("> %s", s);
    clrl(1, n1 + i);
  }
  n1 += i - 1;
  return(1);
}

void change(long a, int b, long c, int d, char *s)
{
  range(a, b);
  printf("%s", s);
  range(c, d);
  printf("\n");
}

void range(long a, int b)
{
  if(b == INF)
    printf("%ld, to End of File", a);
  else if(b == 0)
    printf("%ld", a);
  else
    printf("%ld,%ld", a, a + b);
}

int cmp(char *s, char *t)
{
  // do this the easy way... work on a copy
  int i, ctr;
  char buf1[BUF_SIZE],buf2[BUF_SIZE];
  
  for(i = 0;i < BUF_SIZE;i++)
  {
    buf1[i] = 0;buf2[i] = 0;
  }
  
  ctr = 0;
  for(i = 0;s[i] != 0;i++)
  {
    if(clarion && s[i] == '!')break;
    if(s[i] == 13 || s[i] == 10 || s[i] == 0)break;
    switch(s[i])
    {
      default: if(bflag && isspace(s[i]))break;
      if(iflag)buf1[ctr] = tolower(s[i]);
      else buf1[ctr] = s[i];
      ctr++;
    }
  }
  
  ctr = 0;
  for(i = 0;t[i] != 0;i++)
  {
    if(clarion && t[i] == '!')break;
    if(t[i] == 13 || t[i] == 10 || t[i] == 0)break;
    switch(t[i])
    {
      default: if(bflag && isspace(t[i]))break;
      if(iflag)buf2[ctr] = tolower(t[i]);
      else buf2[ctr] = t[i];
      ctr++;
    }
  }
  return(strcmp(buf1, buf2));
}

void error(char *s, char *t)
{
  fprintf(stderr, "idiff: %s%s\n", s, t);
  fcloseall();
  exit(1);
}

void progerr(char *s)
{
  error("program error ", s);
}

/* stub for resychronization beyond limits of text buf */
int hardsynch()
{
  change(n0, INF, n1, INF, " [ lines changed at lines ] ");
  printf("---change record omitted\n");
  error("can't resynchronize", ""); /* exits here */

  return(0);  /* never gets to here */
}
