
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import java.util.*;
import java.io.*;
import org.webmacro.util.*;

/**
  * Template objects represent the user defined layout into which the 
  * webmacro package will substitute values. It is a very simple kind of
  * interpreted language containing text, blocks, and directives. Text is
  * to be passed through verbatim. LList group text and directives into
  * linear lists and sublists. Directives determine how subsequent blocks
  * are to be processed and constitute the commands of the language. 
  * <p>
  * The Template is lazily evaluated: it does not parse or open the 
  * supplied filename until it is used. Once it has parsed the file, it 
  * never alters its data. The intent is to allow a Template to be parsed
  * once (somewhat expensive) and then used many times; and also not to 
  * incur any parsing costs at all if the Template is never actually used.
  */

abstract public class Template 
{

   /**
     * What this template contains is a top level block
     */
   private Block myContent; 

   /**
     * Template parameters
     */
   private Hashtable myParameters;

   /**
     * Get the stream the template should be read from. Parse will 
     * call this method in order to locate a stream.
     * @exception IOException if unable to read template
     */
   protected abstract Reader getReader() throws IOException;

   /**
     * Return a name for this template. For example, if the template reads
     * from a file you might want to mention which it is--will be used to
     * produce error messages describing which template had a problem.
     */
   public abstract String toString(); 


   /**
     * Force the template to parse now. Normally the template will not parse
     * the supplied file until the data is actually needed. However if you 
     * want to parse all of your templates at the start of the application 
     * to avoid incurring this call during an interactive session, you can 
     * call the parse() function at an appropriate time. Alternately, you
     * could call this function to reparse a template if you know that it
     * has changed.
     * <p>
     * @exception ParseException if the sytax was invalid and we could not recover
     * @exception IOException if we could not successfullly read the parseTool
     */
   public final void parse() throws IOException, ParseException
   {

      // thread policy:
      //
      // unsynchronized code elsewhere will access myContent. We must
      // ensure that any data copied into myContent is ready for public 
      // use--which means dealing with two subtle issues. First, the Block
      // created by this method must be written back to main memory BEFORE
      // being referenced by myContent. Second, once we add it to myContent,
      // our copy of myContent has to be copied back to main memory as well.
      //
      // Therefore we synchronize around the access to myContent so as to 
      // accomplish both these things. newContent will be written to main 
      // memory at the start of the synchronized block, and myContent will
      // be written to main memory at the close.

      Block newContent = new Block();
      Hashtable newParameters = null;
      Reader source = null;
      try {
         source = getReader();
         ParseTool in = new ParseTool(this.toString(),source); 
         newParameters = in.getParamContext();
         do {
            try {
               in.nextToken(); // read the first token
               newContent.addElement((Block) Block.parse(in)); 
            } catch (ParseException e) {
               Engine.log.exception(e);
               if (in.ttype != in.TT_EOF) {
                  in.nextToken(); // skip a token before continuing
               }
            }
         } while (in.ttype != in.TT_EOF);
      } catch (FileNotFoundException e) {
         newContent = null; // don't let the old one survive
         Engine.log.error("Template: Could not find template: " + this);
         throw e;
      } catch (IOException e) {
         newContent = null; // don't let the old one survive
         Engine.log.error("Template: Could not read template: " + this);
         throw e;
      } finally {
         try { source.close(); } catch (Exception e) { }
         synchronized(this) {
            myParameters = newParameters;
            myContent = newContent; 
         }
      }
   }

   /**
     * Parse the Template against the supplied context data and 
     * return it as a string. If the operation fails for some reason, 
     * such as unable to read template or unable to introspect the context
     * then this method will return a null string.
     */
   public final Object evaluate(Object data)
   {
      StringWriter sw = new SizedStringWriter(4096); // 4 kilobytes arbitrary
      try {
         if (write(sw,data)) {
            return sw.toString();
         } else {
            return null;
         }
      } catch (IOException e) {
         Engine.log.exception(e);
         Engine.log.error("Template: Could not write to StringWriter!");
         return null;
      }
   }


   /**
     * A macro has a write method which takes a context and applies
     * it to the macro to create a resulting String value, which is 
     * then written to the supplied stream. Something will always be
     * written to the stream, even if the operation is not really 
     * successful because of a parse error (an error message will 
     * be written to the stream in that case.)
     * <p>
     * @exception IOException if there is a problem writing to the Writer
     * @return whether the operation was a success
     */
   public final boolean write(Writer out, Object data) 
      throws IOException
   {

      // thread policy: Access to myContent is unsynchronized here at 
      // the cost of having a slightly stale copy. This is OK, because 
      // you might have requested it slightly earlier anyway. You will 
      // always get a consistent copy--either the current one, or one 
      // that was the current one a few milliseconds ago. This is because
      // a thread setting myContent may not update main memory immediately,
      // and this thread may not update its copy of myContent immediately
      // (it may have a stale copy it read earlier).

      Block content = myContent; // copy to a local var in case it changes

      // Make sure that all the contents of "content" are up to date--that
      // our thread is fully synchronized with main memory, so that we don't
      // have a half-created version. Synchronize on anything to do this: 
      // we will use an object we don't share with any other thread.
      synchronized(data) { } 

      if (content == null) {
         try {
            synchronized(this) {
               // double check under the lock
               if (myContent == null) {
                  parse();   
               }
               content = myContent;
            }
         } catch (Exception e) {
            Engine.log.exception(e);
            Engine.log.error("Template: Unable to read template: " + this);
            out.write("<!--\n Template failed to read. Reason: ");
            out.write(e.toString());
            out.write(" \n-->");
            return false;
         }
      } 

      try {
         content.write(out,data);
      } catch (InvalidContextException e) {
         Engine.log.exception(e);
         String warning = 
            "Template: Missing data in Map passed to template " + this;
         Engine.log.warning(warning);
         
         out.write("<!--\n Could not interpret template. Reason: ");
         out.write(warning);
         out.write(e.toString());
         out.write(" \n-->");
         return false;
      }
      return true;
   }

   /**
     * A template may contain parameters, set by the #param directive.
     * These are statically evaluated during the parse phase of the 
     * template and shared between all users of the template. They 
     * are present so that the template can provide some meta information
     * to its user as to what kind of data it expects to find in the Context,
     * or other information about its use. 
     * <p>
     * If the template has not already been parsed, it will be parsed. Thus
     * this method may throw ParseException or IOException if there is some
     * failure in accessing or parsing the template.
     * @exception IOException if an error occurred reading the template
     * @exception ParseException if an error occurred parsing the template
     */
   public Object getParam(String name) 
      throws IOException, ParseException 
   {

      // perform double checked lock in case template has not yet 
      // been parsed.

      try {
         return myParameters.get(name);
      } catch (NullPointerException e) {
         synchronized(this) {
            parse();
            return myParameters.get(name);
         }
      }
   }
}

