/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.system;

import com.jniwrapper.*;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Kernel32;
import com.jniwrapper.win32.LastErrorException;

/**
 * This class provides information and methods relevant to SYSTEM_INFO structure,
 * and provides additional methods related to system information.
 *
 * @author Serge Piletsky
 * @author Alexander Evsukov
 */
public class SystemInfo extends Structure
{
    private UInt16 _processorArchitecture = new UInt16();
    private UInt16 _reserved = new UInt16();
    private UInt32 _pageSize = new UInt32();
    private UInt32 _minAppAddress = new UInt32();
    private UInt32 _maxAppAddress = new UInt32();
    private UInt32 _activeProcessorMask = new UInt32();
    private UInt32 _numberOfProcessors = new UInt32();
    private UInt32 _processorType = new UInt32();
    private UInt32 _allocationGranularity = new UInt32();
    private UInt16 _processorLevel = new UInt16();
    private UInt16 _processorRevision = new UInt16();

    static final String FUNCTION_GET_SYSTEM_INFO = "GetSystemInfo";
    static final FunctionName FUNCTION_GET_COMPUTER_NAME = new FunctionName("GetComputerName");
    static final FunctionName FUNCTION_GET_USER_NAME = new FunctionName("GetUserName");

    /**
     * Construct and initializes new instance with actual values representing
     * system information.
     */
    public SystemInfo()
    {
        init(new Parameter[]
        {
            _processorArchitecture,
            _reserved,
            _pageSize,
            _minAppAddress,
            _maxAppAddress,
            _activeProcessorMask,
            _numberOfProcessors,
            _processorType,
            _allocationGranularity,
            _processorLevel,
            _processorRevision
        });

        Function function = Kernel32.get(FUNCTION_GET_SYSTEM_INFO);
        function.invoke(null, new Pointer(this));
    }

    public long getProcessorArchitecture()
    {
        return _processorArchitecture.getValue();
    }

    public long getPageSize()
    {
        return _pageSize.getValue();
    }

    public long getMinAppAddress()
    {
        return _minAppAddress.getValue();
    }

    public long getMaxAppAddress()
    {
        return _maxAppAddress.getValue();
    }

    public long getActiveProcessorMask()
    {
        return _activeProcessorMask.getValue();
    }

    public long getNumberOfProcessors()
    {
        return _numberOfProcessors.getValue();
    }

    public long getProcessorType()
    {
        return _processorType.getValue();
    }

    public long getAllocationGranularity()
    {
        return _allocationGranularity.getValue();
    }

    public long getProcessorLevel()
    {
        return _processorLevel.getValue();
    }

    public long getProcessorRevision()
    {
        return _processorRevision.getValue();
    }

    /**
     * Returns the name of the local computer.
     * @return name of computer
     */
    public static String getComputerName()
    {
        Function function = Kernel32.get(FUNCTION_GET_COMPUTER_NAME);
        ZeroTerminatedString computerName = Kernel32.getInstance().stringParam("", 256);
        UInt32 size = new UInt32(computerName.getLength());
        Bool retVal = new Bool();
        function.invoke(retVal, new Pointer(computerName), new Pointer(size));
        if (retVal.getValue())
        {
            String result = computerName.getValue();
            return result;
        }
        else
        {
            throw new LastErrorException("Unable to get computer name.");
        }
    }

    /**
     * Returns the user name logged into the system.
     * @return name of user
     */
    public static String getUserName()
    {
        final Function function = AdvApi32.get(FUNCTION_GET_USER_NAME);
        ZeroTerminatedString userName = AdvApi32.getInstance().stringParam("", 256);
        UInt32 size = new UInt32(userName.getLength());
        Bool retVal = new Bool();
        function.invoke(retVal, new Pointer(userName), new Pointer(size));
        if (retVal.getValue())
        {
            String result = userName.getValue();
            return result;
        }
        else
        {
            throw new LastErrorException("Unable to get user name.");
        }
    }
}
