/* Quat - A 3D fractal generation program */
/* Copyright (C) 1997,98 Dirk Meyer */
/* (email: dirk.meyer@studbox.uni-stuttgart.de) */
/* mail:  Dirk Meyer */
/*        Marbacher Weg 29 */
/*        D-71334 Waiblingen */
/*        Germany */
/* */
/* This program is free software; you can redistribute it and/or */
/* modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2 */
/* of the License, or (at your option) any later version. */
/* */
/* This program is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */
/* */
/* You should have received a copy of the GNU General Public License */
/* along with this program; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */

#include <windows.h>
#include <commdlg.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>  /* tolower */


#include "resource.h"
#ifdef WINELIB
#   include "wineres.h"
#endif
#include "zlib.h"
#include "quat.h"
#include "winver.h"  
#include "win2.h"           

#ifndef WM_CTLCOLOR
#   define WM_CTLCOLOR    WM_CTLCOLOREDIT
#endif

int MakeBitmapFromZBuffer(HWND hWnd);
int MenuInCalcMode(void);
int MenuAfterCalc(void);

/* Defined in quat.c */
extern VER_ReturnVideoInfo ReturnVideoInfo;
extern VER_SetColors SetColors;
extern VER_Initialize Initialize;
extern VER_Done Done;
extern VER_update_bitmap update_bitmap; 
extern VER_getline getline; 
extern VER_putline putline;
extern VER_check_event check_event;
extern VER_Change_Name Change_Name;
extern VER_Debug Debug;
extern VER_eol eol;

/* Defined in title.c */
extern unsigned char TitleImg[];
extern long TitleImg_Length;
extern long TitleImg_X, TitleImg_Y;

int minsize;
int picx, picy;                                     
int pixx, pixy;
int bmx, bmy;
int xadd=0, yadd = 0, imgxstart, imgystart, zbufxstart, zbufystart;   
/* *A* HGLOBAL RGBmap, ZBuf; */
unsigned char *RGBmap, *ZBuf;
unsigned char imginmem = 0, imgnew = 1, 
   imgready = 0, imgmodified = 0, canmodify = 1;
/* RGBmap: handle from GlobalAlloc for RGB values of PNG image in memory */
/* imgnew: flag. Shows if calculation had already started */
/* imgready: flag. Shows if calculation ready. */
/* canmodify: flag. Shows if parameters can be modified, or not (pic in mem) */
unsigned char zbufinmem = 0, zbufready = 0, zbufmodified = 0/*, zbufcanmodify = 1*/;
/* unsigned char zbuforimg;  flag for MSWIN_update_bitmap. Set while Image|Open or ZBuf|Open */
HBITMAP hBitmap, hOldBitmap;
HBRUSH hBrush, hOldBrush;
HPEN hPen, hOldPen;
HDC hMemoryDC;
HMENU hMenu/*, hPop*/;
HPALETTE hPal, hOldPal1 = 0, hOldPal2 = 0, hZBufPal;
HWND myhWnd;     
char TrueCol=1;
int resize_by_Quat = 0; /* flag whether user resize or by the program itself */
int already_user_resized = 0; /* flag whether user changed windows size */        
int is_maximized;    
int redraw_background;   
int hide_scrollbar;
int ENDFLAG=0; /* Set by WM_CLOSE, used by MSWIN_check_event */
extern time_t calc_time; 
time_t old_time;      
int pixperevent = 4;

struct frac_struct *frac;
struct view_struct *view;
struct realpal_struct *realpal;
double *cut_buf;
char *colscheme; 
char *act_file; 
char *act_name;

BOOL InitApplication(HINSTANCE);
BOOL App_End(HINSTANCE hInstance);
BOOL InitInstance(HINSTANCE, int);
void TitleImage(void);
LRESULT CALLBACK MainWndProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK About(HWND, UINT, WPARAM, LPARAM);
int StartCalc(HWND hWnd);
int CalcZBUF(HWND hWnd);
int DoIMGOpen(HWND hWnd, char *givenfile);
int DoZBUFOpen(HWND hWnd, char *givenfile);
int DoIMGClose(HWND hWnd);
int DoZBUFClose(HWND hWnd);
int DoIMGSave(HWND hWnd);
int DoZBUFSave(HWND hWnd);
int DoIMGSaveAs(HWND hWnd);
int DoZBUFSaveAs(HWND hWnd);
int DoContinue(HWND hWnd);
int DoChangeName(HWND hWnd);
int Cleanup(void);
int InitMem(int x, int y, char *Error, char zbuf);

HANDLE hInst;

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
   MSG msg;                   
   char Error[256], file[256], file2[256];
   int i;
   TEXTMETRIC tm;
   HDC hDC;

   /*
      GetSystemInfo
   if (GetWinFlags() & WF_80x87 == 0)   
   {
      MessageBox(NULL, "Sorry, FPU (Coprocessor) needed to run Quat.", "Quat: Error", MB_OK);
      PostQuitMessage(0);
      return(0L);
   }          
   */
   SetSlash('\\');
   frac = malloc(sizeof(struct frac_struct));
   view = malloc(sizeof(struct view_struct));
   realpal = malloc(sizeof(struct realpal_struct));
   cut_buf = malloc(sizeof(double)*140);
   colscheme = malloc(251);
   act_file = malloc(256);
   act_name = malloc(80);
   if (!frac || !view || !realpal || !cut_buf || !colscheme || !act_file || !act_name)
   {
      free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
      free(act_file); free(act_name);
      MessageBox(NULL, "Not enough memory to allocate even basic structures.", "Quat: Error", MB_OK);
      PostQuitMessage(0);
      return 0L;
   }     

   hDC = GetDC(NULL);
   if (!GetTextMetrics(hDC, &tm)) minsize = 400;
   else minsize = 42*tm.tmAveCharWidth;
   ReleaseDC(NULL, hDC);

/*   if (GetCharWidth(GetDC(NULL), 'a', 'a', &i))
   {
      minsize = i*42;
   }
   else minsize = 400; */
   if (!InitApplication(hInstance))
      return (FALSE);   

   if (!InitInstance(hInstance, nCmdShow))
      return (FALSE);
   
   TitleImage();
   strncpy(file, lpCmdLine, 255);
   /* for (i=0; i<(int)strlen(file) && file[i]!=' '; i++); */
   /* if (file[i]==' ') file[i] = 0; */
   if (file[0]!=0) 
   {
      for (i=0; i<(int)strlen(file)+1; i++) file2[i] = tolower((int)file[i]);
      sprintf(Error, "Don't know what to do with given file '%s'.", file);
      if (strrchr(file2, '.'))
      {
         if (strcmp(strrchr(file2, '.'), ".png")==0) DoIMGOpen(myhWnd, file);
         else if (strcmp(strrchr(file2, '.'), ".zpn")==0) DoZBUFOpen(myhWnd, file);
         else if (strcmp(strrchr(file2, '.'), ".ini")==0 || strcmp(strrchr(file2, '.'), ".col") == 0)
            DoReadINI(myhWnd, file);
         else MessageBox(myhWnd, Error, "Quat: Error", MB_OK);
      }
      else MessageBox(myhWnd, Error, "Quat: Error", MB_OK);
   }     
   while (GetMessage(&msg, 0, 0, 0))        
   {
      TranslateMessage(&msg);  
      DispatchMessage(&msg);
      if (!imginmem)
      {
         if (!zbufinmem) EnableMenuItem(hMenu, (UINT) IDM_Image_Open, MF_ENABLED);
         else EnableMenuItem(hMenu, (UINT) IDM_Image_Open, MF_GRAYED);
         if (zbufinmem && !zbufready) EnableMenuItem(hMenu, (UINT) IDM_Image_Calculate, MF_GRAYED);
         else EnableMenuItem(hMenu, (UINT) IDM_Image_Calculate, MF_ENABLED);
         EnableMenuItem(hMenu, (UINT) IDM_Image_Close, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) IDM_Image_Save, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) IDM_Image_SaveAs, MF_GRAYED);
         if (zbufinmem)
         {                                                              
            EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_INI, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_PNG, MF_GRAYED);
            EnableMenuItem(hMenu, (UINT) IDM_Param_New, MF_GRAYED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_OPEN, MF_GRAYED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_CLOSE, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVEAS, MF_ENABLED);
            if (zbufmodified) EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVE, MF_ENABLED);
            else EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVE, MF_GRAYED);
            if (zbufready) EnableMenuItem(hMenu, (UINT) ID_CALCULATION_STARTRESUMEZBUF, MF_GRAYED);
            else EnableMenuItem(hMenu, (UINT) ID_CALCULATION_STARTRESUMEZBUF, MF_ENABLED);
         }    
         else 
         {
            EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_INI, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_PNG, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) IDM_Param_New, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) ID_CALCULATION_STARTRESUMEZBUF, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_OPEN, MF_ENABLED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_CLOSE, MF_GRAYED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVEAS, MF_GRAYED);
            EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVE, MF_GRAYED);
         }
      }
      else 
      { 
         EnableMenuItem(hMenu, (UINT) IDM_Image_Open, MF_GRAYED);
         if (imgready) EnableMenuItem(hMenu, (UINT) IDM_Image_Calculate, MF_GRAYED);
         else EnableMenuItem(hMenu, (UINT) IDM_Image_Calculate, MF_ENABLED);
         EnableMenuItem(hMenu, (UINT) IDM_Image_Close, MF_ENABLED);
         if (imgmodified) EnableMenuItem(hMenu, (UINT) IDM_Image_Save, MF_ENABLED); 
         else EnableMenuItem(hMenu, (UINT) IDM_Image_Save, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) IDM_Image_SaveAs, MF_ENABLED);
         EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_INI, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_PNG, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) IDM_Param_New, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_OPEN, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_CLOSE, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVE, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVEAS, MF_GRAYED);
         EnableMenuItem(hMenu, (UINT) ID_CALCULATION_STARTRESUMEZBUF, MF_GRAYED);
      }
   }
   
   App_End(hInstance);

   free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
   free(act_file); free(act_name); 
   Cleanup();
 /*  DestroyMenu(hPop);*/
 /*  DestroyMenu(hMenu);*/
   return (msg.wParam);     
}

BOOL InitApplication(hInstance)
HINSTANCE hInstance;                  
{
   WNDCLASS  wc;
   ATOM A1, A2, A3, A4, A5, A6;
                               
   wc.style = 0;                   
   wc.lpfnWndProc = MainWndProc;     
   wc.cbClsExtra = 0;                
   wc.cbWndExtra = 0;                 
   wc.hInstance = hInstance;       
   wc.hIcon = LoadIcon(hInstance, "IDI_ICON");
   wc.hCursor = LoadCursor(0, IDC_ARROW);
   wc.hbrBackground = GetStockObject(WHITE_BRUSH);
   wc.lpszMenuName = NULL; 
   wc.lpszClassName = "QUATWClass092";

   A1 = RegisterClass(&wc);
                              
   wc.style = 0;                   
   wc.lpfnWndProc = MandelSelector;     
   wc.cbClsExtra = 0;                
   wc.cbWndExtra = 0;                 
   wc.hInstance = hInstance;       
   wc.hCursor = LoadCursor(0, IDC_ARROW);
   wc.hbrBackground = GetStockObject(BLACK_BRUSH);
   wc.lpszMenuName = NULL; 
   wc.lpszClassName = (LPSTR)"QUATMandelSelector";
   A2 = RegisterClass((LPWNDCLASS)&wc);
   wc.lpfnWndProc = JuliaPreview;
   wc.lpszClassName = (LPSTR)"QUATJuliaPreview";
   A3 = RegisterClass((LPWNDCLASS)&wc);

   wc.style = 0; wc.lpfnWndProc = ViewSelector;     
   wc.cbClsExtra = 0; wc.cbWndExtra = 0;                 
   wc.hInstance = hInstance;       
   wc.hCursor = LoadCursor(0, IDC_ARROW);
   wc.hbrBackground = GetStockObject(BLACK_BRUSH);
   wc.lpszMenuName = NULL; 
   wc.lpszClassName = "QUATViewSelector";
   A4 = RegisterClass(&wc);

   wc.style = 0; wc.lpfnWndProc = PalettePreview;     
   wc.cbClsExtra = 0; wc.cbWndExtra = 0;                 
   wc.hInstance = hInstance;       
   wc.hCursor = LoadCursor(0, IDC_ARROW);
   wc.hbrBackground = GetStockObject(BLACK_BRUSH);
   wc.lpszMenuName = NULL; 
   wc.lpszClassName = "QUATPalettePreview";
   A5 = RegisterClass(&wc);
   wc.lpfnWndProc = ColorClipboard;
   wc.lpszClassName = "QUATClipboard";
   A6 = RegisterClass(&wc);

   if (A1 && A2 && A3 && A4 && A5 && A6) return(TRUE);
   else return(FALSE);
}

BOOL App_End(HINSTANCE hInstance)
{
   DestroyWindow(myhWnd);
   return(0);
}

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
   HWND hWnd;
   RECT R;            
    
   hInst = hInstance;

#ifndef WINELIB
   hMenu = LoadMenu(hInst, "MAINMENU");            
#else
   hMenu = LoadMenuIndirect(_Resource_Men_MAINMENU_0_data);
#endif
   R.top = 0; R.left = 0; R.bottom = 1; R.right = minsize;
   AdjustWindowRectEx(&R, WS_OVERLAPPEDWINDOW,TRUE,0);
   pixx = R.right - R.left; pixy = R.bottom-R.top-1;
   resize_by_Quat = 1;
   hWnd = CreateWindow("QUATWClass092", PROGNAME, WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL, CW_USEDEFAULT,             
      CW_USEDEFAULT, pixx, pixy, 0, hMenu, hInstance, NULL);

   if (!hWnd)
      return (FALSE);

   myhWnd = hWnd;
   ShowWindow(hWnd, nCmdShow);  
   UpdateWindow(hWnd);         
   resize_by_Quat = 0;
   return (TRUE);          
}

void TitleImage()
{
   long i;
   unsigned char *line;
   z_stream d_stream;
   int err;
   char Error[256];

   if (TitleImg_Length==0 || TitleImg_X==0 || TitleImg_Y==0) return;
   line = (unsigned char *)calloc(TitleImg_X*3+10, 1);
   if (!line) return;
   MSWIN_Initialize((int)TitleImg_X, (int)TitleImg_Y, Error);
   d_stream.zalloc = (alloc_func)0;
   d_stream.zfree = (free_func)0;
   inflateInit(&d_stream);
   d_stream.avail_in = TitleImg_Length;
   d_stream.next_in = &(TitleImg[0]);
   for (i=0; i<TitleImg_Y; i++) 
   {
      d_stream.avail_out = TitleImg_X*3;
      d_stream.next_out = &(line[0]);
      err = inflate(&d_stream, Z_PARTIAL_FLUSH);
      MSWIN_update_bitmap(0,TitleImg_X-1,TitleImg_X,i,line,0);
   }
   inflateEnd(&d_stream);
/*   picx = TitleImg_X; picy = TitleImg_Y; */
   free(line);
}

LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, 
   LPARAM lParam)
{
   HDC hDC;
   UINT i;
    
   switch (message)
   {
      case WM_CREATE:
      {                                              
         char Error[256]; 
         /* int i; */
         
         /* for (i=0; i<140; i++) cut_buf[i] = 0; */
         SetDefaults(frac, view, realpal, colscheme, cut_buf); 
         if (BuildName(act_file, act_name, ".png", "Noname", Error))
         {                   
            MessageBox(hWnd, Error, "Quat: Error", MB_OK);
            free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
            free(act_file); free(act_name);
            PostQuitMessage(0);
            return 0L;
         }       
         strncpy(act_name, act_file, 80);
         /* its no Error, but "Error" is a free string */
         sprintf(Error, "%s - %s", PROGNAME, act_name);
         SetWindowText(hWnd, Error);    
         ShowScrollBar(hWnd, SB_BOTH, FALSE);
         break;
      }
      case WM_QUERYNEWPALETTE:
         if (hPal!=0)
         {
            HPALETTE hOldPal;
             
            hDC = GetDC(hWnd);
            hOldPal = SelectPalette(hDC, hPal, 0);
            i = RealizePalette(hDC); 
            SelectPalette(hDC, hOldPal, 0);
            ReleaseDC(hWnd, hDC);
            if (i) 
            {
               InvalidateRect(hWnd, (LPRECT)NULL, 1);
               return 1;
            }
         }
         return FALSE;
         break;
      case WM_PALETTECHANGED:
         if ((HWND)wParam != hWnd)
         {
            if (hPal)
            {
               HPALETTE hOldPal;
               hDC = GetDC(hWnd);
               hOldPal = SelectPalette(hDC, hPal, 0);
               i = RealizePalette(hDC);
               if (i != 0) 
                  InvalidateRect(hWnd, (LPRECT) NULL, 1);
               SelectPalette(hDC, hOldPal, 0);
               ReleaseDC(hWnd, hDC);
            }
         }
         break;
      case WM_SIZE:
      {        
         RECT R;
         int old, old2;                                                  
               
         if (!resize_by_Quat && wParam != SIZE_MAXIMIZED && !is_maximized) 
            already_user_resized = 1;
         if (IsZoomed(hWnd)) EnableMenuItem(hMenu, (UINT) IDM_ADJUST, MF_GRAYED);
         else if (is_maximized) EnableMenuItem(hMenu, (UINT) IDM_ADJUST, MF_ENABLED);
         is_maximized = IsZoomed(hWnd);
         SetScrollPos(myhWnd, SB_HORZ, 0, FALSE);
         SetScrollPos(myhWnd, SB_VERT, 0, FALSE);
         pixx = (int)LOWORD(lParam); pixy = (int)HIWORD(lParam);
         xadd = (pixx-picx)/2; if (xadd<0) xadd=0;
         yadd = (pixy-picy)/2; if (yadd<0) yadd=0; 
         old = resize_by_Quat;
         old2 = redraw_background;
         resize_by_Quat = 1;
         redraw_background = 1;   /* against flickering when introducing scrollbars */
         if (pixx<picx && !hide_scrollbar) 
         {
            ShowScrollBar(myhWnd, SB_HORZ, TRUE); 
            SetScrollRange(myhWnd, SB_HORZ, 0, picx-pixx, FALSE);
            SetScrollPos(myhWnd, SB_HORZ, 0, TRUE);
         }
         else ShowScrollBar(myhWnd, SB_HORZ, FALSE); 
         if (pixy<picy && !hide_scrollbar) 
         {
            ShowScrollBar(myhWnd, SB_VERT, TRUE); 
            SetScrollRange(myhWnd, SB_VERT, 0, picy-pixy, FALSE);
            SetScrollPos(myhWnd, SB_VERT, 0, TRUE);
         }
         else ShowScrollBar(myhWnd, SB_VERT, FALSE); 
         redraw_background = old2;
         resize_by_Quat = old;
         R.top = 0; R.left=0; R.bottom = pixy; R.right= pixx;
         InvalidateRect(hWnd, &R, (redraw_background==0));
         UpdateWindow(hWnd);
         return(0);
         break;
      }     
      case WM_HSCROLL:
      {        
         RECT R;
         int nPos, pos, min, max;
        
         nPos = HIWORD(wParam);
         pos = GetScrollPos(hWnd, SB_HORZ);
         GetScrollRange(hWnd, SB_HORZ, &min, &max);
         switch((int)LOWORD(wParam))
         {                   
            case SB_LEFT: pos = min; break;
            case SB_RIGHT: pos = max; break;
            case SB_LINELEFT: pos--; break;
            case SB_LINERIGHT: pos++; break;
            case SB_PAGELEFT: pos -= 20; break;
            case SB_PAGERIGHT: pos += 20; break;
            case SB_THUMBPOSITION: pos = nPos; break;
            case SB_THUMBTRACK: pos = nPos; break;
         }                                 
         if (pos<min) pos = min;
         if (pos>max) pos = max;
         SetScrollPos(hWnd, SB_HORZ, pos, TRUE);
         xadd = -pos;
         R.top = 0; R.left=0; R.bottom = pixy; R.right= pixx;
         InvalidateRect(hWnd, &R, FALSE);
         return(TRUE);
         break;
      }
      case WM_VSCROLL:
      {        
         RECT R;
         int nPos, pos, min, max;
        
         nPos = HIWORD(wParam);
         pos = GetScrollPos(hWnd, SB_VERT);
         GetScrollRange(hWnd, SB_VERT, &min, &max);
         switch((int)LOWORD(wParam))
         {                   
            case SB_LEFT: pos = min; break;
            case SB_RIGHT: pos = max; break;
            case SB_LINELEFT: pos--; break;
            case SB_LINERIGHT: pos++; break;
            case SB_PAGELEFT: pos -= 20; break;
            case SB_PAGERIGHT: pos += 20; break;
            case SB_THUMBPOSITION: pos = nPos; break;
            case SB_THUMBTRACK: pos = nPos; break;
         }                                 
         if (pos<min) pos = min;
         if (pos>max) pos = max;
         SetScrollPos(hWnd, SB_VERT, pos, TRUE);
         yadd = -pos;
         R.top = 0; R.left=0; R.bottom = pixy; R.right= pixx;
         InvalidateRect(hWnd, &R, FALSE);
         return(TRUE);
         break;
      }
      case WM_PAINT:
      {
         PAINTSTRUCT ps; 
         RECT R;
         HPALETTE hOldPal;
                         
         if (GetUpdateRect(hWnd, &R, TRUE))
         {
            hDC = BeginPaint(hWnd, &ps);
            BitBlt(hDC, R.left, R.top, R.right-R.left, R.bottom-R.top, hMemoryDC, R.left-xadd, R.top-yadd, SRCCOPY);
            hOldPal = SelectPalette(hDC, hPal, 0);
            RealizePalette(hDC); 
            SelectPalette(hDC, hOldPal, 0);
            EndPaint(hWnd, &ps);
        /*    ReleaseDC(hWnd, hDC);  */   
            return 0;
         }
         return(TRUE);
         break;
      }
      case WM_COMMAND:    
         switch (wParam)
         {
            case IDM_Exit:
            {
               if (DoIMGClose(hWnd)==-1) return(0);
               if (DoZBUFClose(hWnd)==-1) return(0);
               free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
               free(act_file); free(act_name); 
               Cleanup();
             /*  DestroyMenu(hPop);*/
             /*  DestroyMenu(hMenu);*/
               PostQuitMessage(0);
               break;
            }       
            case IDM_ADJUST:
            {
               RECT R;          
               int pixx, pixy;
               
               resize_by_Quat = 1;
               pixx = picx; pixy = picy;                              
               if (pixx<minsize) pixx=minsize;
               R.top = 0; R.left = 0; R.bottom = pixy; R.right = pixx;
               AdjustWindowRectEx(&R, WS_OVERLAPPEDWINDOW,TRUE,0);
               pixx = R.right-R.left;                     
               pixy = R.bottom-R.top;                           
               hide_scrollbar = 1;  /* WM_SIZE would reset it again */
               redraw_background = 1;
               ShowScrollBar(hWnd, SB_BOTH, FALSE);
               hide_scrollbar = 0;   
               redraw_background = 0;
               SetWindowPos(myhWnd, (HWND)NULL, 0, 0, pixx, pixy, SWP_NOMOVE);
               resize_by_Quat = 0;
               R.left = 0; R.top = 0; R.bottom = pixy+1; R.right = pixx+1;
               InvalidateRect(myhWnd, &R, TRUE);
               /*UpdateWindow(myhWnd);
               */return(TRUE);
               break;
            }
            case IDM_About:
            {                             
#ifndef WIN32
               DLGPROC dlgprc;
               dlgprc = MakeProcInstance((FARPROC)About, hInst);
               DialogBox(hInst, "ID_ABOUTBOX", hWnd, dlgprc);
               FreeProcInstance((FARPROC)dlgprc);
#else
#   ifndef WINELIB
               DialogBox(hInst, "ID_ABOUTBOX", hWnd, (DLGPROC)About);
#   else
               DialogBoxIndirect(hInst, _Resource_Dlg_ID_ABOUTBOX_0_data, hWnd, (DLGPROC)About);
#   endif
#endif		
               return(TRUE);
               break;
            }
            case IDM_Image_Open:
            {
               DoIMGOpen(hWnd, NULL);
               return(TRUE);
               break;
            }
            case IDM_Image_Close:
            {
               DoIMGClose(hWnd);
               return(TRUE);
               break;
            }
            case IDM_Image_Save:
            {
               DoIMGSave(hWnd);
               return(TRUE);
               break;
            }
            case IDM_Image_SaveAs:
            {
               DoIMGSaveAs(hWnd);
               return(TRUE);
               break;
            }
            case IDM_Image_Calculate:
            {
               if (StartCalc(hWnd)!=-2) return(TRUE);
               else
               {
                  free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
                  free(act_file); free(act_name);
                  Cleanup();
              /*    DestroyMenu(hPop);
                  DestroyMenu(hMenu);*/
                  PostQuitMessage(0);
                  return 0L;
               }    
               return(TRUE);
               break;
            }          
            case ID_CALCULATION_STARTRESUMEZBUF:
            {
               if (CalcZBUF(hWnd)!=-2) return(TRUE);
               else
               {
                  free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
                  free(act_file); free(act_name);
                  Cleanup();
               /*   DestroyMenu(hPop);
                  DestroyMenu(hMenu); */
                  PostQuitMessage(0);
                  return 0L;
               }    
               return(TRUE);
               break;
            }
            case ID_ZBUFFER_OPEN:
            {
               DoZBUFOpen(hWnd, NULL);
               return(TRUE);
               break;
            }       
            case ID_ZBUFFER_CLOSE:
            {
               DoZBUFClose(hWnd);
               return(TRUE);
               break;
            }       
            case ID_ZBUFFER_SAVE:
            {
               DoZBUFSave(hWnd);
               return(TRUE);
               break;
            }
            case ID_ZBUFFER_SAVEAS:
            {
               DoZBUFSaveAs(hWnd);
               return(TRUE);
               break;
            }
            case IDM_PARAM_FROM_INI:
            {
               DoReadINI(hWnd, NULL);
               return(TRUE);
               break;
            }                      
            case IDM_PARAM_FROM_PNG:
            {
               DoReadPNG(hWnd);
               return(TRUE);
               break;
            }    
            case IDM_Param_New:
            {
               DoNew(hWnd);
               return(TRUE);
               break;
            }        
            case IDM_Param_Save:
            {               
               DoParamSave(hWnd, "INI files (*.ini)|*.ini", "ini",
                  PS_OBJ + PS_VIEW + PS_COL + PS_OTHER + PS_USEFILE);
               return(TRUE);
               break;
            }
            case IDM_Param_Obj:
            {
#ifndef WIN32
               DLGPROC dlgprc;  
               dlgprc = MakeProcInstance((FARPROC)DoObjectEditor, hInst);
               DialogBox(hInst, "ID_OBJECTEDITOR", hWnd, dlgprc);
               FreeProcInstance((FARPROC)dlgprc);
#else
#   ifndef WINELIB
               DialogBox(hInst, "ID_OBJECTEDITOR", hWnd, (DLGPROC)DoObjectEditor);
#   else
               DialogBoxIndirect(hInst, _Resource_Dlg_ID_OBJECTEDITOR_0_data, hWnd, (DLGPROC)DoObjectEditor);
#   endif
#endif		
               return(TRUE);
               break;
            } 
            case IDM_Param_View:
            {                                                  
#ifndef WIN32
               DLGPROC dlgprc;
               dlgprc = MakeProcInstance((FARPROC)DoViewEditor, hInst);
               DialogBox(hInst, "ID_VIEWEDITOR", hWnd, dlgprc);
               FreeProcInstance((FARPROC)dlgprc);
#else
#   ifndef WINELIB 
               DialogBox(hInst, "ID_VIEWEDITOR", hWnd, (DLGPROC)DoViewEditor);
#   else
               DialogBoxIndirect(hInst, _Resource_Dlg_ID_VIEWEDITOR_0_data, hWnd, (DLGPROC)DoViewEditor); 
#   endif
#endif		
               return(TRUE);
               break;
            }  
            case IDM_Param_Col:
            {                                                  
#ifndef WIN32
               DLGPROC dlgprc;
               dlgprc = MakeProcInstance((FARPROC)DoColorEditor, hInst);
               DialogBox(hInst, "ID_COLOREDITOR", hWnd, dlgprc);
               FreeProcInstance((FARPROC)dlgprc);
#else
#   ifndef WINELIB		
               DialogBox(hInst, "ID_COLOREDITOR", hWnd, (DLGPROC)DoColorEditor);
#   else
               DialogBoxIndirect(hInst, _Resource_Dlg_ID_COLOREDITOR_0_data, hWnd, (DLGPROC)DoColorEditor);
#   endif
#endif		
               return(TRUE);
               break;
            }  
            case IDM_Param_Intersec:
            {
#ifndef WIN32
               DLGPROC dlgprc;
               dlgprc = MakeProcInstance((FARPROC)DoIntersecEditor, hInst);
               DialogBox(hInst, "ID_INTERSECEDITOR", hWnd, dlgprc);
               FreeProcInstance((FARPROC)dlgprc);
#else
#   ifndef WINELIB		
               DialogBox(hInst, "ID_INTERSECEDITOR", hWnd, (DLGPROC)DoIntersecEditor);
#   else
               DialogBoxIndirect(hInst, _Resource_Dlg_ID_INTERSECEDITOR_0_data, hWnd, (DLGPROC)DoIntersecEditor);
#   endif
#endif		
               return(TRUE);
               break;
            }
            case IDM_Param_Other:
            {
#ifndef WIN32
               DLGPROC dlgprc;
               dlgprc = MakeProcInstance((FARPROC)DoOtherEditor, hInst);
               DialogBox(hInst, "ID_OTHER", hWnd, dlgprc);
               FreeProcInstance((FARPROC)dlgprc);
#else 
#   ifndef WINELIB		
               DialogBox(hInst, "ID_OTHER", hWnd, (DLGPROC)DoOtherEditor);
#   else
               DialogBoxIndirect(hInst, _Resource_Dlg_ID_OTHER_0_data, hWnd, (DLGPROC)DoOtherEditor);  
#   endif
#endif		
               return(TRUE);
               break;
            }
            default:                 
               return (DefWindowProc(hWnd, message, wParam, lParam));
         }                       
      case WM_CLOSE:
      {
         if (DoIMGClose(hWnd)==-1) return(0);
         if (DoZBUFClose(hWnd)==-1) return(0);
         DestroyWindow(hWnd);                   
         ENDFLAG = 1;
         return(0);    /* when closed during calculation, flag reports to MSWIN_check_event */
      }
      case WM_DESTROY:        
      {
/*          free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
         free(act_file); free(act_name); 
         Cleanup();
         DestroyMenu(hPop);
         DestroyMenu(hMenu);*/
         PostQuitMessage(0);
         break;
      }                         
      default:                  /* Passes it on if unproccessed    */
         return (DefWindowProc(hWnd, message, wParam, lParam));
   }
   return (0);
}

LRESULT CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, 
   LPARAM lParam)
{
   char s[256];

    switch (message)
    {
       case WM_INITDIALOG:         
          if (strcmp(PROGSTATE, "")==0) SetDlgItemText(hDlg, IDC_STATICDEV, "");
		  sprintf(s, "%s %s%s", PROGNAME, PROGVERSION, PROGSTATE);
          SetDlgItemText(hDlg, IDC_FIRSTLINE, s);
          return (TRUE);

       case WM_COMMAND:               
          if (wParam == IDOK)
          {
             EndDialog(hDlg, TRUE); 
             return (TRUE);
          }
          break;
    }
    return (FALSE);              
}    
 
int MakeBitmapFromZBuffer(HWND hWnd)
{
  /* unsigned char _huge *Buf; */
   long i, j, mode, xres;
   BYTE C;
   long l, k;   
   UINT idx;
   HDC hDC;                
          
   hDC = GetDC(NULL);
   i = GetDeviceCaps(hDC, RASTERCAPS);
   if ((i & RC_PALETTE)!=0) mode = 1; else mode = 0;
   if (mode==1)
   {                                   
      HDC hDC2 = GetDC(myhWnd);
      SelectPalette(hDC2, hZBufPal, 0); 
      SelectPalette(hMemoryDC, hZBufPal, 0);
/*      DeleteObject(hPal);
      hPal = hzbufPal; */
      RealizePalette(hDC2);
      ReleaseDC(hWnd, hDC2);
   }
/* *A*   Buf = GlobalLock(ZBuf); */
/* und im folgenden alles ersetzt: Buf -> ZBuf */
   xres = view->xres*view->antialiasing;
   if (IsStereo(view)) xres *= 2;
   for (j=0; j<view->yres*view->antialiasing; j++)   
   {
      for (i=0; i<xres; i++) 
      {
         k = 3*(xres*j+i);
         l = (long)*(ZBuf+k) << 16 | (long)*(ZBuf+k+1) << 8 | (long)*(ZBuf+k+2); 
      /*   _fmemcpy((char _huge *)&l+1, (void _huge *)(Buf+k), 3);
         *l2 = *(l2+1); tmp = *(l2+3); *(l2+3) = *(l2+2); *(l2+2) = tmp;*/
         
         C = 255-(BYTE)(2.55*(float)l/(float)view->zres);
         /* indexed palette */                  
         if (mode!=0)
         {
            idx = GetNearestPaletteIndex(hZBufPal, RGB(C, C, C));
            SetPixel(hMemoryDC, i+1, j+1, PALETTEINDEX(idx));
         }
         /* else true color */
         else
            SetPixel(hMemoryDC, i+1, j+1, RGB(C, C, C));
      }
   }
/* *A*   GlobalUnlock(ZBuf);*/
   return(0);
}

int DoIMGOpen(HWND hWnd, char *givenfile)
{
   OPENFILENAME ofn;
   char chReplace, *filter, *file, *file_t, *Error, c=0;
   int i, xres;
   unsigned char ready;
   int xs, ys;
   struct frac_struct f;
   struct view_struct v;
   struct realpal_struct r;
   char colsch[251];
   double cutb[140];
                           
   filter = malloc(256); file = malloc(256); file_t = malloc(256); Error = malloc(300);
   if (!Error || !file || !filter || !file_t)
   {
      MessageBox(hWnd, "Could not allocate memory", "Quat: Error", MB_OK);
      free(Error); free(file_t); free(file); free(filter);
      return(-1);
   }
   file[0] = 0;
   if (!givenfile)
   {
      strcpy(filter, "PNG images (*.png)|*.png|All files (*.*)|*.*|"); 
      chReplace = filter[strlen(filter) - 1];
      for (i = 0; filter[i] != '\0'; i++) 
         if (filter[i] == chReplace) filter[i] = '\0';
   
      memset(&ofn, 0, sizeof(OPENFILENAME));
   
      ofn.lStructSize = sizeof(OPENFILENAME);
      ofn.hwndOwner = hWnd;
      ofn.lpstrFilter = filter;
      ofn.nFilterIndex = 1;
      ofn.lpstrFile = file;
      ofn.nMaxFile = 256;
      ofn.lpstrInitialDir = "";
      ofn.lpstrFileTitle = file_t;
      ofn.nMaxFileTitle = 256;
      ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
                           
      c = GetOpenFileName(&ofn);
   }
   else 
   {
      strncpy(file, givenfile, 256);
      strncpy(file_t, strrchr(givenfile, '\\')+1, 256);
   }
   if (givenfile || c)
   {                           
      ReturnVideoInfo = MSWIN_ReturnVideoInfo;
      SetColors = MSWIN_SetColors;
      Initialize = MSWIN_Initialize;
      Done = MSWIN_Done;
      update_bitmap = MSWIN_update_bitmap;
      getline = MSWIN_getline;                    
      putline = MSWIN_putline;
      check_event = DUMMY_check_event;
      Change_Name = MSWIN_Change_Name;
      Debug = MSWIN_Debug;
      eol = MSWIN_eol;
    
      /* zbuforimg = 0;  Flag: image */
      if (ReadParametersPNG(file, Error, frac, view, realpal, colscheme, cut_buf))
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         imgmodified = 0;
         DoIMGClose(hWnd);
         free(Error); free(file_t); free(file); free(filter);
         return(-1);
      }
      xres = view->xres*view->antialiasing; 
      if (view->interocular!=0) xres *= 2;
      if (InitMem(xres, view->yres, Error, 0)) 
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         free(Error); free(file_t); free(file); free(filter);
         return(-1);
      }
      if (ReadParametersAndImage(Error, file, /*&RGBmap,*/ &ready, &xs, &ys,
         &f, &v, &r, colsch, &cutb[0], 0)!=0)
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         imgmodified = 0;
         DoIMGClose(hWnd);
         free(Error); free(file_t); free(file); free(filter);
         return(-1);
      }                                                    
      strncpy(act_name, file_t, 79); strncpy(act_file, file, 255);
      imgready = ready;
      imgmodified = 0;
      canmodify = 0;   
      imginmem = 1;
      imgxstart = xs;
      imgystart = ys;
      *frac = f;
      *view = v;
      *realpal = r;
      strncpy(colscheme, colsch, 251);
      memcpy(cut_buf, &cutb[0], sizeof(double)*140);
      sprintf(Error, "%s - %s", PROGNAME, act_name);
      SetWindowText(hWnd, Error);
      if (ready) sprintf(Error, "Image read successfully. Calculation has been completed.");
      else sprintf(Error, "Image read successfully. Calculation is in line %i.", imgystart);
      MessageBox(hWnd, Error, "Quat: Message", MB_OK);
   }  
   free(Error); free(file_t); free(file); free(filter);
   return(0);
}

int DoZBUFOpen(HWND hWnd, char *givenfile)
{
   OPENFILENAME ofn;
   char chReplace, *filter, *file, *file_t, *Error, c=0;
   int i;     
   unsigned char ready;
   int xs, ys;
   struct frac_struct f;
   struct view_struct v;
   struct realpal_struct r;
   char colsch[251];
   double cutb[140];
   int xres;
                           
   filter = malloc(256); file = malloc(256); file_t = malloc(256); Error = malloc(300);
   if (!Error || !file || !filter || !file_t)
   {
      MessageBox(hWnd, "Could not allocate memory", "Quat: Error", MB_OK);
      free(Error); free(file_t); free(file); free(filter);
      return(-1);
   }
   file[0] = 0;
   if (!givenfile)
   {
      strcpy(filter, "ZBuffers (*.zpn)|*.zpn|All files (*.*)|*.*|"); 
      chReplace = filter[strlen(filter) - 1];
      for (i = 0; filter[i] != '\0'; i++) 
         if (filter[i] == chReplace) filter[i] = '\0';

      memset(&ofn, 0, sizeof(OPENFILENAME));
   
      ofn.lStructSize = sizeof(OPENFILENAME);
      ofn.hwndOwner = hWnd;
      ofn.lpstrFilter = filter;
      ofn.nFilterIndex = 1;
      ofn.lpstrFile = file;
      ofn.nMaxFile = 256;
      ofn.lpstrInitialDir = "";
      ofn.lpstrFileTitle = file_t;
      ofn.nMaxFileTitle = 256;
      ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
                           
      c = GetOpenFileName(&ofn);
   }
   else
   {
      strncpy(file, givenfile, 256);
      strncpy(file_t, strrchr(givenfile, '\\')+1, 256);
   }
   if (givenfile || c)
   {                           
      ReturnVideoInfo = MSWIN_ReturnVideoInfo;
      SetColors = MSWIN_SetColors;
      Initialize = MSWIN_Initialize;
      Done = MSWIN_Done;
      update_bitmap = MSWIN_update_bitmap;
      getline = MSWIN_getline;                    
      putline = MSWIN_putline;
      check_event = DUMMY_check_event;
      Change_Name = MSWIN_Change_Name;
      Debug = MSWIN_Debug;
      eol = MSWIN_eol;

      f = *frac; v = *view; r = *realpal;
      strncpy(colsch, colscheme, 251);
      memcpy(cutb, &cut_buf[0], sizeof(double)*140);
      i = ReadParametersPNG(file, Error, frac, view, realpal, colscheme, cut_buf);
      xres = view->xres*view->antialiasing; 
      if (view->interocular!=0) xres *= 2;
      if (InitMem(xres, view->yres*view->antialiasing, Error, 1)) 
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         free(Error); free(file_t); free(file); free(filter);
         return(-1);
      }
      i += ReadParametersAndImage(Error, file, &ready, &xs, &ys,
         frac, view, realpal, colscheme, &cut_buf[0], 1);
      if (i!=0)
      {        
         *frac = f; *view = v; *realpal = r;
         strncpy(colscheme, colsch, 251);
         memcpy(cut_buf, &cutb[0], sizeof(double)*140);
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         zbufmodified = 0;
         DoZBUFClose(hWnd);
         free(Error); free(file_t); free(file); free(filter);
         return(-1);
      }                                                    
/*      strncpy(act_name, file_t, 79); strncpy(act_file, file, 255);  */
      zbufready = ready;
      zbufmodified = 0;
      zbufinmem = 1;
      zbufxstart = xs;
      zbufystart = ys;
      /* sprintf(Error, "%s - %s", PROGNAME, act_name);
      SetWindowText(hWnd, Error);*/
      hZBufPal = hPal;
      if (ready) sprintf(Error, "ZBuffer read successfully. Calculation has been completed.");
      else sprintf(Error, "ZBuffer read successfully. Calculation is in line %i.", zbufystart);
      MessageBox(hWnd, Error, "Quat: Message", MB_OK);
   }                  
   free(Error); free(file_t); free(file); free(filter);
   return(0);
}

int DoIMGClose(HWND hWnd)
{            
   RECT R;
   char Error[256];
   
   if (!imginmem) return(0);
   if (imgmodified)
   {
      if (MessageBox(hWnd, "Image: There are unsaved modifications. Do you want to close?",
      "Quat: Confirmation", MB_YESNO)==IDNO) return(-1);
   }
   if (!already_user_resized && !IsZoomed(myhWnd) && !zbufinmem)
   {
      pixx = 0; pixy = 0;                              
      if (pixx<minsize) pixx=minsize;
      R.top = 0; R.left = 0; R.bottom = pixy; R.right = pixx;
      resize_by_Quat = 1;
      AdjustWindowRectEx(&R, WS_OVERLAPPEDWINDOW, TRUE,0);
      SetWindowPos(myhWnd, (HWND)NULL, 0, 0, R.right-R.left, R.bottom-R.top, SWP_NOMOVE);
      resize_by_Quat = 0;
   }        
   if (!zbufinmem) 
   {
      Cleanup();
      picx = 0; picy = 0;
      xadd = 0; yadd = 0;
   }
   else
   {
      picx = view->xres*view->antialiasing; picy = view->yres*view->antialiasing+2;
      if (IsStereo(view)) picx *= 2;                                              
      picx += 2;
      if (!already_user_resized && !IsZoomed(myhWnd))
      {
         pixx = picx; pixy = picy;
         if (pixx<minsize) pixx=minsize;
         R.top = 0; R.left = 0; R.bottom = pixy; R.right = pixx;
         resize_by_Quat = 1;
         AdjustWindowRectEx(&R, WS_OVERLAPPEDWINDOW, TRUE,0);
         SetWindowPos(myhWnd, (HWND)NULL, 0, 0, R.right-R.left, R.bottom-R.top, SWP_NOMOVE);
         resize_by_Quat = 0;
      }        
      {          
         HDC hDC;
                
         SelectObject(hMemoryDC, hOldBitmap);
         DeleteObject(hBitmap);
         DeleteObject(hPal);
         hDC = GetDC(myhWnd);
         bmx = picx; bmy = picy;
         hBitmap = CreateCompatibleBitmap(hDC, picx, picy);
         xadd = (pixx-picx)/2; yadd = (pixy-picy)/2;
         if (xadd<0) xadd = 0; if (yadd<0) yadd = 0;
         if (hBitmap!=NULL)
         {
            hOldBitmap = SelectObject(hMemoryDC, hBitmap);
            ReleaseDC(myhWnd, hDC);
            Rectangle(hMemoryDC, 0, 0, picx, picy);
            MakeBitmapFromZBuffer(hWnd);
            R.left = 0; R.top = 0; R.bottom = pixy+1; R.right = pixx+1;
            InvalidateRect(myhWnd, &R, TRUE);
            UpdateWindow(myhWnd);
         }
      }
   }
/* *A*   GlobalFree(RGBmap);*/
   free(RGBmap);
   RGBmap = NULL;
   calc_time = old_time;
   imgxstart = 0; imgystart = 0;
   imginmem = 0; imgnew = 1;
   imgready = 0; imgmodified = 0; canmodify = 1;
   if (BuildName(act_file, act_name, ".png", "Noname", Error))
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
      free(act_file); free(act_name);
      PostQuitMessage(0);
      return 0L;
   }       
   strncpy(act_name, act_file, 80);
   sprintf(Error, "%s - %s", PROGNAME, act_name);
   SetWindowText(hWnd, Error);    
   ShowScrollBar(hWnd, SB_BOTH, FALSE);
   if (!zbufinmem) TitleImage();
   R.left = 0; R.top = 0; R.bottom = pixy+1; R.right = pixx+1;
   InvalidateRect(myhWnd, &R, TRUE);
   
   return(0);
}    

int DoZBUFClose(HWND hWnd)
{            
   RECT R;
   char Error[256];
   
   if (!zbufinmem) return(0);
   if (zbufmodified)
   {
      if (MessageBox(hWnd, "ZBuffer: There are unsaved modifications. Do you want to close?",
      "Quat: Confirmation", MB_YESNO)==IDNO) return(-1);
   }
   if (!already_user_resized && !IsZoomed(myhWnd))
   {
      pixx = 0; pixy = 0;                              
      if (pixx<minsize) pixx=minsize;
      R.top = 0; R.left = 0; R.bottom = pixy; R.right = pixx;
      resize_by_Quat = 1;
      AdjustWindowRectEx(&R, WS_OVERLAPPEDWINDOW, TRUE,0);
      SetWindowPos(myhWnd, (HWND)NULL, 0, 0, R.right-R.left, R.bottom-R.top, SWP_NOMOVE);
      resize_by_Quat = 0;
   }        
   DeleteObject(hZBufPal);
   Cleanup();
/* *A*   GlobalFree(ZBuf);*/
   free(ZBuf);
   ZBuf = NULL;
   calc_time = 0; old_time = 0;
   picx = 0; picy = 0;
   xadd = 0; yadd = 0; zbufxstart = 0; zbufystart = 0;
   zbufinmem = 0;
   zbufready = 0; zbufmodified = 0;
   if (BuildName(act_file, act_name, ".png", "Noname", Error))
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      free(frac); free(view); free(realpal); free(cut_buf); free(colscheme);
      free(act_file); free(act_name);
      PostQuitMessage(0);
      return 0L;
   }       
   strncpy(act_name, act_file, 80);
   sprintf(Error, "%s - %s", PROGNAME, act_name);
   SetWindowText(hWnd, Error);    
   ShowScrollBar(hWnd, SB_BOTH, FALSE);
   TitleImage();
   R.left = 0; R.top = 0; R.bottom = pixy+1; R.right = pixx+1;
   InvalidateRect(myhWnd, &R, TRUE);
   
   return(0);
}    

int DoIMGSave(HWND hWnd)
{               
   char Error[256];
      
   if (SavePNG(Error, act_file, 0, imgystart, NULL, frac, view, realpal, 
      colscheme, cut_buf, 0)!=0)
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      return(-1);
   }
   imgmodified = 0;    
   MessageBox(hWnd, "Image written successfully.", "Quat: Message", MB_OK);
   return(0);
}

int DoZBUFSave(HWND hWnd)
{               
   char Error[256], file[256];
      
   if (BuildName(file, NULL, ".zpn", act_file, Error))
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      return(-1);
   }
   if (SavePNG(Error, file, 0, zbufystart, NULL, frac, view, realpal, 
      colscheme, cut_buf, 1)!=0)
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      return(-1);
   }
   zbufmodified = 0;         
   sprintf(Error, "ZBuffer '%s' written successfully.", file);
   MessageBox(hWnd, Error, "Quat: Message", MB_OK);
   return(0);
}

int DoIMGSaveAs(HWND hWnd)
{               
   OPENFILENAME ofn;
   char chReplace, filter[256], file[256], file_t[256], Error[300];
   int i;
      
/*   filter = malloc(256); file = malloc(256); file_t = malloc(256); Error = malloc(300);
   if (!Error || !file || !filter || !file_t)
   {
      MessageBox(hWnd, "Could not allocate memory", "Quat: Error", MB_OK);
      free(Error); free(file_t); free(filter); free(file);
      return(-1);
   }*/
   file[0] = 0;
   strcpy(filter, "PNG images (*.png)|*.png|All files (*.*)|*.*|"); 
   chReplace = filter[strlen(filter) - 1];
   for (i = 0; filter[i] != '\0'; i++) 
      if (filter[i] == chReplace) filter[i] = '\0';
   memset(&ofn, 0, sizeof(OPENFILENAME));
   ofn.lStructSize = sizeof(OPENFILENAME);
   ofn.hwndOwner = hWnd;
   ofn.lpstrFilter = filter;
   ofn.nFilterIndex = 1;
   ofn.lpstrFile = file;
   ofn.nMaxFile = 256;
   ofn.lpstrInitialDir = "";
   ofn.lpstrFileTitle = file_t;
   ofn.nMaxFileTitle = 256;
   ofn.lpstrDefExt = "png";
   ofn.Flags = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
   if (GetSaveFileName(&ofn))
   {                          
      if (SavePNG(Error, file, 0, imgystart, NULL, frac, view, realpal, 
         colscheme, cut_buf, 0)!=0)
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         return(-1);
      }
      imgmodified = 0;
      MessageBox(hWnd, "Image written successfully.", "Quat: Message", MB_OK);
      strncpy(act_file, file, 256); strncpy(act_name, file_t, 80);
      sprintf(Error, "%s - %s", PROGNAME, act_name);
      SetWindowText(hWnd, Error); 
   }
 /*  free(Error); free(file); free(filter); free(file);*/
   return(0);
}

int DoZBUFSaveAs(HWND hWnd)
{               
   OPENFILENAME ofn;
   char chReplace, filter[256], file[256], file_t[256], Error[300];
   int i;
      
/*   filter = malloc(256); file = malloc(256); file_t = malloc(256); Error = malloc(300);
   if (!Error || !file || !filter || !file_t)
   {
      MessageBox(hWnd, "Could not allocate memory", "Quat: Error", MB_OK);
      free(Error); free(file_t); free(filter); free(file);
      return(-1);
   }*/
   file[0] = 0;
      
   if (BuildName(file, NULL, ".zpn", act_file, Error))
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      free(Error); free(file_t); free(filter); free(file);
      return(-1);
   }
   strcpy(filter, "ZBuffers (*.zpn)|*.zpn|All files (*.*)|*.*|"); 
   chReplace = filter[strlen(filter) - 1];
   for (i = 0; filter[i] != '\0'; i++) 
   if (filter[i] == chReplace) filter[i] = '\0';
   memset(&ofn, 0, sizeof(OPENFILENAME));
   ofn.lStructSize = sizeof(OPENFILENAME);
   ofn.hwndOwner = hWnd;
   ofn.lpstrFilter = filter;
   ofn.nFilterIndex = 1;
   ofn.lpstrFile = file;
   ofn.nMaxFile = 256;
   ofn.lpstrInitialDir = "";
   ofn.lpstrFileTitle = file_t;
   ofn.nMaxFileTitle = 256;
   ofn.lpstrDefExt = "zpn";
   ofn.Flags = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
   if (GetSaveFileName(&ofn))
   {                           
      if (SavePNG(Error, file, 0, zbufystart, NULL, frac, view, realpal, 
         colscheme, cut_buf, 1)!=0)
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         return(-1);
      }
      zbufmodified = 0;        
      sprintf(Error, "ZBuffer '%s' written successfully.", file);
      MessageBox(hWnd, Error, "Quat: Message", MB_OK);
/*      strncpy(act_file, file, 256); strncpy(act_name, file_t, 80);
      sprintf(Error, "%s - %s", PROGNAME, act_name);
      SetWindowText(hWnd, Error);
*/   }
/*   free(Error); free(file); free(filter); free(file);*/
   return(0);
}

int MenuInCalcMode()
{
   EnableMenuItem(hMenu, (UINT) IDM_Stop, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Exit, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) IDM_Image_Open, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) IDM_Image_Close, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) IDM_Image_Save, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) IDM_Image_SaveAs, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) IDM_Image_Calculate, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_INI, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_PNG, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_New, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Save, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Obj, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_View, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Col, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Intersec, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Other, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) ID_CALCULATION_STARTRESUMEZBUF, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_OPEN, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_CLOSE, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVE, MF_GRAYED);    
   EnableMenuItem(hMenu, (UINT) ID_ZBUFFER_SAVEAS, MF_GRAYED);    
   
   return(0);
}           

int MenuAfterCalc()
{
   EnableMenuItem(hMenu, (UINT) IDM_Stop, MF_GRAYED);
   EnableMenuItem(hMenu, (UINT) IDM_Exit, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_INI, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_PARAM_FROM_PNG, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_New, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Save, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Obj, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_View, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Col, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Intersec, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) IDM_Param_Other, MF_ENABLED);
   EnableMenuItem(hMenu, (UINT) ID_CALCULATION_STARTRESUMEZBUF, MF_ENABLED);
   /* Image_Open,Close,Save,SaveAs are enabled by Winmain */
   /* ZBuffer_Open,Close,Save,SaveAs also. */
   return(0);
}

int StartCalc(HWND hWnd)
{                                    
   char *Error;
   int i, zflag, xres;
   
   Error = malloc(256);
   if (!Error)
   {
      MessageBox(hWnd, "DoCreate: Memory allocation failed.", "Quat: Error", MB_OK);
      return 0;
   }
 
   ReturnVideoInfo = MSWIN_ReturnVideoInfo;
   SetColors = MSWIN_SetColors;
   Initialize = MSWIN_Initialize;
   Done = MSWIN_Done;
   update_bitmap = MSWIN_update_bitmap;
   getline = MSWIN_getline;                    
   putline = MSWIN_putline;
   check_event = MSWIN_check_event;
   Change_Name = MSWIN_Change_Name;
   Debug = MSWIN_Debug;
   eol = MSWIN_eol;

   Error[0] = 0;
   MenuInCalcMode();
   imgmodified = 1; 
   /* zbuforimg = 0; Flag: Calc an image */
   if (zbufinmem) 
   {
      zflag = 2; old_time = calc_time;
   }
   else 
   {
      zflag = 0; old_time = 0;
   }
   
   xres = view->xres*view->antialiasing; 
   if (view->interocular!=0) xres *= 2;
   if (RGBmap==NULL) if (InitMem(xres, view->yres, Error, 0)) 
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      free(Error);
      return(-1);
   }
   
   i = CreateImage(Error, &imgxstart, &imgystart, frac, view, realpal, colscheme, cut_buf, 
      pixperevent, zflag);
   if (i==0 || i==-2 || i==-128) 
   { 
      imginmem = 1; imgnew = 1; canmodify = 0; 
   }
   if (imgystart==view->yres) imgready = 1;
   /* i == -2: Quat window is being destroyed */
   /* else: Error or calculation stopped */
   if (i!=-2)
   {
      if (Error[0]!=0)
      {
         MessageBox(hWnd, Error, "Quat: Message", MB_OK);
      }
      MenuAfterCalc();
   }        
   free(Error);
   return(i);
}

int CalcZBUF(HWND hWnd)          
{
   char *Error;
   int i, xres;
   
   Error = malloc(256);
   if (!Error)
   {
      MessageBox(hWnd, "DoCreate: Memory allocation failed.", "Quat: Error", MB_OK);
      return 0;
   }

   ReturnVideoInfo = MSWIN_ReturnVideoInfo;
   SetColors = MSWIN_SetColors;
   Initialize = MSWIN_Initialize;
   Done = MSWIN_Done;
   update_bitmap = MSWIN_update_bitmap;
   getline = MSWIN_getline;
   putline = MSWIN_putline;
   check_event = MSWIN_check_event;
   Change_Name = MSWIN_Change_Name;
   Debug = MSWIN_Debug;
   eol = MSWIN_eol;

   Error[0] = 0;
   MenuInCalcMode();
   zbufmodified = 1;                       
/*   zbuforimg = 1;  flag: calc ZBuffer */
      
   xres = view->xres*view->antialiasing; 
   if (view->interocular!=0) xres *= 2;
   if (ZBuf==NULL) if (InitMem(xres, view->yres*view->antialiasing, Error, 1)) 
   {
      MessageBox(hWnd, Error, "Quat: Error", MB_OK);
      free(Error);
      return(-1);
   }

   i = CreateZBuf(Error, &zbufxstart, &zbufystart, frac, view, cut_buf, pixperevent);
   if (i==0 || i==-2 || i==-128) 
   { 
      zbufinmem = 1; 
      hZBufPal = hPal;
   }
   if (zbufystart==view->yres) zbufready = 1;
   /* i == -2: Quat window is being destroyed */
   /* else: Error or calculation stopped */
   if (i!=-2)
   {
      if (Error[0]!=0)
      {
         MessageBox(hWnd, Error, "Quat: Message", MB_OK);
      }
      MenuAfterCalc();
   }        
   free(Error);
   return(i);
   
}

int MSWIN_ReturnVideoInfo(struct vidinfo_struct *vidinfo)
{                                                   
   HDC hDC; 
   int i;                              
  
   hDC = GetDC(myhWnd);
   i = GetDeviceCaps(hDC, RASTERCAPS);
   if ((i & RC_PALETTE)!=0)
   {
      vidinfo->maxcol = GetDeviceCaps(hDC, SIZEPALETTE)-GetDeviceCaps(hDC, NUMRESERVED);
      vidinfo->rdepth = GetDeviceCaps(hDC, COLORRES)/3;
      vidinfo->gdepth = GetDeviceCaps(hDC, COLORRES)/3;
      vidinfo->bdepth = GetDeviceCaps(hDC, COLORRES)/3; 
      /* The following lines are a workaround for WINELIB */
      if (vidinfo->rdepth==0) vidinfo->rdepth = 8;
      if (vidinfo->gdepth==0) vidinfo->gdepth = 8;
      if (vidinfo->bdepth==0) vidinfo->bdepth = 8;
      vidinfo->rgam = 0.6; vidinfo->ggam = 0.8; vidinfo->bgam = 0.6;    
      TrueCol = 0;
   }
   else
   {
     TrueCol = 1;           /* Internal flag */
     vidinfo->maxcol = -1;  /* flag for truecolor */
     vidinfo->rdepth = 8; vidinfo->gdepth = 8; vidinfo->bdepth = 8;
     vidinfo->rgam = 0.6; vidinfo->ggam = 0.8; vidinfo->bgam = 0.6;
   }
   ReleaseDC(myhWnd, hDC);
   
   return(0);
}            
                                          
int InitMem(int x, int y, char *Error, char zbuf)
{
   unsigned char *hg;
      
   hg = malloc((size_t)x*(size_t)y*3);
   if (!hg)
   {
      if (!zbuf && zbufinmem)
         sprintf(Error, "Not enough memory. Try to close the ZBuffer and calculate an image directly.");
      else if (!zbuf) sprintf(Error, "Couldnt allocate memory for PNG.");
      else sprintf(Error, "Couldnt allocate memory for ZBuffer.");
      return(-1);
   }
/*  *A*    hg = GlobalAlloc(GHND, (DWORD)x*(DWORD)y*3);
   if (hg==NULL)
   {
      if (zbuforimg==0 && zbufinmem)
         sprintf(Error, "Not enough memory. Try to close the ZBuffer and calculate an image directly.");
      else sprintf(Error, "Couldnt allocate memory for PNG.");
      return(-1);
   }     */
   if (!zbuf) RGBmap = hg;
   else ZBuf = hg;
   return(0);
}

int MSWIN_Initialize(int x, int y, char *Error)
{                                                 
   RECT R, R2, R3; 
   HDC hDC;          
                                                                                 
   /* +2 s because of the border */
   picx = x+2; picy = y+2;
   /* Check whether bitmap has right dimensions. It doesnt have if you switch from
      an AA-ZBuffer to an image */
   if (hBitmap!=0 && (bmx!=picx || bmy!=picy))
   {                 
      SelectObject(hMemoryDC, hOldBitmap);
      DeleteObject(hBitmap);
      hDC = GetDC(myhWnd);
      bmx = picx; bmy = picy;
      hBitmap = CreateCompatibleBitmap(hDC, picx, picy);
      if (hBitmap==NULL)
      {
         sprintf(Error, "Couldnt create bitmap.");
         return(-1);
      }             
      hOldBitmap = SelectObject(hMemoryDC, hBitmap);
      ReleaseDC(myhWnd, hDC);
      Rectangle(hMemoryDC, 0, 0, picx, picy);
      R.left = 0; R.top = 0; R.bottom = pixy+1; R.right = pixx+1;
      xadd = (pixx-picx)/2; yadd = (pixy-picy)/2;
      if (xadd<0) xadd = 0; if (yadd<0) yadd = 0;
      InvalidateRect(myhWnd, &R, TRUE);
      UpdateWindow(myhWnd);
   }
   if (!already_user_resized && !IsZoomed(myhWnd))
   {
      pixx = x+2; pixy = y+2;                              
      if (pixx<minsize) pixx = minsize;
      R.top = 0; R.left = 0; R.bottom = pixy; R.right = pixx;
      resize_by_Quat = 1;
      AdjustWindowRectEx(&R, WS_OVERLAPPEDWINDOW,TRUE,0);
      SetWindowPos(myhWnd, (HWND)NULL, 0, 0, R.right-R.left, R.bottom-R.top, SWP_NOMOVE);
      /* Check whether Windows did the size change */
      GetWindowRect(myhWnd, &R2);
      GetWindowRect(GetDesktopWindow(), &R3);  
      R.bottom -= R.top; R.top = 0;
      R.right -= R.left; R.left = 0;
      if (R2.right-R2.left<pixx) R.right = R3.right - R3.left;
      if (R2.bottom-R2.top<pixy) R.bottom = R3.bottom - R3.top;
      if (R2.right-R2.left<pixx || R2.bottom-R2.top<pixy) 
      {
         SetWindowPos(myhWnd, (HWND)NULL, 0, 0, R.right, R.bottom, 0);
         if (R2.right-R2.left<pixx) pixx = R.right-R.left;
         if (R2.bottom-R2.top<pixy) pixy = R.bottom-R.top;
      }
      resize_by_Quat = 0;
   }        
   xadd = (pixx-picx)/2; yadd = (pixy-picy)/2;
   if (xadd<0) xadd = 0; if (yadd<0) yadd = 0;
   SetScrollPos(myhWnd, SB_HORZ, 0, FALSE);
   SetScrollPos(myhWnd, SB_VERT, 0, FALSE);
   if (pixx<picx) 
   {
      ShowScrollBar(myhWnd, SB_HORZ, TRUE); 
      SetScrollRange(myhWnd, SB_HORZ, 0, picx-pixx, FALSE);
      SetScrollPos(myhWnd, SB_HORZ, 0, TRUE);
   }
   else ShowScrollBar(myhWnd, SB_HORZ, FALSE);
   if (pixy<picy) 
   {
      ShowScrollBar(myhWnd, SB_VERT, TRUE); 
      SetScrollRange(myhWnd, SB_VERT, 0, picy-pixy, FALSE);
      SetScrollPos(myhWnd, SB_VERT, 0, TRUE);
   }
   if (hBitmap==0)
   {
      /* not already initialized. This could be the case if a calculation was continued or 
         a zbuffer/img already in mem. */
      hDC = GetDC(myhWnd);
      bmx = picx; bmy = picy;
      hBitmap = CreateCompatibleBitmap(hDC, picx, picy);
      if (hBitmap==NULL)
      {
         sprintf(Error, "Couldnt create bitmap.");
         return(-1);
      }
      hMemoryDC = CreateCompatibleDC(hDC);
      hOldBitmap = SelectObject(hMemoryDC, hBitmap);
      hBrush = CreateSolidBrush(RGB(255, 255, 255));
      hOldBrush = SelectObject(hMemoryDC, hBrush);
      hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
      hOldPen = SelectObject(hMemoryDC, hPen);
      Rectangle(hMemoryDC, 0, 0, picx, picy);
      ReleaseDC(myhWnd, hDC);              
      R.left = 0; R.top = 0; R.bottom = pixy+1; R.right = pixx+1;
      InvalidateRect(myhWnd, &R, TRUE);
      UpdateWindow(myhWnd);
   } 
   return(0);
}
     
int MSWIN_Done()
{
   return(0);
}

void MSWIN_Debug(char *s)
{
   MessageBox(NULL, s, "Debug", MB_OK);
   return;
}
                                          
int MSWIN_SetColors(struct disppal_struct *disppal)
{            
   LOGPALETTE *plgpl;
   HDC hDC;
   int i, bits;
                                             
   plgpl = (LOGPALETTE *) malloc(sizeof(LOGPALETTE) + disppal->maxcol*sizeof(PALETTEENTRY));
   if (!plgpl) return(-1);
   hDC = GetDC(NULL);
   bits = GetDeviceCaps(hDC, COLORRES)/3;
   plgpl->palNumEntries=disppal->maxcol;
   plgpl->palVersion = 0x300;
   for (i=0; i<disppal->maxcol; i++)
   {
      plgpl->palPalEntry[i].peRed = disppal->cols[i].r<<(8-bits);
      plgpl->palPalEntry[i].peGreen = disppal->cols[i].g<<(8-bits);
      plgpl->palPalEntry[i].peBlue = disppal->cols[i].b<<(8-bits);
      plgpl->palPalEntry[i].peFlags = 0;
   }
   hPal = CreatePalette(plgpl);
   free(plgpl);
   ReleaseDC(NULL, hDC);
   hDC = GetDC(myhWnd);
   hOldPal1 = SelectPalette(hDC, hPal, 0); 
   hOldPal2 = SelectPalette(hMemoryDC, hPal, 0); 
   RealizePalette(hDC);
   ReleaseDC(myhWnd, hDC);  
   return(0);
}                

int MSWIN_update_bitmap(long x1, long x2, long xres, int y, unsigned char *Buf, int which)
{                           
   long i;         
   RECT Rect; 
   UINT idx;
   BYTE R=0, G=0, B=0;
   long l;
                  
   for (i=x1; i<=x2; i++)
   {
      if (which==0)
      {
         R = Buf[3*i]; G = Buf[3*i+1]; B = Buf[3*i+2];
      }
      else
      {
         l = (long)Buf[3*i] << 16 | (long)Buf[3*i+1] << 8 | (long)Buf[3*i+2];
         R = 255-(BYTE)(2.55*(float)l/(float)view->zres);
         G = R; B = R;
      }
      if (!TrueCol)
      {
         idx = GetNearestPaletteIndex(hPal, RGB(R, G, B));
         SetPixel(hMemoryDC, i+1, y+1, PALETTEINDEX(idx));
      }
      else
         SetPixel(hMemoryDC, i+1, y+1, RGB(R, G, B));  
   }  
   Rect.left = xadd+x1; Rect.top = yadd+y; Rect.right = xadd+x2+2; Rect.bottom = yadd+y+2;
   InvalidateRect(myhWnd, &Rect, FALSE);
   UpdateWindow(myhWnd);
   
   return(0);
}        

void MSWIN_eol(int line)
{
   char s[300];

   sprintf(s, "%s - %i lines of %s", PROGNAME, line, act_name);
   SetWindowText(myhWnd, s);
   return;
}

int MSWIN_getline(unsigned char *line, int y, long xres, int whichbuf)
{
   /* char _huge *Buf; */
   
/* *A*   if (whichbuf==0) Buf = GlobalLock(RGBmap);      
   if (whichbuf==1) Buf = GlobalLock(ZBuf);        
   if (Buf!=NULL) _fmemcpy((void _huge *)line, Buf+((long)y*(long)xres*3L), xres*3);
   if (whichbuf==0) GlobalUnlock(RGBmap);
   if (whichbuf==1) GlobalUnlock(ZBuf);
*/
   if (whichbuf==0) memcpy(line, RGBmap+((size_t)y*(size_t)xres*3L), xres*3);
   if (whichbuf==1) memcpy(line, ZBuf+((size_t)y*(size_t)xres*3L), xres*3);
   return(0);
}

int MSWIN_putline(long x1, long x2, long xres, int y, unsigned char *Buf, int whichbuf)
{
   /* char _huge *Buf; */
   
/* *A*   if (whichbuf==0) Buf = GlobalLock(RGBmap);      
   if (whichbuf==1) Buf = GlobalLock(ZBuf);        
   if (Buf!=NULL) _fmemcpy((void _huge *)line, Buf+((long)y*(long)xres*3L), xres*3);
   if (whichbuf==0) GlobalUnlock(RGBmap);
   if (whichbuf==1) GlobalUnlock(ZBuf);
*/
   if (whichbuf==0) memcpy(RGBmap+(y*xres+x1)*3L, Buf+x1*3L, (x2-x1+1)*3);
   if (whichbuf==1) memcpy(ZBuf+(y*xres+x1)*3L, Buf+x1*3L, (x2-x1+1)*3);
   return(0);
}

int MSWIN_check_event()
/* returns -2 if Quat window is being closed */
/* -1 if calculation is stopped */
{
   MSG msg;
   
   while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))   
   {
      TranslateMessage(&msg);
      DispatchMessage(&msg); 
      if (ENDFLAG) return(-2);   /* Window was closed */

      if ((msg.message==WM_COMMAND)&&(msg.wParam==IDM_Stop)) return(-128);
      else return(0);
   } 
   return(0);
}            

int MSWIN_Change_Name(char *s)
{
   char s2[256];

   strcpy(s2, "Quat - ");
   strcat(s2, s);
   SetWindowText(myhWnd, s2);
   return(0);
}
                       
int Cleanup()
{           
   if (hOldPal1!=0)
   {
      HDC hDC;
                          
      hDC = GetDC(myhWnd);
      SelectPalette(hDC, hOldPal1, 0);
      RealizePalette(hDC);
      ReleaseDC(myhWnd, hDC);   
      SelectPalette(hMemoryDC, hOldPal2, 0);
      DeleteObject(hPal);
      hOldPal1 = 0; hOldPal2 = 0;
   }
   if (hOldBitmap!=0) SelectObject(hMemoryDC, hOldBitmap);
   hOldBitmap = 0;
   DeleteObject(hBitmap);     
/*   DeleteObject(RGBmap); */
   if (hOldBrush!=0) SelectObject(hMemoryDC, hOldBrush);
   hOldBrush = 0;
   DeleteObject(hBrush); 
   if (hOldPen!=0) SelectObject(hMemoryDC, hOldPen);
   hOldPen = 0;
   DeleteObject(hPen);
   DeleteDC(hMemoryDC);   
   hBitmap = 0;
   return(0);
} 
   
