/* PhotoOrganizer - ThumbnailsPanel
 * Copyright (C) 1999-2000 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer.renderer;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.InputStream;
import java.io.IOException;

import javax.swing.*;
import javax.swing.event.MouseInputAdapter;
import javax.swing.border.*;
import javax.swing.tree.*;

import rogatkin.*;
import photoorganizer.formats.*;
import photoorganizer.*;

public class ThumbnailsPanel extends JPanel {
    public static final int THUMB_SIZE_X = 162;
    public static final int THUMB_SIZE_Y = 142; // should be 960*THUMB_SIZE_X/1280+FONT_HEIGHT
    static final int THUMBS_IN_ROW = 5;
    private int labelTextHeight = 16;
	// labelTextHeight = new Label("WWW").getPreferredSize().height;
	
    public ThumbnailsPanel(Controller controller) {
        this.controller = controller;
	collectionpanel = (PhotoCollectionPanel)controller.component(Controller.COMP_COLLECTION);
	albumpanel      = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
        setMinimumSize(Resources.MIN_PANEL_DIMENSION);
	setImageView();
	imagepanel.setThumbnailsPanel(this);
    }

    void setImageView() {
	imagepanel = (PhotoImagePanel)controller.component(Controller.COMP_IMAGEPANEL);
    }

    void calculateLayout() {
	Serializer s = controller.getSerializer();
	reg_border = ThumbnailsOptionsTab.createBorder(
	          (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.BORDER));
        selected_border = ThumbnailsOptionsTab.createBorder(
	          (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.SELECTEDBORDER));

	Integer i = (Integer)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.VERTAXIS);
	v=THUMBS_IN_ROW;
	h=0;
	if (i != null && i.intValue() == 0) {
	    i = (Integer)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.FIXAXIS);
	    if (i!= null) {
		v=0; h=i.intValue();
	    }
	} else {
	    i = (Integer)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.FIXAXIS);
	    if (i!= null) {
		h=0; v=i.intValue();
	    }
	}
        sx = THUMB_SIZE_X;
        sy = THUMB_SIZE_Y;
        i = (Integer)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.CELLWIDTH);
        if (i != null)
            sx = i.intValue();
        else
            s.setProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.CELLWIDTH, new Integer(sx));
        i = (Integer)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.CELLHEIGHT);
        if (i != null)
            sy = i.intValue();
        else
            s.setProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.CELLHEIGHT, new Integer(sy));
        setLayout(new GridLayout(h,v));
    }
    
    public Component createThumbnail(BasicJpeg format) {
	return new ThumbnailComponent(format);
    }

    public Dimension getPreferredSize() {
        return preferredsize;
    }
    
    public void updateImages(File file) {
        updateImages(new File[]{file});
    }

    public boolean updateImages(File []files) {
        removeAll();
        calculateLayout();
        BasicJpeg format;
	boolean fsel = false;
        for(int i = 0; i < files.length; i++) {
            if (controller.isLowMemory()) {
                JOptionPane.showMessageDialog(this, Resources.LABEL_OUTOFMEMORY,
                    Resources.TITLE_ERROR, JOptionPane.ERROR_MESSAGE);
                break;
            }
            if (files[i].isFile()) {
                format = new BasicJpeg(files[i]);
                if (format.isValid()) {
                    add(createThumbnail(format));
		    fsel = true;
		}
            } else if (files[i].isDirectory()) {
                String []names = files[i].list();
                String path = files[i].getPath();
                for(int n = 0; n < names.length; n++) {
                    if (controller.isLowMemory()) {
                        JOptionPane.showMessageDialog(this, Resources.LABEL_OUTOFMEMORY,
                            Resources.TITLE_ERROR, JOptionPane.ERROR_MESSAGE);
                        break;
                    }
                    format = new BasicJpeg(new File(path, names[n]));
                    if (format.isValid())
                        add(createThumbnail(format));
                }
            }
        }
        adjustDimension();
	return fsel;
    }

    public void addImage(File file) {
	calculateLayout();
        BasicJpeg format = new BasicJpeg(file);
	if (format.isValid())
	    add(createThumbnail(format));
        adjustDimension();
    }
    
    void adjustDimension() {
	if (v!=0)
	    preferredsize =
		new Dimension(v*sx, (getComponentCount()/v+1)*sy);
	else
	    preferredsize =
		new Dimension((getComponentCount()/h+1)*sx, h*sy);
        setSize(preferredsize);
	revalidate();
        repaint();
    }

    JPopupMenu getRightButtonMenu(ActionListener listener, boolean use_alternative) {
	return new FastMenu(listener, controller);
    }

    void doSpecificAction(BasicJpeg format, ActionEvent a, Thumbnail source) {
	String cmd = a.getActionCommand();
	if (cmd.equals(Resources.MENU_ADDTOCOLLECT)) {
	    collectionpanel.add(new File(format.getLocationName()));
	} else if (cmd.equals(Resources.MENU_ADDTOALBUM)) {
            AlbumSelectionDialog asd = albumpanel.getSelectionDialog();
            asd.setTitle(Resources.TITLE_SELECT_ALBUM+":"+format);
            asd.setVisible(true);
            TreePath[] tps = asd.getSelectedAlbums();
            if (tps != null) {
                albumpanel.addToAlbum(new BasicJpeg[]{format}, tps);
		Serializer s = controller.getSerializer();
		if (Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.MOVETOFOLDER), 0) == 1 &&
                    Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.USEALBUMFOLDER), 0) == 1)
		    ;

            }
	} else if (cmd.equals(Resources.MENU_SHOW)) {
	    showFullImage(format, source);
	} else if (cmd.equals(Resources.MENU_RENAME)) {
            Serializer s = controller.getSerializer();
	    boolean success = false;
            if (format.isValid() && Serializer.getInt(s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.ASKEDIT), 0) == 0) {
                success = format.renameTo(new File(format.getParentPath(),
		    FileNameFormat.makeValidPathName(RenameOptionsTab.getNewName(format, controller))));
            } else {
		String value;
                if (format.isValid())
                    value = format.getParentPath()+File.separatorChar+
                    FileNameFormat.makeValidPathName(RenameOptionsTab.getNewName(format, controller));
		else
		    value = format.getLocationName();
                value = (String)JOptionPane.showInputDialog(this,
                    Resources.LABEL_NEW_NAME, Resources.TITLE_RENAME,
                    JOptionPane.QUESTION_MESSAGE, null, null, value);
                if (value != null)
                    success = format.renameTo(new File(value.toString()));
            }
	    if (success)
		source.update();
	} else if (cmd.equals(Resources.MENU_DELETE)) {
	    Serializer s = controller.getSerializer();
	    if (Serializer.getInt(s.getProperty(MiscellaneousOptionsTab.SECNAME, MiscellaneousOptionsTab.SHOWWARNDLG), 0) == 1 &&
		JOptionPane.showConfirmDialog(this, Resources.LABEL_COFIRMDELETE+format.getLocationName(),
		Resources.TITLE_COFIRMATION, JOptionPane.OK_CANCEL_OPTION) != JOptionPane.OK_OPTION)
		return;
            File of = new File(format.getLocationName());
	    if (of.delete()) {
		remove((Component)source);
		adjustDimension();
	    }
	} else if (cmd.equals(Resources.MENU_PRINT)) {
            controller.print(new File[]{new File(format.getLocationName())});
        }
    }

    public String getImageTitle(BasicJpeg format, boolean thumbnail) {
        String o = null;
	boolean showComment = Serializer.getInt(controller.getSerializer().getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.SHOWCOMMENT), 0) == 1;
	if (showComment)
	    o = format.getComment();
	if (o == null || o.length() == 0)
	    o = (String)controller.getSerializer().getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.LABELMASK);
	AbstractImageInfo ii = format.getImageInfo();
        if (thumbnail) {
            if (o != null)
		return new FileNameFormat(o).format(format);
            else if (ii != null)
                return ""+ii.getDateTimeOriginal();
        } else {
            if (ii != null)
                return Resources.LABEL_NOTUMBNAIL+format.getImageInfo().getMake();
        }
	return Resources.LABEL_NOTUMBNAIL+format.getLocationName();
    }

    void showFullImage(BasicJpeg format, Thumbnail source) {
	imagepanel.updateView(format);
	controller.updateCaption(format.getLocationName());
	setCurrent(source);
    }

    public void setCurrent(Thumbnail tn) {
	current_thumb = tn;
    }

    public void showNext() {
	Thumbnail tn;
	for (int i=0; i<getComponentCount()-1; i++) {
	    if (current_thumb == getComponent(i)) {
		tn = (Thumbnail)getComponent(i+1);
		doSpecificAction(tn.getImageInfo(), new ActionEvent(tn, 0, Resources.MENU_SHOW), tn);
		break;
	    }
	}
    }

    public void showPrev() {
	Thumbnail tn;
	for (int i=getComponentCount()-1; i>0; i--) {
	    if (current_thumb == getComponent(i)) {
		tn = (Thumbnail)getComponent(i-1);
		doSpecificAction(tn.getImageInfo(), new ActionEvent(tn, 0, Resources.MENU_SHOW), tn);
		break;
	    }
	}
    }

    class ThumbnailComponent extends JLabel implements ActionListener, Thumbnail {
        Dimension preferredsize = new Dimension(ThumbnailsPanel.this.sx, ThumbnailsPanel.this.sy);
		Dimension thumbnailSize = new Dimension(preferredsize.width-2*2, preferredsize.height-2*2-labelTextHeight); // text height
        private BasicJpeg format;
        
        public ThumbnailComponent() {
            super(new ImageIcon());
            //setHorizontalAlignment(SwingConstants.CENTER)
	    setHorizontalTextPosition(SwingConstants.CENTER);
            setVerticalTextPosition(SwingConstants.BOTTOM);
            addMouseListener(new MouseInputAdapter() {
		public void mouseClicked(MouseEvent e) {
		    int m = e.getModifiers();
		    if (e.getClickCount() == 2 && (m & InputEvent.BUTTON1_MASK) > 0) {
			// show full image
                        doSpecificAction(format, new ActionEvent(ThumbnailComponent.this, 0, Resources.MENU_SHOW), ThumbnailComponent.this);
		    } else if ((m & InputEvent.BUTTON3_MASK) > 0)
			getRightButtonMenu(ThumbnailComponent.this, e.isControlDown()).show(ThumbnailComponent.this,
			    e.getX(), controller.adjustMenuY(ThumbnailComponent.this.getLocationOnScreen().y+e.getY(), 200)-ThumbnailComponent.this.getLocationOnScreen().y);
		}
	    });
	    setBorder(reg_border);
        }
        
        public Dimension getPreferredSize() {
            return preferredsize;
        }
	
	public void actionPerformed(ActionEvent a) {
	    String cmd = a.getActionCommand();
	    if (cmd.equals(Resources.MENU_PROPERTIES)) {
		PropertiesPanel.showProperties(format, controller);
	    } else 
		doSpecificAction(format, a, this);
	}
        
        public ThumbnailComponent(BasicJpeg format) {
            this();
            updateImage(format);
        }
        
        void updateImage(BasicJpeg format) {
            this.format = format;
	    String o = Serializer.arrayToString(controller.getSerializer().getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.TOOLTIPMASK));
            Icon ico = null;
            if (format.getImageInfo() != null)
                ico = format.getImageInfo().getThumbnailIcon(format, thumbnailSize);
	    if (o != null) {
		setToolTipText(new FileNameFormat(o).format(format));
	    } else
		setToolTipText(format.getImageInfo().getReport());
            if (ico != null) {
                setIcon(ico);
                setText(getImageTitle(format, true));
            } else
                setText(getImageTitle(format, false));
        }

        public void paint(Graphics g) {
            try {
                super.paint(g);
            } catch (NullPointerException e) {
                // work around Sun's VM bug
            }
        }

	public BasicJpeg getImageInfo() {
	    return format;
	}
	
	public void select(boolean on) {
	    setBorder(on?selected_border:reg_border);
	}

	public void update() {
	    updateImage(format);
	}

	public String toString() {
	    return new File(format.getLocationName()).getName();
	}
    }

    Dimension preferredsize = new Dimension(0, 160);
    Controller controller;
    private Thumbnail current_thumb;
    protected PhotoCollectionPanel collectionpanel;
    protected AlbumPane albumpanel;
    protected PhotoImagePanel imagepanel;
    private int v,h;
    private int sx, sy; // cell size
    private Border reg_border, selected_border;
}

