/** \file CON_Mesh.h
    3D Mesh and MeshBuilder interfaces

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_CON_MESH
#define H_CON_MESH

#include <istream.h>
#include <CON_Basic.h>
#include <CON_Misc.h>
#include <CON_Math3D.h>
#include <CON_Texture.h>

// Forward declarations
class SoundBuffer;

/**
A Mesh of triangles that represent an object in 3D Space.
The interface provides methods for: creating the mesh, moving it in 3D space and rendering it.
*/

class Mesh : public Interface, public Transformable
{
public:
  /** Render the object to the current scene using the 
      transformation given by transform.  If transform == NULL,
      the Mesh's own internal transform will be used. */
  virtual long render(const Matrix3D* transform=NULL) = 0;

  /** Returns status of object (0=ok, else error) */
  virtual long status() const = 0;

  /** Locate a 3D sound buffer in this object's location */
  virtual long orientSound(SoundBuffer* S) = 0;

  /** Check for collision with this object
      Returns a non-zero value if a collision is detected. If transform ==
      NULL, both this and Mesh O are to be transformed by their own internal 
      transformations; otherwise, the given transform is applied to O and is 
      assumed to be relative to this (no transform is applied to this). */
  virtual long collision(const Mesh* O, const Transformable* transform=NULL) const = 0;

  /** Check for collision with this object overriding both source and dest
      mesh transformables.
      Returns a non-zero value if a collision is detected. */
  virtual long collision(const Transformable& t1, const Mesh* O2, const Transformable& t2) const = 0;

  /** Should be called from main world advance if blinking materials are used. */
  virtual long advance(const float Fraction) = 0;

  /** Get the object's rough radius, (non-optimized bounding sphere) */
  virtual float    getRadius() const = 0;

  /** Set the object's radius, other than its actual one. */
  virtual long     setRadius(const float Radius) = 0;

  /** Exports the object to a text .X file */
  virtual long exportXFile(ostream& os) const = 0;

  /** Exports the model's data to the supplied areas.
      Specify NULL for any parameter to skip retrieval of that info.
      Vertices Array size should be  5 * VerticesNumber
      Faces Array size should be  4 * FacesNumber
      Materials Array size should be 17 * MaterialsNumber */
  virtual long exportData(float* Vertices, int* VerticesNumber,
                          short* Faces, int* FacesNumber, 
                          float* Materials, int* MaterialsNumber) const = 0;

/********************************************************
  section: Historical interface functions for backwards compatibility.  
  Use functions inherited from Transformable in new code. **/

  /** Set the object's position */
  virtual long setPosition(const float x, const float y, const float z) = 0;

  /** Move the object relative to its current position */
  virtual long relocate(const float dx, const float dy, const float dz) = 0;

  /** Rotate around any or all axes (angles in radians) */
  virtual long rotate(const float ax, const float ay, const float az) = 0;

  /** Set the position and orientation according to this object */
  virtual long align(const Mesh* O) = 0;

  /** Get direction vector, scaled to Size */
  virtual Vector3D direction(const float Size=1.0f) = 0;

  /** Get the up direction vector, scaled to Size */
  virtual Vector3D direction_up(const float Size=1.0f) = 0;

  /* scale the mesh by s. */
  virtual long scale(const float s) = 0;

  /* transform the mesh */
  virtual long transform(const Matrix3D& M) = 0;
};


/** Loads a mesh from a .X file. name specifies file name
    The Mesh* Parameter should be uninitialized,
    unlike previous versions!
*/
DLLExport long loadXFile(const char* Name, Mesh*& O, TextureCache* TC=NULL, float Scale=1.0f);


/** The MeshBuilder interface is used to construct objects from scratch.
    It need not be used when loading objects from .X files. */

class MeshBuilder : public Interface {
public:
  /** This clears all vertex, face and material information.  
      Use after finalize() to start building a new (fresh) mesh. */
  virtual void clear() = 0;

  /** return status code. */
  virtual long status() const = 0;

  /** Add a vertex (point) at the specified position, optionally specifying texture coordinates */
  virtual long addVertex(const Vector3D& p, const fVector2D& t=fVector2D()) = 0;

  /** Get how many vertices have been entered so far (useful for independent subobjects) */
  virtual int  getSetupVerticesNumber() const = 0;

  /** In case texture coords are loaded separately from the world coords, 
      it is possible to set them later. vertex index is required. */
  virtual long setVertexTextureCoords(const int index, const fVector2D& t=fVector2D(0.0f, 0.0f)) = 0;

  /** Add a triangle with a,b,c specifying the vertices that form this triangle
      and the material name specifies the material it is made of.
      Note: The index order determines its visible side (Clockwise) */
  virtual long addFace(const short a, const short b, const short c, const char* MaterialName, const int MaterialNumber=-1) = 0;

  /** Same as the previous, but uses the index of the material instead,
      In case you don't have its name */
  virtual long addFace(const short a, const short b, const short c, const int MaterialNumber) = 0;

  /** Add a material with the key name and the list of colors that make the material. */
  virtual long addMaterial(const char* Name, const TColor Diffuse, const TColor Specular, 
                           const TColor Emissive, const float Power, Texture* texture=NULL) = 0;

  /** Add a material, used internally by the .X import */
  virtual long addMaterial(const char* Name, const void* MaterialDesc, Texture* texture=NULL) = 0;

  /** This enters the vertex, face and material information into MeshBuilder
      from Mesh m. Note that the vertices are trasnformed by the Mesh's m_Transform
      before adding them.  */
  virtual long rebuild(const Mesh* m) = 0;

  /** This method can be used to calculate normals for models that have unordered 
      faces.  It returns a new MeshBuilder filled with the modified info. */
  virtual MeshBuilder* calculateNormals(ProgressMonitor* PM=NULL) const = 0;

  /** This method creates a copy of the current object, but reduces the polygon
      count approximately according to the reduction ratio.
      This is used for creating different Levels Of Details  (LODs).
      It returns a new MeshBuilder filled with the modified info.  */
  virtual MeshBuilder* createReducedObject(const float Reduction=0.75f, ProgressMonitor* PM=NULL) const = 0;

  /** this determines whether MeshBuilder::finalize() will output a normal or a billboard mesh.
      billboard Mesh's are always oriented to face the camera (used for 2D artwork in a 3D world).
      default parameter of 1 means 'yes' (create billboards). */
  virtual void setBillBoard(int flag=1) = 0;

  /** get billboard state */
  virtual int  getBillBoard() const = 0;

  /** Setup complete, generate a Mesh.
      This function must be run in order to process all data entered by the previous functions and
      produces a new Mesh.
      The 'JoinVertices' parameter determines if nearby vertices should be joined into one.
      nearby is determined if the square distance is less than this parameter.
      Therefore, -1.0 means 'No'.  GouraudThreshold (GT) determines the angle (in radians) between faces
      at which Gouraud shading should be applied (vertex normals are averaged when the angle is <= GT).
      In default situation (GT=fPI), a faster normal calculation method is used (the normals of 
      all faces sharing a vertex are averaged without regard to the angle between them).
      finalize() returns the newly genertaed Mesh.  If it returns NULL, an error occured.
      Chech status() for the cause.  finalize() does not modify the setup face, vertex or matarial
      info so it may be called multiple times to produce incrementally built models or ones
      with different materials or texture coords.
  */
  virtual Mesh* createMesh(const float JoinVertices=-1.0f, const float GouraudThreshold=fPI) = 0;

   /** Create a new Sphere.  This is not a billboard, but a real 3D Sphere.
       Coordinates specify the initial position for the mesh.
       Size is the radius of the sphere.
       A texture can be applied and will wrap the sphere. */
   virtual Mesh* createSphere(const Vector3D& pos, const float Size, Texture* texture, int Inv=0) = 0;

   /** Create a new BillBoard Sphere.  This is a a billboard that is simply a circle.
       Coordinates specify the initial position for the mesh.
       Note: this function clears MeshBuilder before and after generation. */
   virtual Mesh* createBillBoardSphere(const Vector3D& pos, const float Size, 
                                       const int Red, const int Green, const int Blue) = 0;
};


/** Use this to create new MeshBuilders.  If mb is not NULL, the returned
    MeshBuilder is initialized with the vertex, face, and material info from mb. */
DLLExport MeshBuilder* newMeshBuilder(const MeshBuilder* mb=NULL);

/** this also will create a new MeshBuilder, but it initialized with the
    info from the given Mesh. (same as {mb=newMeshBuilder(); mb->rebuild(m);} ).*/
DLLExport MeshBuilder* newMeshBuilder(const Mesh* m);

#endif // H_CON_MESH
