#include <stdlib.h>
#include <string.h>

#include "mathparser.h"

const entry keywords[]=
{
    entry("tan",  PAR_TN),
    entry("sin",  PAR_SN),
    entry("cos",  PAR_CS),
    entry("atan", PAR_ATN),
    entry("asin", PAR_ASN),
    entry("acos", PAR_ACS),
    entry("ln",   PAR_LN),
    entry("sqrt", PAR_QR),
    entry("exp",  PAR_EX),
    entry("rtod", PAR_R2D),                       
    entry("mod",  PAR_MOD),
    entry("pi",   PAR_PI),
    entry("",     0)
};

// Degrees To Rads = (180.0 / 3.141592654) = 57.29577951
// sin cos tan (deg / 57.29577951)

extern const char* ErrorStr[] =
{
    "success",
    "syntax error in a float number",
    "imcomplete expression",
    "unknown identifier",
    "internal compiler error",
    "syntax error in factor",
    NULL
};

// ----------------------------------------------------------------------------

entry::entry(char *s, int i)
{
    if ((lexptr = new char[strlen(s)+1]) != NULL)
    {
        strcpy (lexptr,s);
        token = i;
    }    
}

// ----------------------------------------------------------------------------

entry::~entry()
{
    if (lexptr != NULL)
        delete[] lexptr;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

parser::parser()
{
    int len;

    numbuf   = NULL;
    lexbuf   = NULL;
    symtable = NULL;

    // Find the number of the keyword list
    for(len = 0;keywords[len].token;len++);
    
    if ((symtable = new entry[len]) == NULL)
        return;

    symtable = (entry*) keywords;

    lastentry = len;
    if ((lexbuf = new char[PAR_BSIZE]) == NULL)
        return;
    
    if ((numbuf = new char[PAR_NSIZE]) == NULL)
        return;

    lineno = 1;
}

// ----------------------------------------------------------------------------

parser::~parser()
{
    if (numbuf != NULL)
        delete[] numbuf;

    if (lexbuf != NULL)
        delete[] lexbuf;
}

// ----------------------------------------------------------------------------

int  parser::analyser(char* pStr, double* pRTNValue)
{
    parseErrorCode = PAR_ERR_SUCCESS;

    if ((pExpStr = pStr) == NULL)
        return 0;

    *pRTNValue = 0.0;
    expStrIdx  = 0;
    expStrLen  = strlen(pStr);
    _strlwr(pStr);

    // Lexical scan input
    lookahead = lexan();

    if (parseErrorCode != 0)
        return -1;

    // Process lexical scan...
    while(lookahead != PAR_DONE)
    {
        if (expr(pRTNValue) != 0)
        {
            if (parseErrorCode == 0)
                parseErrorCode = PAR_ERR_UNKIDNT;

            return -1;
        }
    }
       
    return 0;
}

// ----------------------------------------------------------------------------

int parser::expr(double* pExprRst)
{
    double expr_number; 
    int    errorCode;
    
    if ((errorCode = term (&expr_number)) != 0)
        return errorCode;
    
    for(;;)
    {
        switch(lookahead)
        {
            case '+':
                {
                    double plusNum;

                    if (match(lookahead) != 0)
                        return -1;

                    if (term (&plusNum) != 0)
                        return -1;

                    expr_number = expr_number + plusNum;
                }
                break;

            case '-':                
                {
                    double minusNum;

                    if (match(lookahead) != 0)
                        return -1;

                    if (term(&minusNum) != 0)
                        return -1;

                    expr_number = expr_number - minusNum;
                }
                break;

            default:
                *pExprRst = expr_number;
                return 0;
        }
    }

    *pExprRst = expr_number;
    return 0;
}

// ----------------------------------------------------------------------------

int parser::term(double* pTermRst)
{

    double term_number;    

    if (power(&term_number) != 0)    
        return -1;    

    while(1)
    {
        switch(lookahead)
        {
            case '*':
                {
                    double pwrNum;

                    if (match(lookahead) != 0)
                        return -1;

                    if (power(&pwrNum) != 0)
                        return -1;

                    term_number = term_number * pwrNum;
                }
                break;

            case '/':
                {
                    double pwrNum;

                    if (match(lookahead) != 0)
                        return -1;

                    if (power(&pwrNum) != 0)
                        return -1;

                    term_number = term_number / pwrNum;
                }
                break;

            continue;
            default:
                *pTermRst = term_number;
                return 0;                
        }
    }

    *pTermRst = term_number;
    return 0;                
}

// ----------------------------------------------------------------------------

int parser::power(double* pPwrRst)
{
    double power_number;
    
    if (factor(&power_number) != 0)
        return -1;    

    if(lookahead == PAR_PW)
    {
        double pwrOfNum;

        if (match(lookahead) != 0)
            return -1;

        if (factor(&pwrOfNum) != 0)
            return -1;

        power_number = pow (power_number, pwrOfNum);
    }

    *pPwrRst = power_number;
    return 0;
}

// ----------------------------------------------------------------------------

int parser::factor(double* pFactRst)
{
    double factor_number;

    switch(lookahead)
    {
        case '(':
            {
                double paramNum;

                if (match ('(') != 0)
                    return -1;

                if (expr(&paramNum) != 0)
                    return -1;

                factor_number = paramNum;

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_NUM:
            factor_number = fnumber;

            if (match (PAR_NUM) != 0)
                return -1;
            break;

        case '-':
            {
                double minusNum;

                if (match ('-') != 0)
                    return -1;

                if (factor(&minusNum) != 0)
                    return -1;

                factor_number = -minusNum;
            }
            break;

        case PAR_TN:
            {
                double tanNum;

                if (match (PAR_TN) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&tanNum) != 0)
                    return -1;

                factor_number = tan(tanNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_SN:
            {
                double sinNum;

                if (match (PAR_SN) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&sinNum) != 0)
                    return -1;

                factor_number = sin(sinNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_CS:
            {
                double cosNum;

                if (match (PAR_CS) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&cosNum) != 0)
                    return -1;

                factor_number = cos(cosNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_ATN:
            {
                double tanNum;

                if (match (PAR_ATN) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&tanNum) != 0)
                    return -1;

                factor_number = atan(tanNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_ASN:
            {
                double sinNum;

                if (match (PAR_ASN) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&sinNum) != 0)
                    return -1;

                factor_number = asin(sinNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_ACS:
            {
                double cosNum;

                if (match (PAR_ACS) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&cosNum) != 0)
                    return -1;

                factor_number = acos(cosNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_R2D:
            {
                double r2dNum;

                if (match (PAR_R2D) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&r2dNum) != 0)
                    return -1;

                factor_number = r2dNum / (180.0 / 3.141592654);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_QR:
            {
                double sqrtNum;

                if (match (PAR_QR) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&sqrtNum) != 0)
                    return -1;

                factor_number = sqrt(sqrtNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_EX:
            {
                double expNum;

                if (match (PAR_EX) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&expNum) != 0)
                    return -1;

                factor_number = exp(expNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_LN:
            {
                double logNum;

                if (match (PAR_LN) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&logNum) != 0)
                    return -1;

                factor_number = log(logNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_MOD:
            {
                double leftNum, rightNum;

                if (match (PAR_MOD) != 0)
                    return -1;

                if (match ('(') != 0)
                    return -1;

                if (expr(&leftNum) != 0)
                    return -1;

                if (match (',') != 0)
                    return -1;

                if (expr(&rightNum) != 0)
                    return -1;

                factor_number = fmod(leftNum, rightNum);

                if (match (')') != 0)
                    return -1;
            }
            break;

        case PAR_PI:
            if (match (PAR_PI) != 0)
                return -1;

            factor_number = 3.141592654;
            break;

        default:
            parseErrorCode = PAR_ERR_SYNFACT;  
            return -1;                      
    }

    *pFactRst = factor_number;
    return 0;
}

// ----------------------------------------------------------------------------

int parser::match (int t)
{
    if (lookahead == t)
    {
        lookahead = lexan();        
        return 0;
    }
        
    return -1;    
}

// ----------------------------------------------------------------------------

int parser::lexan (void)
{
    int itemData;

    for (;;)
    {
        itemData = pExpStr[expStrIdx];
        expStrIdx++;

        if ((itemData == PAR_EOS) || (expStrIdx >= expStrLen))
            return PAR_DONE;
        else if((itemData ==' ')  || 
                (itemData =='\t') || 
                (itemData =='\r'))        
            continue;        
        else if(itemData=='\n')
            lineno++;
        else if(isdigit(itemData))
        {
            int idx = 0;

            exp_flag = PAR_FALSE;

            while(is_float(itemData))
            {
                // Out of token number space!
                if (idx >= PAR_NSIZE-1)
                {                  
                    parseErrorCode = PAR_ERR_COMPILE;                                      
                    return PAR_DONE;
                }

                numbuf[idx] = (char)itemData;
                idx++;

                // Read next char from stream...
                itemData = pExpStr[expStrIdx];
                expStrIdx++;
                if (expStrIdx > expStrLen)
                {
                    parseErrorCode = PAR_ERR_INCPEXP;
                    return PAR_DONE;
                }
            }

            // Put back data that was not part of float.
            if (itemData != PAR_EOS)
                expStrIdx--;                

            // terminate string and try and convert float value            
            numbuf[idx]= PAR_EOS;
            if ((fnumber = atof(numbuf)) < 0)
                return PAR_DONE;
            
            return PAR_NUM; // return token type.
        }
        else if (isalpha(itemData))
        {            
            int idx = 0;

            while(isalnum(itemData))
            {
                // Out of token number space!
                if (idx >= PAR_BSIZE-1)
                {
                    parseErrorCode = PAR_ERR_COMPILE;                                      
                    return PAR_DONE;
                }

                lexbuf[idx] = (char)itemData;
                idx++;

                // Read next char from stream...                
                itemData = pExpStr[expStrIdx];
                expStrIdx++;
                if (expStrIdx > expStrLen)
                {
                    parseErrorCode = PAR_ERR_INCPEXP; 
                    return PAR_DONE;
                }
            }

            // Terminate the string...
            lexbuf[idx] = PAR_EOS;

            if (itemData != PAR_EOS)
                expStrIdx--;

            int ident = lookup(lexbuf);

            if (ident <= -1)
            {
                parseErrorCode = PAR_ERR_INCPEXP; 
                return PAR_DONE;
            }

            return symtable[ident].token;
        }
        else if (itemData == '*')
        {
            int ch;
            
            ch = pExpStr[expStrIdx];
            expStrIdx++;
            if (expStrIdx > expStrLen)
            {
                parseErrorCode = PAR_ERR_INCPEXP; 
                return PAR_DONE;
            }

            if (ch=='*')
                return PAR_PW;
            else
            {
                expStrIdx--;
                return itemData;
            }
        }
        else 
        {
            switch (itemData)
            {
                case '(':
                case ')':
                case '+':
                case '-':
                case '/':
                case '*':
                case ',':
                    return itemData;                    

                default:
                    parseErrorCode = PAR_ERR_UNKIDNT;                
                    return PAR_DONE;
            }
        }
    }
}

// ----------------------------------------------------------------------------

int parser::is_float(int itemData)
{
    int itemDataRD;

    if (isdigit(itemData) != 0)
    {
        exp_flag = PAR_FALSE;
        return 1;
    }

    switch (itemData)
    {
        case('e'):
        case('E'):
            exp_flag = PAR_TRUE;

            itemDataRD = pExpStr[expStrIdx];
            expStrIdx++;
            if (expStrIdx > expStrLen)
            {
                parseErrorCode = PAR_ERR_INCPEXP;
                return -1;
            }

            if ((isdigit(itemDataRD) != 0)   || 
                (itemDataRD          == '+') || 
                (itemDataRD          == '-'))
            {
                expStrIdx--;
                return 1;
            }

            parseErrorCode = PAR_ERR_FLTERR;
            return -1;

        case('.'):
            exp_flag = PAR_FALSE;
            return 1;

        case('-'):
        case('+'):
            if(exp_flag == PAR_FALSE)
                return 0;
            else
                return 1;
    }

    return 0;
}

// ----------------------------------------------------------------------------

int parser::lookup(char* pTokStr)
{  
    for (int idx = lastentry; idx >= 0; idx--)
    {
        if (strcmp(symtable[idx].lexptr, pTokStr) == 0)
                return idx;
    }

    return -1;
}