{$INCLUDE ..\cDefines.inc}
unit cRegistry;

interface

uses
  // Delphi
  Windows,

  // Fundamentals
  cUtils;



{                                                                              }
{                      Windows Registry functions v3.01                        }
{                                                                              }
{         This unit is copyright  2002 by David Butler (david@e.co.za)        }
{                                                                              }
{                  This unit is part of Delphi Fundamentals.                   }
{                    Its original file name is cWindows.pas                    }
{       The latest version is available from the Fundamentals home page        }
{                     http://fundementals.sourceforge.net/                     }
{                                                                              }
{                I invite you to use this unit, free of charge.                }
{        I invite you to distibute this unit, but it must be for free.         }
{             I also invite you to contribute to its development,              }
{             but do not distribute a modified copy of this file.              }
{                                                                              }
{          A forum is available on SourceForge for general discussion          }
{             http://sourceforge.net/forum/forum.php?forum_id=2117             }
{                                                                              }
{ Description:                                                                 }
{   Windows Registry functions.                                                }
{                                                                              }
{ Revision history:                                                            }
{   2002/09/22  v3.01  Created cRegistry unit from cWindows.                   }
{                                                                              }



{                                                                              }
{ Registry                                                                     }
{                                                                              }
Procedure SplitRegName (const Name : String; var Key, ValueName : String);

Function  RegKeyExists (const RootKey : HKEY; const Key : String) : Boolean;
Function  RegValueExists (const RootKey : HKEY; const Key, Name : String) : Boolean;

Function  RegSetValue (const RootKey : HKEY; const Key, Name : String;
          const ValueType : Cardinal; const Value : Pointer;
          const ValueSize : Integer) : Boolean; overload;
Function  RegSetValue (const RootKey : HKEY; const Name : String;
          const ValueType : Cardinal; const Value : Pointer;
          const ValueSize : Integer) : Boolean; overload;

Function  SetRegistryString (const RootKey : HKEY; const Key : String;
          const Name : String; const Value : String) : Boolean; overload;
Function  SetRegistryString (const RootKey : HKEY; const Name : String;
          const Value : String) : Boolean; overload;

Function  SetRegistryDWord (const RootKey : HKEY; const Name : String;
          const Value : LongWord) : Boolean;

Function  SetRegistryBinary (const RootKey : HKEY; const Name: String;
          const Value; const ValueSize : Integer) : Boolean;

Function  RegGetValue (const RootKey : HKEY; const Key, Name : String;
          const ValueType : Cardinal; var RegValueType : Cardinal;
          var ValueBuf : Pointer; var ValueSize : Integer) : Boolean; overload;
Function  RegGetValue (const RootKey : HKEY; const Name : String;
          const ValueType : Cardinal; var RegValueType : Cardinal;
          var ValueBuf : Pointer; var ValueSize : Integer) : Boolean; overload;

Function  GetRegistryString (const RootKey : HKEY; const Key, Name : String) : String; overload;
Function  GetRegistryString (const RootKey : HKEY; const Name : String) : String; overload;

Function  GetRegistryDWord (const RootKey : HKEY; const Key, Name : String) : LongWord;

Function  DeleteRegistryValue (const RootKey : HKEY; const Key, Name : String) : Boolean;
Function  DeleteRegistryKey (const RootKey : HKEY; const Key : String) : Boolean;

Function  ConnectRegistry (const MachineName : String; const RootKey : HKEY;
          var RemoteKey : HKEY) : Boolean;
Function  DisconnectRegistry (const RemoteKey : HKEY) : Boolean;

Function  EnumRegistryValues (const RootKey : HKEY; const Name : String;
          var ValueList : StringArray) : Boolean;
Function  EnumRegistryKeys (const RootKey : HKEY; const Name : String;
          var KeyList : StringArray) : Boolean;



implementation

uses
  // Delphi
  SysUtils,

  // Fundamentals
  cStrings;



{                                                                              }
{ Registry                                                                     }
{                                                                              }
Procedure SplitRegName (const Name : String; var Key, ValueName : String);
var S : String;
    I : Integer;
  Begin
    S := WithoutSuffix (WithoutPrefix (Name, '\'), '\');
    I := Pos ('\', S, [foReverse]);
    if I <= 0 then
      begin
        Key := S;
        ValueName := '';
        exit;
      end;
    Key := CopyLeft (S, I - 1);
    ValueName := CopyFrom (S, I + 1);
  End;

{ Exists                                                                       }
Function RegKeyExists (const RootKey : HKEY; const Key : String) : Boolean;
var Handle : HKEY;
  Begin
    if RegOpenKeyEx (RootKey, PChar (Key), 0, KEY_READ, Handle) = ERROR_SUCCESS then
      begin
        Result := True;
        RegCloseKey (Handle);
      end else
      Result := False;
  End;

Function RegValueExists (const RootKey : HKEY; const Key, Name : String) : Boolean;
var Handle : HKEY;
  Begin
    if RegOpenKeyEx (RootKey, PChar (Key), 0, KEY_READ, Handle) = ERROR_SUCCESS then
      begin
        Result := RegQueryValueEx (Handle, Pointer (Name), nil, nil, nil, nil) = ERROR_SUCCESS;
        RegCloseKey (Handle);
      end else
      Result := False;
  End;

{ Set                                                                          }
Function RegSetValue (const RootKey : HKEY; const Key, Name : String;
         const ValueType : Cardinal; const Value : Pointer;
         const ValueSize : Integer) : Boolean;
var D : DWORD;
    Handle : HKEY;
  Begin
    Result := False;
    if ValueSize < 0 then
      exit;
    if RegCreateKeyEx (RootKey, PChar (Key), 0, nil, REG_OPTION_NON_VOLATILE,
        KEY_WRITE, nil, Handle, @D) <> ERROR_SUCCESS then
      exit;
    Result := RegSetValueEx (Handle, Pointer (Name), 0, ValueType, Value, ValueSize) = ERROR_SUCCESS;
    RegCloseKey (Handle);
  End;

Function RegSetValue (const RootKey : HKEY; const Name : String;
         const ValueType : Cardinal; const Value : Pointer;
         const ValueSize : Integer) : Boolean;
var K, N : String;
  Begin
    SplitRegName (Name, K, N);
    Result := RegSetValue (RootKey, K, N, ValueType, Value, ValueSize);
  End;

Function SetRegistryString (const RootKey : HKEY; const Key : String; const Name : String; const Value : String) : Boolean;
  Begin
    Result := RegSetValue (RootKey, Key, Name, REG_SZ, PChar (Value), Length (Value) + 1);
  End;

Function SetRegistryString (const RootKey : HKEY; const Name : String; const Value : String) : Boolean;
  Begin
    Result := RegSetValue (RootKey, Name, REG_SZ, PChar (Value), Length (Value) + 1);
  End;

Function SetRegistryDWord (const RootKey : HKEY; const Name : String; const Value : LongWord) : Boolean;
  Begin
    Result := RegSetValue (RootKey, Name, REG_DWORD, @Value, Sizeof (LongWord));
  End;

Function SetRegistryBinary (const RootKey : HKEY; const Name: String; const Value; const ValueSize : Integer) : Boolean;
  Begin
    Result := RegSetValue (RootKey, Name, REG_BINARY, @Value, ValueSize);
  End;

{ Get                                                                          }
Function RegGetValue (const RootKey : HKEY; const Key, Name : String;
         const ValueType : Cardinal; var RegValueType : Cardinal;
         var ValueBuf : Pointer; var ValueSize : Integer) : Boolean;
var Handle  : HKEY;
    Buf     : Pointer;
    BufSize : Cardinal;
  Begin
    Result := False;
    ValueSize := 0;
    ValueBuf := nil;
    if RegOpenKeyEx (RootKey, PChar (Key), 0, KEY_READ, Handle) <> ERROR_SUCCESS then
      exit;
    BufSize := 0;
    RegQueryValueEx (Handle, Pointer (Name), nil, @RegValueType, nil, @BufSize);
    if BufSize <= 0 then
      exit;
    GetMem (Buf, BufSize);
    if RegQueryValueEx (Handle, Pointer (Name), nil, @RegValueType, Buf, @BufSize) = ERROR_SUCCESS then
      begin
        ValueBuf := Buf;
        ValueSize := Integer (BufSize);
        Result := True;
      end;
    if not Result then
      FreeMem (Buf);
    RegCloseKey (Handle);
  End;

Function RegGetValue (const RootKey : HKEY; const Name : String;
         const ValueType : Cardinal; var RegValueType : Cardinal;
         var ValueBuf : Pointer; var ValueSize : Integer) : Boolean;
var K, N : String;
  Begin
    SplitRegName (Name, K, N);
    Result := RegGetValue (RootKey, K, N, ValueType, RegValueType, ValueBuf, ValueSize);
  End;

Function GetRegistryString (const RootKey : HKEY; const Key, Name : String) : String;
var Buf   : Pointer;
    Size  : Integer;
    VType : Cardinal;
  Begin
    Result := '';
    if not RegGetValue (RootKey, Key, Name, REG_SZ, VType, Buf, Size) then
      exit;
    if (VType = REG_DWORD) and (Size >= Sizeof (LongWord)) then
      Result := IntToStr (PLongWord (Buf)^) else
      begin
        SetLength (Result, Size);
        MoveMem (Buf^, Pointer (Result)^, Size);
      end;
    FreeMem (Buf);
  End;

Function GetRegistryString (const RootKey : HKEY; const Name : String) : String;
var K, N : String;
  Begin
    SplitRegName (Name, K, N);
    Result := GetRegistryString (RootKey, K, N);
  End;

Function GetRegistryDWord (const RootKey : HKEY; const Key, Name : String) : LongWord;
var Buf   : Pointer;
    Size  : Integer;
    VType : Cardinal;
  Begin
    Result := 0;
    if not RegGetValue (RootKey, Key, Name, REG_DWORD, VType, Buf, Size) then
      exit;
    if (VType = REG_DWORD) and (Size >= Sizeof (LongWord)) then
      Result := PLongWord (Buf)^;
    FreeMem (Buf);
  End;

{ Delete                                                                       }
Function DeleteRegistryValue (const RootKey : HKEY; const Key, Name : String) : Boolean;
var Handle : HKEY;
  Begin
    if RegOpenKeyEx (RootKey, PChar (Key), 0, KEY_WRITE, Handle) = ERROR_SUCCESS then
      begin
        Result := RegDeleteValue (Handle, Pointer (Name)) = ERROR_SUCCESS;
        RegCloseKey (Handle);
      end else
      Result := False;
  End;

Function DeleteRegistryKey (const RootKey : HKEY; const Key : String) : Boolean;
var Handle : HKEY;
    K, N   : String;
  Begin
    SplitRegName (Key, K, N);
    if RegOpenKeyEx (RootKey, PChar (K), 0, KEY_WRITE, Handle) = ERROR_SUCCESS then
      begin
        Result := RegDeleteKey (Handle, Pointer (N)) = ERROR_SUCCESS;
        RegCloseKey (Handle);
      end else
      Result := False;
  End;

{ Remote Registries                                                            }
Function ConnectRegistry (const MachineName : String; const RootKey : HKEY;
         var RemoteKey : HKEY) : Boolean;
  Begin
    Result := RegConnectRegistry (PChar (MachineName), RootKey, RemoteKey) = ERROR_SUCCESS;
  End;

Function DisconnectRegistry (const RemoteKey : HKEY) : Boolean;
  Begin
    Result := RegCloseKey (RemoteKey) = ERROR_SUCCESS;
  End;

{ Enumerate                                                                    }
Function RegEnum (const RootKey : HKEY; const Name : String;
         var ResultList : StringArray; const DoKeys : Boolean) : Boolean;
var Buf     : Array [0..2047] of Char;
    BufSize : Cardinal;
    I       : Integer;
    Res     : Integer;
    S       : String;
    Handle  : HKEY;
  Begin
    ResultList := nil;
    Result := RegOpenKeyEx (RootKey, PChar(Name), 0, KEY_READ, Handle) = ERROR_SUCCESS;
    if not Result then
      exit;
    I := 0;
    Repeat
      BufSize := Sizeof (Buf);
      if DoKeys then
        Res := RegEnumKeyEx (Handle, I, @Buf [0], BufSize, nil, nil, nil, nil)
      else
        Res := RegEnumValue (Handle, I, @Buf [0], BufSize, nil, nil, nil, nil);
      if Res = ERROR_SUCCESS then
        begin
          SetLength (S, BufSize);
          if BufSize > 0 then
            MoveMem (Buf [0], Pointer (S)^, BufSize);
          Append (ResultList, S);
          Inc (I);
        end;
    Until Res <> ERROR_SUCCESS;
    RegCloseKey (Handle);
  End;

Function EnumRegistryValues (const RootKey : HKEY; const Name : String; var ValueList : StringArray) : Boolean;
  Begin
    Result := RegEnum (RootKey, Name, ValueList, False);
  End;

Function EnumRegistryKeys (const RootKey : HKEY; const Name : String; var KeyList : StringArray) : Boolean;
  Begin
    Result := RegEnum (RootKey, Name, KeyList, True);
  End;




end.

