/* ------------------------------------------------------------------------ */
/* System       : Library Routines                                          */
/* Program      : bmpmcg.c                                                  */
/* Description  : .BMP load routine for MCGA video mode                     */
/*                (C) CopyLeft Bill Buckels 1997.                           */
/*                All Rights Reversed.                                      */
/*                                                                          */
/* You have a royalty-free right to use, modify, reproduce and              */
/* distribute this source code in any way you find useful,                  */
/* provided that you agree that Bill Buckels has no warranty obligations    */
/* or liability resulting from said distribution in any way whatsoever.     */
/* ------------------------------------------------------------------------ */


#include "std.h"
#include "pixhdr.h"
#include "vgapal.h"
#include "stdkeys.h"

/* -------------------------------------------------------------------- */
/* Function BMPMCGA                                                     */
/* Loads a 256 Color Windows BMP to a MAXIMUM display size of 320 x 200 */
/* I have purposely kept this very simple basically to demonstrate      */
/* how we read the .BMP header and use that info to load the image.     */
/*                                                                      */
/* args are pretty much self explanatory...                             */
/*                                                                      */
/* -------------------------------------------------------------------- */
int bmpmcga(char *filename, int xorg, int yorg, int abort, int setpalette)
{
    FILE *fp;
    unsigned idx, x, y, y1, offset, packet, height, width; /* scanlines, etc. */
    int status;
    unsigned long hres, vres, seeksize;
    BITMAPFILEHEADER BitMapFileHeader;
    BITMAPINFO bmp;
    unsigned char buffer[MAX_SCANLINE_LEN];
    char *vgaptr = (char *)0xa0000000L;


    /* 1.1 open the file */
    if (NULL == (fp=fopen(filename, "rb")))
      return erreturn(NULL, "Can't open file %s",
                        filename, INVALID_HANDLE, abort);

    /* 1.2 read the header stuff into the appropriate structures */
    fread((char *)&BitMapFileHeader.bfType[0],
               sizeof(BITMAPFILEHEADER),1,fp);
    fread((char *)&bmp.bmiHeader.biSize,
                 sizeof(BITMAPINFO),1,fp);


    /* 1.3 validate the header information */
    if (SUCCESS != (status =
        checkbmpheader((BITMAPINFO *)&bmp,
                       (BITMAPFILEHEADER *)&BitMapFileHeader,
                       sizeof(buffer), MCGA_VIDEO))) {
      fclose(fp);
      return erreturn(NULL, "%s is not a supported 256 color bmp file",
                        filename, status, abort);
    }

    /* 2. set/reset the palette if requested to do so */
    if (setpalette) {
      /* 2.1 clear the previous palette */
      clearpalette();
      /* 2.2 build the palette */
      for (idx = 0; idx <MCGA_PALETTE_ENTRIES; idx++)
        setpalettereg(bmp.bmiColors[idx].rgbRed,
                      bmp.bmiColors[idx].rgbGreen,
                      bmp.bmiColors[idx].rgbBlue, idx);
      /* 2.3 reset the DAC with the new palette */
      loadpalette(0, MCGA_PALETTE_ENTRIES);
    }


    /* 3. Use the image size values to set the load origin and      */
    /*    relevant size information...                              */
    /*  - this function allows for load origin coordinates, but     */
    /*    if these aren't in range we will still centre the image   */
    /*  - we also allow for clipping regions to be used, i.e.       */
    /*    within the bounds of the lower-left of the viewport...    */
    /*    nothing fancy though.                                     */

    /* the following are simple arithmetic equasions...             */
    /* it really serves no purpose to comment these much since the  */
    /* equasions are self-explanatory so I have left these sparsely */
    /* commented to preserve readability.                           */

    /* Y-axis calculations */
    vres = bmp.bmiHeader.biHeight;

    height = (unsigned) vres;
    while (height > MCGA_SCAN_LINES) /* make sure we are in the viewport */
      height--;

    if (yorg > DEFAULT && yorg < MCGA_SCAN_LINES) {    /* in range ???   */
      y1 = yorg;                                       /* allow override */
      while ((height + y1) > MCGA_SCAN_LINES)  /* adjust clipping region */
        height--;
    }
    else
      y1 = (MCGA_SCAN_LINES - height) / 2;  /* centre on viewport */



    /* X-axis calculations (basically same rules as Y-axis) */
    hres = bmp.bmiHeader.biWidth;

    width = packet = (unsigned )hres;
    while (width > MCGA_LINE_SIZE) /* adjust clipping region */
      width--;

    if (xorg > DEFAULT && xorg < MCGA_LINE_SIZE) {    /* in range ??? */
      x = xorg;
      while ((width + x) > MCGA_LINE_SIZE)
        width--;
    }
    else
      x = (MCGA_LINE_SIZE - width) / 2;

    /* adjust the file packet...                               */
    /* align scanlines on 32 bit boundaries... please note...  */
    /* this is an important feature of the windows BMP format  */
    while (packet % 4)
      packet++;
    hres = (long)packet;

    /* 4. seek to the first scanline in the file */
    /*    and put the bitmap on the screen       */
    seeksize =  0L;
    seeksize += vres;
    seeksize *= hres;
    seeksize += BitMapFileHeader.bfOffBits;

    /* compute offset into screen memory */
    offset = (y1 * MCGA_LINE_SIZE) + x;

    /* pretty much a standard read... */
    /* these are stored upside down though, so we need to seek */
    /* between scanlines...                                    */
    for(y = 0; y < vres; y++)
    {
        if (y < height && 0 < width) {
          seeksize -= packet;
          fseek(fp, seeksize, SEEK_SET);
          fread(buffer, packet, 1, fp);
          memcpy(&vgaptr[offset], &buffer[0], width);
          offset += MCGA_LINE_SIZE;
        }
        else
          break;
    }
    fclose(fp);
    return SUCCESS;
}
