//==============================================================================
// BLIST.CPP -- Copyright  2000 Adam King
//		Basic Dynamic List
//==============================================================================
#define STRICT
#include <windows.h>
#include <windowsx.h>
#pragma hdrstop
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <excpt.h>

#include "blist.h"
#include "wutils.h"


BASICLIST::BASICLIST()
{
	memset(this, 0, sizeof(BASICLIST));
}

BASICLIST::~BASICLIST()
{
	Clear();
}

void BASICLIST::Clear()
{
	while(Pop() != NULL);
	ItemCount = 0;
}

LISTERR BASICLIST::Add(char * szStr, BOOL keepDupes, BOOL caseSens)
{
	BLIST_FOUNDITEM	pExisting;
	BLIST_ITEM *		ptr;

	if(szStr == NULL)
		return ERR_EMPTY;

	// If requested, don't allow duplicates...
	if(keepDupes == FALSE)
	{
		pExisting = InList(szStr, caseSens);
		if(pExisting.Res == BLIST_FR_FOUND)
			return ERR_EXISTS;
	}

	// Allocate memory for BLIST_ITEM structure
	ptr = new BLIST_ITEM;
	if(!ptr)
	{
		MessageBox(NULL, "Cannot create List Item (1)", "Add", MB_OK);
		return ERR_CREATE;
	}

	memset(ptr, 0, sizeof(BLIST_ITEM));

	// Allocate enough memory for <szStr>
	unsigned int len = strlen(szStr) + 1;
	try {
		ptr->szItem = new char[len];
	}
	catch(...)
	{
		// Error Clean up
		delete [] ptr;
		MessageBox(NULL, "Cannot create List Item (2)", "Add", MB_OK);
		return ERR_CREATE;
	}

	// Save this item
	strcpy(ptr->szItem, szStr);
	ptr->NextItem = BList;

	BList = ptr;

	ItemCount++;

	return ERR_NONE;
}

#pragma argsused
BLIST_ITEM * BASICLIST::Pop()
{
	BLIST_ITEM *	ptr;

	if(NULL != BList)
	{
		ptr = BList;
		if(ptr->szItem)
		{
			delete [] ptr->szItem;
			ptr->szItem = NULL;
		}
		BList = BList->NextItem;
		delete ptr;

		ItemCount--;
	}
	return BList;
}

BLIST_ITEM * BASICLIST::RemoveLastItem()
{
	BLIST_ITEM *	ptr;
	BLIST_ITEM *	nextItem;

	ptr = BList;
	while(ptr != NULL)
	{
		// Get pointer to next list entry
		nextItem = ptr->NextItem;
		if(nextItem != NULL)
		{
			// Is the next item the last item ?
			if(nextItem->NextItem == NULL)
			{
				// It is, so remove it
				if(nextItem->szItem)
				{
					delete [] nextItem->szItem;
					nextItem->szItem = NULL;
				}
				ptr->NextItem = NULL;
				delete nextItem;

				ItemCount--;
			}
		}
		ptr = ptr->NextItem;
	}
	return BList;
}

BOOL BASICLIST::RemoveString(char * str)
{
	BLIST_ITEM *	ptr;
	BLIST_ITEM *	nextItem;

	ptr = BList;
	while(ptr != NULL)
	{
		// Get pointer to next list entry
		nextItem = ptr->NextItem;
		if(nextItem != NULL)
		{
			// Is the next item the item to delete ?
			if(lstrcmp(nextItem->szItem, str) == 0)
			{
				// It is, so remove it
				if(nextItem->szItem)
				{
					delete [] nextItem->szItem;
					nextItem->szItem = NULL;
				}
				ptr->NextItem = nextItem->NextItem;
				delete nextItem;

				ItemCount--;
				break;
			}
		}
		ptr = ptr->NextItem;
	}
	return TRUE;
}

BOOL BASICLIST::FindStr(char * szStr, BOOL caseSens)
{
	BLIST_ITEM *	ptr;
	BOOL		found = FALSE;
	int		res;

	if(BList == NULL)
		return FALSE;

	ptr = BList;
	while(!found && ptr != NULL)
	{
		// Check the string part...
		if(ptr->szItem == NULL && szStr == NULL)
		{
			// They're both NULL so assume a find
			found = TRUE;
		} else if(ptr->szItem[0] == '\0') {
			if(szStr[0] == '\0')
			{
				// Both strings are blank so again, assume a find
				found = TRUE;
			}
		} else {
			if(ptr->szItem != NULL && szStr != NULL)
			{
				if(ptr->szItem[0] == szStr[0])
				{
					// First character is the same, so check the full strings
					res = (caseSens) ? strcmp(ptr->szItem, szStr) : stricmp(ptr->szItem, szStr);
					if(res == 0)
					{
						// Strings are the same
						found = TRUE;
					}
				}
			}
		}

		ptr = ptr->NextItem;
	}
	return found;
}

BLIST_FOUNDITEM BASICLIST::InList(char * szStr, BOOL caseSens)
{
	BLIST_ITEM *		ptr;
	BLIST_FOUNDITEM	item;
	int			res;

	item.Res = BLIST_FR_NONE;
	item.Item = NULL;

	if(BList != NULL && szStr[0] != '\0')
	{
		ptr = BList;
		while(item.Res == BLIST_FR_NONE && ptr != NULL)
		{
			if(NULL != ptr->szItem)
			{
				if(ptr->szItem[0] != '\0')
				{
					// Try whole words only
					if(ptr->szItem[0] == szStr[0] && ptr->szItem[1] == szStr[1])
					{
						res = (caseSens) ? strcmp(ptr->szItem, szStr) : stricmp(ptr->szItem, szStr);
						if(res == 0)
						{
							item.Res = BLIST_FR_FOUND;
							item.Item = ptr;
						}
					}
				}
			}

			ptr = ptr->NextItem;
		}
	}
	return item;
}

char * BASICLIST::GetNthStr(long n)
{
	BLIST_ITEM *	ptr;
	BOOL	found = FALSE;
	long	cnt = 0;

	ptr = BList;
	while(ptr != NULL)
	{
		if(cnt == n)
		{
			found = TRUE;
			break;
		}
		ptr = ptr->NextItem;
		cnt++;
	}
	return (found) ? ptr->szItem : "";
}

int BASICLIST::Count()
{
	BLIST_ITEM *	ptr;
	int		num = 0;

	if(BList)
	{
		ptr = BList;
		while(ptr != NULL)
		{
			num++;
			ptr = ptr->NextItem;
		}
	}
	return(num);
}

BLIST_ITEM * BASICLIST::GetStrItem(char * str)
{
	BLIST_ITEM *	ptr;

	ptr = BList;
	while(ptr != NULL)
	{
		if(stricmp(ptr->szItem, str) == 0)
			return ptr;
		ptr = ptr->NextItem;
	}
	return NULL;
}

int BASICLIST::GetStrPos(char * str)
{
	BLIST_ITEM *	ptr;
	int		n = 0;

	ptr = BList;
	while(ptr != NULL)
	{
		if(stricmp(ptr->szItem, str) == 0)
			return n;
		n++;
		ptr = ptr->NextItem;
	}
	return NULL;
}

void BASICLIST::FillList(HWND hList, BOOL reset, long data)
{
	LRESULT	ret;
	BLIST_ITEM *	ptr;

	if(reset)
	{
		SendMessage(hList, LB_RESETCONTENT, 0, 0);
		SendMessage(hList, LB_SETHORIZONTALEXTENT, 0, 0L);
	}
	ptr = BList;
	while(ptr != NULL)
	{
		// Store the item data
		ret = LBAddString(hList, ptr->szItem);
		if(ret == CB_ERR || ret == CB_ERRSPACE)
			break;

		if(ret != LB_ERR || ret != LB_ERRSPACE)
			SendMessage(hList, LB_SETITEMDATA, WPARAM(ret), data);

		ptr = ptr->NextItem;
	}
	SendMessage(hList, LB_SETCURSEL, 0, 0L);
}

void BASICLIST::FillCombo(HWND hCombo, BOOL reset)
{
	LRESULT	ret;
	BLIST_ITEM *	ptr;

	if(reset)
		SendMessage(hCombo, CB_RESETCONTENT, 0, 0);
	ptr = BList;
	while(ptr != NULL)
	{
		// Store the item data
		ret = SendMessage(hCombo, CB_ADDSTRING, 0, (LPARAM)((LPCSTR)ptr->szItem));
		if(ret == CB_ERR || ret == CB_ERRSPACE)
			break;
		ptr = ptr->NextItem;
	}
	SendMessage(hCombo, CB_SETCURSEL, 0, 0L);
}

void BASICLIST::FillCombos(HWND hCombo1, HWND hCombo2, BOOL reset)
{
	LRESULT	ret;
	BLIST_ITEM *	ptr;

	if(reset)
	{
		SendMessage(hCombo1, CB_RESETCONTENT, 0, 0);
		SendMessage(hCombo2, CB_RESETCONTENT, 0, 0);
	}
	ptr = BList;
	while(ptr != NULL)
	{
		// Store the item data
		ret = SendMessage(hCombo1, CB_ADDSTRING, 0, (LPARAM)((LPCSTR)ptr->szItem));
		if(ret == CB_ERR || ret == CB_ERRSPACE)
			break;
		if(hCombo2)
		{
			ret = SendMessage(hCombo2, CB_ADDSTRING, 0, (LPARAM)((LPCSTR)ptr->szItem));
			if(ret == CB_ERR || ret == CB_ERRSPACE)
				break;
		}
		ptr = ptr->NextItem;
	}
}

char * BASICLIST::GetRandomItemStr()
{
	BLIST_ITEM *	ptr;
	int		cnt;
	int		chosen;

	// Randomly choose an Item from the list...
	cnt = Count();
	if(!cnt)
		return NULL;

	chosen = random(cnt);

	ptr = BList;
	cnt = 0;
	while(ptr != NULL)
	{
		if(cnt == chosen)
			return ptr->szItem;
		cnt++;
		ptr = ptr->NextItem;
	}
	return NULL;
}

BLIST_ITEM * BASICLIST::GetRandomItem()
{
	BLIST_ITEM *	ptr;
	int		cnt;
	int		chosen;

	// Randomly choose an Item from the list...
	cnt = Count();
	if(!cnt)
		return NULL;

	chosen = random(cnt);

	ptr = BList;
	cnt = 0;
	while(ptr != NULL)
	{
		if(cnt == chosen)
			return ptr;
		ptr = ptr->NextItem;
		cnt++;
	}
	return NULL;
}

int BASICLIST::FillMenu(HMENU hMenu, UINT BaseID, char num)
{
	BLIST_ITEM *	ptr;
	int		count = Count();

	if(count)
	{
		if(num > count)
			num = count;

		ptr = BList;
		count = 0;
		while(ptr != NULL && count < num)
		{
			if(!AppendMenu(hMenu, MF_ENABLED, BaseID + count, ptr->szItem))
				break;
			count++;
			ptr = ptr->NextItem;
		}
	} else
		AppendMenu(hMenu, MF_STRING | MF_GRAYED, BaseID, "[Empty List]");

	return count;
}

BLIST_ITEM * BASICLIST::GetNthItem(int n)
{
	int		count = 0;
	BLIST_ITEM *	ptr = BList;

	// Loop through all Items in list
	while(ptr != NULL)
	{
		// Return current Item if correct index
		if(count == n)
			return ptr;

		// Point to next Item in list
		ptr = ptr->NextItem;
		count++;
	}
	// Return NULL if not found
	return ptr;
}

// Sort: Performs a ShellSort which is similar to the BubbleSort. However, it
// begins by comparing elements that are far apart (separated by the
// value of the iOffset variable, which is initially half the distance
// between the first and last element), then comparing elements that
// are closer together. When iOffset is one, the last iteration is
// merely a bubble sort.
//
void BASICLIST::SortAlpha(BOOL reversed)
{
	int 		iOffset, iSwitch, iLimit, iRow;
	BLIST_ITEM *	item1;
	BLIST_ITEM *	item2;
	char		ch1, ch2;

	 // Set comparison offset to half the number of data items
	 iOffset = ItemCount / 2;

	 while(iOffset)
	 {
		  // Loop until offset gets to zero.
		  iLimit = ItemCount - iOffset - 1 ;
		  do
		  {
				iSwitch = FALSE;     // Assume no switches at this offset.

				// Compare elements and switch ones out of order.
				for(iRow = 0; iRow <= iLimit; iRow++ )
				{
					 item1 = GetNthItem(iRow);
					 item2 = GetNthItem(iRow + iOffset);
					 if(item1->szItem && item2->szItem)
					 {
						 ch1 = toupper(item1->szItem[0]);
						 ch2 = toupper(item2->szItem[0]);
						 if(!reversed)
						 {
							  if(ch1 > ch2)
							  {
								  Swap(iRow, iRow + iOffset);
								  iSwitch = iRow;
							  }
						 } else if(ch1 < ch2) {
							  Swap(iRow, iRow + iOffset);
							  iSwitch = iRow;
						 }
					 }
				}

				// Sort on next pass only to where last switch was made.
				iLimit = iSwitch - iOffset;
		  } while( iSwitch );

		  // No switches at last offset, try one half as big.
		  iOffset = iOffset / 2;
	 }
}

void BASICLIST::Swap(int idx1, int idx2)
{
	BLIST_ITEM *	ptr;
	BLIST_ITEM *	itm1 = NULL;
	BLIST_ITEM *	itm2 = NULL;
	BLIST_ITEM 	tmp;
	int		cnt = 0;

	// Get item at <idx1>
	ptr = BList;
	while(ptr != NULL)
	{
		if(cnt == idx1)
		{
			itm1 = ptr;
			break;
		}
		ptr = ptr->NextItem;
		cnt++;
	}

	// Get item at <idx2>
	while(ptr != NULL)
	{
		if(cnt == idx2)
		{
			itm2 = ptr;
			break;
		}
		ptr = ptr->NextItem;
		cnt++;
	}

	// If both items are valid...
	if(itm1 && itm2)
	{
		// Swap the items
		tmp.szItem = itm1->szItem;

		itm1->szItem = itm2->szItem;

		itm2->szItem = tmp.szItem;
	}
}


