{************************************************}
{                                                }
{   Turbo Vision Demo                            }
{   Replaces the old Calendar unit               }
{   1996, Cosmin Truta <cosmint@cs.ubbcluj.ro>   }
{                                                }
{************************************************}

unit Calendar;

{$F+,O+,X+,R-,S-}

interface

uses Dos, Objects, Drivers, Views, App;

type

  PCalendarView = ^TCalendarView;
  TCalendarView = object(TView)
    CurYear, CurMonth, CurDay: Word;
    Year: Word;
    Delta: Integer;
    constructor Init(var Bounds: TRect);
    constructor Load(var S: TStream);
    function DayOfWeek(ADay, AMonth, AYear: Integer): Integer;
    procedure Draw; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure SetYear(AYear: Word); virtual;
    procedure Store(var S: TStream);
  end;

  PCalendarWindow = ^TCalendarWindow;
  TCalendarWindow = object(TWindow)
    constructor Init;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

const
  RCalendarView: TStreamRec = (
     ObjType: 10020;
     VmtLink: Ofs(TypeOf(TCalendarView)^);
     Load:    @TCalendarView.Load;
     Store:   @TCalendarView.Store
  );
  RCalendarWindow: TStreamRec = (
     ObjType: 10021;
     VmtLink: Ofs(TypeOf(TCalendarWindow)^);
     Load:    @TCalendarWindow.Load;
     Store:   @TCalendarWindow.Store
  );

procedure RegisterCalendar;

implementation

const

  DaysInMonth: array[1 .. 12] of Integer =
    (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

  DaysStr: array[0 .. 7] of string[2] =
    ('  ', 'Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa');

  MonthsStr: array[1 .. 12] of string[9] =
    (' January ',
     ' February',
     '  March  ',
     '  April  ',
     '   May   ',
     '   June  ',
     '   July  ',
     '  August ',
     'September',
     ' October ',
     ' November',
     ' December');

{ TCalendarView }
constructor TCalendarView.Init(var Bounds: TRect);
var
  W: Word;
begin
  inherited Init(Bounds);
  GrowMode := gfGrowHiX or gfGrowHiY;
  Options := Options or ofSelectable;
  GetDate(CurYear, CurMonth, CurDay, W);
  Year := CurYear;
  if (Size.Y < 24) and (CurMonth >= 9) then Delta := 24 - Size.Y
  else Delta := 0;
end;

constructor TCalendarView.Load(var S: TStream);
var
  W: Word;
begin
  inherited Load(S);
  GetDate(CurYear, CurMonth, CurDay, W);
  S.Read(Year, SizeOf(Year) + SizeOf(Delta));
end;

function TCalendarView.DayOfWeek(ADay, AMonth, AYear: Integer): Integer;
var
  century, yr, dw: Integer;
begin
  if AMonth < 3 then
  begin
    Inc(AMonth, 10);
    Dec(AYear);
  end
  else
    Dec(AMonth, 2);
  century := AYear div 100;
  yr := AYear mod 100;
  dw := (((26 * AMonth - 2) div 10) + ADay + yr + (yr div 4) +
    (century div 4) - (2 * century)) mod 7;
  if dw < 0 then DayOfWeek := dw + 7
  else DayOfWeek := dw;
end;

procedure TCalendarView.Draw;
const
  Width = 72;
var
  B: TDrawBuffer;
  Lines: array[0 .. 7] of string[Width];
  Y, Group, LinToday, ColToday: Integer;
  Color, NormalColor, BoldColor, SpecialColor: Byte;

procedure CalcLines(Group: Integer);
var
  Month, Day, B, Lin, Col, I: Integer;
  DayStr: string[2];
begin
  for I := 0 to 7 do
  begin
    FillChar(Lines[I][1], Width, ' ');
    Lines[I][0] := Chr(Width);
  end;
  LinToday := -1;
  if Group >= 3 then Exit;
  for I := 0 to 3 do
  begin
    Month := Group shl 2 + I + 1; B := 0;
    if (Month = 2) and (Year mod 4 = 0) then
      if (Year mod 100 <> 0) xor (Year mod 400 = 0) then B := 1;
    Lin := DayOfWeek(1, Month, Year) + 1;
    Col := 18 * I + 1;
    Move(MonthsStr[Month][1], Lines[0][Col + 3], 9);
    for Day := 1 to DaysInMonth[Month] + B do
    begin
      Str(Day : 2, DayStr);
      Move(DayStr[1], Lines[Lin][Col], 2);
      if (Year = CurYear) and (Month = CurMonth) and (Day = CurDay) then
      begin
        LinToday := Lin; ColToday := Col;
      end;
      Inc(Lin);
      if Lin > 7 then
      begin
        Lin := 1; Inc(Col, 3);
      end;
    end;
  end;
end;

begin  { TCalendarView.Draw }
  NormalColor := GetColor(6);
  BoldColor := GetColor(2);
  SpecialColor := GetColor(7);
  MoveChar(B, ' ', BoldColor, Size.X);
  Group := -1;
  for Y := 0 to Size.Y - 1 do
  begin
    if (Group < 0) or ((Y + Delta) and 7 = 0) then
    begin
      Group := (Y + Delta) shr 3;
      if Group < 3 then CalcLines(Group);
    end;
    if Group < 3 then
    begin
      MoveStr(B[1], DaysStr[(Y + Delta) mod 8], BoldColor);
      MoveStr(B[75], DaysStr[(Y + Delta) mod 8], BoldColor);
      if (Y + Delta) and 7 = 0 then Color := BoldColor
      else Color := NormalColor;
      MoveStr(B[3], Lines[(Y + Delta) and 7], Color);
      if (Y + Delta) and 7 = LinToday then
      begin
        B[ColToday + 2] := Lo(B[ColToday + 2]) + SpecialColor shl 8;
        B[ColToday + 3] := Lo(B[ColToday + 3]) + SpecialColor shl 8;
      end;
    end
    else MoveChar(B, ' ', BoldColor, Size.X);
    WriteLine(0, Y, Size.X, 1, B);
  end;
end;

procedure TCalendarView.HandleEvent(var Event: TEvent);
begin
  inherited HandleEvent(Event);
  if (State and sfSelected) <> 0 then
  begin
    if Event.What = evKeyDown then
    begin
      case Event.KeyCode of
        kbUp:
          if Delta > 0 then Dec(Delta) else Delta := 0;
        kbDown:
          if Delta + Size.Y < 24 then Inc(Delta);
        kbHome, kbPgUp:
          Delta := 0;
        kbEnd, kbPgDn:
          if Size.Y < 24 then Delta := 24 - Size.Y else Delta := 0;
        kbLeft:
          if Year > 1000 then SetYear(Year - 1);
        kbRight:
          if Year < 9999 then SetYear(Year + 1);
        else Exit;
      end;
      if (Event.KeyCode <> kbLeft) and (Event.KeyCode <> kbRight) then
        DrawView;
      ClearEvent(Event);
    end;
  end;
end;

procedure TCalendarView.SetYear(AYear: Word);
var
  Title: PString;
  Y: string[4];
begin
  Year := AYear;
  if Owner <> nil then
  begin
    Title := PWindow(Owner)^.Title;
    if Pos('Calendar ', Title^) = 1 then
    begin
      Str(Year, Y);
      Move(Y[1], Title^[10], Length(Y));
      Title^[0] := Chr(9 + Length(Y));
      Owner^.DrawView;
    end;
  end;
end;

procedure TCalendarView.Store(var S: TStream);
begin
  inherited Store(S);
  S.Write(Year, SizeOf(Year) + SizeOf(Delta));
end;

{ TCalendarWindow }
constructor TCalendarWindow.Init;
var
  R: TRect;
  Calendar: PCalendarView;
begin
  R.Assign(0, 0, 80, 26);
  if Desktop^.Size.Y < 26 then R.B.Y := Desktop^.Size.Y;
  inherited Init(R, 'Calendar xxxx', wnNoNumber);
  GrowMode := gfGrowAll or gfGrowRel;
  Options := (Options or ofCentered) and not ofBuffered;
  Palette := wpCyanWindow;
  GetExtent(R);
  R.Grow(-1, -1);
  Calendar := New(PCalendarView, Init(R));
  Insert(Calendar);
  Calendar^.SetYear(Calendar^.Year);
end;

procedure TCalendarWindow.HandleEvent(var Event: TEvent);
var
  Mouse: TPoint;
begin
  inherited HandleEvent(Event);
  if (Event.What = evMouseDown) or (Event.What = evMouseAuto) then
  begin
    MakeLocal(Event.Where, Mouse);
    if Mouse.X = 0 then
      Message(Current, evKeyDown, kbLeft, nil)
    else
    if Mouse.X = Size.X - 1 then
      Message(Current, evKeyDown, kbRight, nil)
    else
      Exit;
    ClearEvent(Event);
  end
  else
  if (Event.What = evKeyDown) and (Event.KeyCode = kbEsc) then
  begin
    Message(@Self, evCommand, cmClose, nil);
    ClearEvent(Event);
  end;
end;

procedure RegisterCalendar;
begin
  RegisterType(RCalendarView);
  RegisterType(RCalendarWindow);
end;

end.
