///////////////////////////////////////////////////////////////////////////////
// JULIAN.CPP * By Gregory Kwok <gkwok@jps.net>
// Calculates Julian date for today or any given day
// 1.0  07 Jan 1997   First version (written in one hour)
// 1.2  09 Jan 1997   Does reverse Julian dates (converts to Gregorian)
//                    Fixed constant bug (July doesn't have 30 days!)
//                    Handles Sep 1752 correctly, like Unix "cal"
///////////////////////////////////////////////////////////////////////////////
// Personal comments: I think this program is one of my most efficient.
// Perhaps it came from working with Scheme this semester, and learning how
// to "nest" calculations inside others, or maybe it's just my experience
// with C/C++. Anyhow, there's some little tricks in here and some pretty
// fancy for loops.
///////////////////////////////////////////////////////////////////////////////

// Include files
#include <ctype.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// Function prototypes
void parseArgs(int, char*[] , struct date* , int* );
void checkParams(const struct date* , int);
void printResults(const struct date* , int);
void printHelp(void);
void convertToGregorian(struct date* , int, int);
int julianDay(const struct date* );
int monthNameToNum(const char* );
const char* numToMonthName(int);
int isLeapYear(int);

// Global constants... The indexes start at 1 because getdate() does,
// so it makes it easier to convert
const char* MonthNames[] = { "",
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
const int MonthLengths[] = { 0,
  31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

#define GREG2JUL (0)
#define JUL2GREG (1)

///////////////////////////////////////////////////////////////////////////////
// main()...I take pride in short main() functions.
// curdate is always passed as a pointer for memory efficiency
int main(int argc, char* argv[])
{
  struct date curdate;
  int mode = GREG2JUL;                           // How are we converting?

  getdate(&curdate);                             // Today is the default date
  parseArgs(argc, argv, &curdate, &mode);        // But user can change that
  checkParams(&curdate, mode);                   // Make sure changes are valid
  printResults(&curdate, mode);                  // And calculate!

  return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Check command line for /R and /? switches, then get month, date, and year
// This is a pretty tedious function...too many ifs and elses :)
void parseArgs(int argc, char* argv[], struct date* d, int* mode)
{
  if (argc >= 2) {                                    // If one or more arguments
    if (argv[1][0] == '/' || argv[1][0] == '-') {     // ..and first arg begins with / or -
      if (!stricmp(argv[1] + 1, "r"))                 // ..and first arg is /R or -R (insensitive)
        *mode = JUL2GREG;
      else if (argv[1][1] == '?') {                   // ..or first arg is /? or -?
        printHelp();
        exit(0);
      }
      else {
        printHelp();
        printf("\nUnrecognized switch: %s\n", argv[1]);
        exit(1);
      }
    }
    else                               // If argc == 2, the other ifs won't get executed
      d->da_day = atoi(argv[1]);       // ..and this is the default (specified day of this month).
  }                                    // ..If argc > 2, d->da_day will get overwritten later

  if (argc >= 4)                       // Get the year if there's more than three arguments
      d->da_year = atoi(argv[3]);      // ..This must happen before the next if

  if (argc >= 3) {
    if (*mode == JUL2GREG)
      convertToGregorian(d, atoi(argv[2]), d->da_year);
    else {
      // Wow! I wasn't sure this would work--
      d->da_mon = (isdigit(argv[1][0]) ? atoi : monthNameToNum)(argv[1]);
      d->da_day = atoi(argv[2]);
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
// Make sure the date is valid
void checkParams(const struct date* d, int mode)
{
  if (mode == GREG2JUL) {              // Converting MMM dd, YYYY -> jjj
    // Year must be >= 0
    if (d->da_year < 0) {
      printf("Invalid year value: %d\n", d->da_year);
      exit(1);
    }

    // Month must be 1 <= x <= 12
    if (d->da_mon < 1 || d->da_mon > 12) {
      printf("Invalid month value: %d\n", d->da_mon);
      exit(1);
    }

    // Check if day is out of range for its month
    // If it's Feb 29 and a leap year, subtract 1 from the day so it works...
    if (d->da_day < 1 || d->da_day - (d->da_mon == 2 && isLeapYear(d->da_year)) > MonthLengths[d->da_mon]) {
      printf("Invalid date: %s %d, %d\n",
        numToMonthName(d->da_mon),
        d->da_day,                     // Note that date.da_day's are chars, so you may not see
        d->da_year);                   // ..the exact number you typed in; it'll be %'ed by 256
      exit(1);
    }

    // Final very-special case check
    if (d->da_year == 1752 && d->da_mon >= 9 && d->da_day >=3 && d->da_day <= 13) {
        puts("Invalid date: Sep 3-13, 1752 didn't exist!");
        exit(1);
    }
  }
  else {                               // Converting jjj YYYY -> MMM dd, YYYY
    int j = julianDay(d);              // Calculate once as we may need it four times
    if (j < 1 || j > 365 + isLeapYear(d->da_year)) {
      printf("Invalid Julian value: %d\n", j);
      exit(1);
    }

    // Special case for 1752 (note that 1752 was a leap year!)
    if (d->da_year == 1752 && j > 355) {
      puts("Invalid Julian value: 1752 had only 355 days!");
      exit(1);
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
void printResults(const struct date* d, int mode)
{
  if (mode == GREG2JUL)                // MMM dd YYYY -> jjj
    printf("%s %02d, %d = %d\n",
      numToMonthName(d->da_mon),
      d->da_day,
      d->da_year,
      julianDay(d));
  else                                 // jjj YYYY -> MMM dd YYYY
    printf("%d = %s %02d, %d\n",
      julianDay(d),
      numToMonthName(d->da_mon),
      d->da_day,
      d->da_year);
}

///////////////////////////////////////////////////////////////////////////////
// Show documentation
void printHelp(void)
{
  puts("\nJULIAN 1.2 \xfe By Gregory Kwok <gkwok@jps.net>\n"
       "\nUsage: julian [/r] [month] [day] [year]"
       "\nExamples:"
       "\n  julian /?          : Display this help message"
       "\n  julian             : Julian date for today"
       "\n  julian 7           : Julian date for the 7th of this month"
       "\n  julian nov 11      : Julian date for November 11 of this year"
       "\n  julian 6 18        : Julian date for June 18 of this year"
       "\n  julian 5 23 2001   : Returns 143, also accepts \"julian may 23 2001\""
       "\n  julian /r 149      : Returns May 29, or May 28 for a leap year"
       "\n  julian /r 233 1996 : Returns August 20, 1996");
}

///////////////////////////////////////////////////////////////////////////////
// Writes given Julian day and year to date structure
void convertToGregorian(struct date* d, int julianDay, int year)
{
  // If on or after supposed "Sep 3 1752" just add 11 before calculating :)
  if (d->da_year == 1752 && julianDay >= 247)
    julianDay += 11;

  // Can't assign julianDay directly to d->da_day because d->da_day is a char :(
  int day = julianDay;

  // Keep subtracting days from d->da_day until it's less than a whole month
  // Increment d->da_mon by 1 each time we subtract
  for (d->da_mon = 1; day > MonthLengths[d->da_mon]; day -= MonthLengths[d->da_mon++])
    ;

  d->da_day  = day;                    // At this point, day is small enough for a char
  d->da_year = year;                   // Yes, for this program, it's redundant, but outside of this program, struct d would be "unprepared"

  // If a leap year and at or past Mar 1 (unconverted), decrement the day, otherwise do nothing
  // If it was the first of the month (now the 0th), decrement the month also (otherwise stop)
  // If the new month is Feb, set the day to 29, else to the length of the month
  if (isLeapYear(d->da_year) && julianDay >= 60 && --d->da_day == 0)
    d->da_day = (--d->da_mon == 2) ? 29 : MonthLengths[d->da_mon];
}

///////////////////////////////////////////////////////////////////////////////
// Converts a date structure to a Julian date
int julianDay(const struct date* d)
{
  // Start julianDay at the day of the given month; add 1 if leap year
  int julianDay = d->da_day + (isLeapYear(d->da_year) && d->da_mon >= 3);

  // Add numbers to julianDay from MonthLengths[] for previous months
  for (int month = 1; month < d->da_mon; julianDay += MonthLengths[month++])
    ;

  // Special case for 1752--if on or after Sep 14, subtract 11 days
  if (d->da_year == 1752 && (d->da_mon > 9 || (d->da_mon == 9 && d->da_day >= 14)))
    julianDay -= 11;

  return julianDay;
}

///////////////////////////////////////////////////////////////////////////////
// Converts "Jan" to 1, etc.
int monthNameToNum(const char* name)
{
  for (int i = 1; i <= 12; i++)
    if (!stricmp(MonthNames[i], name))
      return i;

  printf("Unrecognized month: %s\n", name);
  exit(1);
  return -1;                           // This line should never be reached
}

///////////////////////////////////////////////////////////////////////////////
// Converts 1 to "Jan", etc.
const char* numToMonthName(int month)
{
  return MonthNames[month];
}

///////////////////////////////////////////////////////////////////////////////
// Returns true if given year is a leap year
// Years are leap years if they are divisible by 4 and *not* divisible by 100,
// but if they are divisible by 400, they are leap years no matter what
int isLeapYear(int year)
{
  return ((!(year % 4) && (year % 100)) ||
           !(year % 400));
}