(*****************************************************************************

  Matrices
    version 1.0

    This unit holds procedures designed to manipulate matrices as arrays
    of real numbers.  All standard matrix procedures are supported.

    Purpose:
       To support the manipulation of matrices with a standard library of
       routines.

    How it works!
       Matrices are allocated using the allocate matrix function.  Then
       a variety of procedures are available to act upon the matrices.
       Finally, when the matrix is no longer needed, it can be discarded.

    Features...
      Error checking in matrix operations.
      Allows use of the math coprocessor if compiled in that compiler state.
      Allocates only the amount of memory needed to hold the specified
        matrix. { no wasted memory }

  Copyright 1990, 1994, All rights reserved.
    Paul R. Renaud

  Compilers:
    Turbo Pascal versions 4.0 to 6.0
    Speed Pascal/2 version 1.5

  System:
    MS-DOS, MDOS, OS/2

*****************************************************************************)

Unit Matrices;

  Interface

    Const
      Max_Matrix_Size = 16;    { Largest possible size is 16 by 16 matrix }

   {$IFOPT N-}
    Const
     {$IFDEF Ver40 }
      Maximum_Matrix_Number: Real = 1.5e38;
     {$ELSE}
      Maximum_Matrix_Number: Real = 1.701411834603918604e38;
     {$ENDIF}
    Type
      Matrix_Number = Real;    { Support coprocessor if desired }
   {$ELSE}
    Const
      Maximum_Matrix_Number: Extended = 1.189731495357231764e4932;
    Type
      Matrix_Number = Extended;
   {$ENDIF}

(***********************************************************

  Function: Allocate the matrix.

    This function is needed to allocate a matrix.  It
    returns the address to the matrix which is allocated to
    exactly fit the parameters.

***********************************************************)

    Function Allocate_Matrix( Row_Size, Column_Size: Byte ): Pointer;

(***********************************************************

  Procedure: Deallocate the matrix.

    This procedure allows you to dispose of the matrix when
    finished with it.  This is useful to free up extra
    memory and automatically will properly dispose of the
    matrix.

***********************************************************)

    Procedure Deallocate_Matrix( Var Matrix: Pointer );

(***********************************************************

  Function: Copy the matrix.

    This function takes one matrix and makes a copy of it.
    If the sizes are incompatible, then it will return false
    and will not make a copy.

***********************************************************)

    Function Copy_Matrix( Var Operand, Result: Pointer ): Boolean;

(***********************************************************

  Procedure: Get the size.

    This procedure takes the matrix supplied and returns the
    size in terms of row and column.

***********************************************************)

    Procedure Get_Size( Var Matrix: Pointer; Var Row, Column: Byte );

(***********************************************************

  Function: Get from matrix.

    This function returns a specified value from the matrix
    directed by the supplied row and column values.

***********************************************************)

    Function Get_From_Matrix( Var Matrix: Pointer; Row, Column: Byte ): Matrix_Number;

(***********************************************************

  Procedure: Put in Matrix.

    This procedure puts the supplied value in data into the
    matrix at the location specified by row and column.

***********************************************************)

    Procedure Put_In_Matrix( Var Matrix: Pointer; Row, Column: Byte; Data: Matrix_Number );

(***********************************************************

  Function: Add matrices.

    This function attempts to add the two supplied matrices
    and returns them in result.  If it fails, it returns
    false.

***********************************************************)

    Function Add_Matrices( Var Operand1, Operand2, Result: Pointer ): Boolean;

(***********************************************************

  Function: Subtract matrices.

    This function attempts to subtract the second matrix
    from the first and return the result in Result.  If it
    fails, it returns false.

***********************************************************)

    Function Subtract_Matrices( Var Operand1, Operand2, Result: Pointer ): Boolean;

(***********************************************************

  Function: Multiply matrices.

    This function attempts to multiply two matrices together
    and return the result in Result.  If it fails, it
    returns false.

***********************************************************)

    Function Multiply_Matrices( Var Operand1, Operand2, Result: Pointer ): Boolean;

(***********************************************************

  Function: Multiply with matrix.

    This function attempts to multiply all values in the
    matrix by a supplied scalar value.  If it fails, it
    returns false.

***********************************************************)

    Function Multiply_With_Matrix( Scalar: Matrix_Number; Var Operand, Result: Pointer ): Boolean;

(***********************************************************

  Function: Transpose matrix.

    This function attempts to transpose the matrix.  If it
    fails, it returns false.

***********************************************************)

    Function Transpose_Matrix( Var Operand, Result: Pointer ): Boolean;

(***********************************************************

  Procedure: Read in the matrix.

    This procedure reads in a matrix from the provided file.
    The matrix is expected to be encoded in the standard
    format.

***********************************************************)

    Procedure Read_In_Matrix( Var InFile: Text; Var Matrix: Pointer );

(***********************************************************

  Procedure: Write out the matrix.

    This procedure takes the matrix and prints it out in
    standard format in the provided file.

***********************************************************)

    Procedure Write_Out_Matrix( Var OutFile: Text; Var Matrix: Pointer );

(***********************************************************

  Procedure: Solve the matrix.

    This procedure solves the given matrix and returns the
    result in the first row of the given matrix.  Make sure
    to use a copy of your original matrix, or it will be
    lost.  This solving is akin to solving an equation with
    X variables and unknowns.

***********************************************************)

    Procedure Solve_Matrix( Var Matrix: Pointer );

(***********************************************************

  Function: Determinate slow.

    This function calculates the determinate using the slow
    method.  The slow method is less likely to create
    numerical overloads, but takes much longer to complete
    the calculations.

***********************************************************)

    Function Determinate_Slow( Var Matrix: Pointer ): Matrix_Number;

(***********************************************************

  Function: Determinate fast.

    This function calculates the determinate using the
    faster method.  The faster method will perform the
    calculation almost instantaneous, but my suffer a
    numerical overload with very large matrices.

***********************************************************)

    Function Determinate_Fast( Var Matrix: Pointer ): Matrix_Number;

(***********************************************************

  Procedure: Generate matrix.

    This procedure generates a random matrix with values
    ranging from negative one thousand to one thousand.

***********************************************************)

    Procedure Generate_Matrix( Var Matrix: Pointer );

{-----------------------------------------------------------------------------}

  Implementation

    Type
      { Holds the sign of the matrix. }
      Sign = ( Positive, Negative );
      { Internal type used to define the matrix. }
      Matrix_Pointer = ^Matrix_Type;
      Matrix_Type = Record
                      Row_Size,
                      Column_Size: Byte;
                      Data: Array[ 1 .. 400 ] of Matrix_Number;
                    End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write error.
    This procedure writes out the string and stops
    the program if an error code is passed to it.

*************************************************)

    Procedure Write_Error( Error_Code: Integer; Sentence: String );
      Begin
        If ( Error_Code <> 0 )
          then
            Begin
              WriteLn( 'Error ', Error_Code, ' in ', Sentence, '.' );
              Halt( Error_Code );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Power.
    This function takes a number and raises it
    to the specified power.

*************************************************)

   Function Power( Number: Matrix_Number; Exponent: Integer ): Matrix_Number;
     Var
       Count: Word;
       Result: Matrix_Number;
       Exponent_Sign: Sign;
     Begin
       If ( Number = 0.0 )
         then
           Power := Number
         else
           Begin
             If ( Exponent < 0 )
               then
                 Begin
                   Exponent_Sign := Negative;
                   Exponent := - Exponent;
                 End
               else
                 Exponent_Sign := Positive;
             Result := Number;
             For Count := 2 to Exponent do
               Result := ( Result * Number );
             Case Exponent_Sign of
               Positive: Power := Result;
               Negative: Power := ( 1 / Result );
             End; { Case }
           End;
     End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Calculate the matrix size.
    This function returns the necessary size of
    the data structure needed to hold the matrix.

*************************************************)

    Function Calculate_Matrix_Size( Row_Size, Column_Size: Byte ): Word;
      Var
        Number: Word;
      Begin
        Number := ( Row_Size * Column_Size );
        If ( Number > 0 )
          then
            Calculate_Matrix_Size := ( Number * SizeOf( Matrix_Number ) ) + ( 2 * SizeOf( Byte ) )
          else
            Calculate_Matrix_Size := 0;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate the matrix.
    As previously defined.

*************************************************)

    Function Allocate_Matrix( Row_Size, Column_Size: Byte ): Pointer;
      Var
        Holder: Pointer;
        Total_Size: Word;
      Begin
        If ( ( Row_Size in [ 1 .. Max_Matrix_Size ] ) and ( Column_Size in [ 1 .. Max_Matrix_Size ] ) )
          then
            Begin
              Total_Size := Calculate_Matrix_Size( Row_Size, Column_Size );
              If ( ( Total_Size = 0 ) or ( Total_Size > MaxAvail ) )
                then
                  Allocate_Matrix := Nil
                else
                  Begin
                    GetMem( Holder, Total_Size );
                    Matrix_Type( Holder^ ).Row_Size := Row_Size;
                    Matrix_Type( Holder^ ).Column_Size := Column_Size;
                    Allocate_Matrix := Holder;
                  End;
            End
          else
            Allocate_Matrix := Nil;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get the size.
    As previously defined.

*************************************************)

    Procedure Get_Size( Var Matrix: Pointer; Var Row, Column: Byte );
      Begin
        If ( Matrix = Nil )
          then
            Begin
              Row := 0;
              Column := 0;
            End
          else
            Begin
              Row := Matrix_Type( Matrix^ ).Row_Size;
              Column := Matrix_Type( Matrix^ ).Column_Size;
            End
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Deallocate the matrix.
    As previously defined.

*************************************************)

    Procedure Deallocate_Matrix( Var Matrix: Pointer );
      Var
        Row_Size,
        Column_Size: Byte;
        Total_Size: Word;
      Begin
        If ( Matrix <> Nil )
          then
            Begin
              Row_Size := Matrix_Type( Matrix^ ).Row_Size;
              Column_Size := Matrix_Type( Matrix^ ).Column_Size;
              Total_Size := Calculate_Matrix_Size( Row_Size, Column_Size );
              FreeMem( Matrix, Total_Size );
              Matrix := Nil;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Get the address.
    This function returns the location of the
    specified row and column in the matrix

*************************************************)

    Function Get_Address( Var Matrix: Matrix_Type; Row, Column: Byte ): Word;
      Begin
        Get_Address := ( Matrix.Column_Size * Pred( Row ) ) + Column;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Get from matrix.
    This function returns the value in the
    specified location of the matrix.

*************************************************)

    Function Get_Matrix( Var Matrix: Matrix_Type; Row, Column: Byte ): Matrix_Number;
      Begin
        If ( ( Row < 1 ) or ( Row > Matrix.Row_Size ) or ( Column < 1 ) or ( Column > Matrix.Column_Size ) )
          then
            Get_Matrix := 0
          else
            Get_Matrix := Matrix.Data[ Get_Address( Matrix, Row, Column ) ];
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Put in matrix.
    This function puts the given value in the
    specified part of the matrix.

*************************************************)

    Procedure Put_Matrix( Var Matrix: Matrix_Type; Row, Column: Byte; Data: Matrix_Number );
      Begin
        If ( ( Row > 0 ) and ( Column > 0 ) and ( Row <= Matrix.Row_Size ) and ( Column <= Matrix.Column_Size ) )
          then
            Matrix.Data[ Get_Address( Matrix, Row, Column ) ] := Data;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Get from matrix.
    As previously defined.

*************************************************)

    Function Get_From_Matrix( Var Matrix: Pointer; Row, Column: Byte ): Matrix_Number;
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Get_From_Matrix := 0
          else
            Get_From_Matrix := Get_Matrix( The_Matrix^, Row, Column );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put in Matrix.
    As previously defined.

*************************************************)

    Procedure Put_in_Matrix( Var Matrix: Pointer; Row, Column: Byte; Data: Matrix_Number );
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix <> Nil )
          then
            Put_Matrix( The_Matrix^, Row, Column, Data );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Add matrices.
    As previously defined.

*************************************************)

    Function Add_Matrices( Var Operand1, Operand2, Result: Pointer ): Boolean;
      Var
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
        The_Operand_1: Matrix_Pointer absolute Operand1;
        The_Operand_2: Matrix_Pointer absolute Operand2;
        The_Result: Matrix_Pointer absolute Result;
      Begin
        If ( ( Operand1 = Nil ) or ( Operand2 = Nil ) or ( Result = Nil ) )
          then
            Write_Error( 204, 'Add_Matrices: Variable not initialize' );
        If ( ( The_Operand_1^.Row_Size = The_Result^.Row_Size ) and
             ( The_Operand_1^.Column_Size = The_Result^.Column_Size ) and
             ( The_Operand_2^.Row_Size = The_Result^.Row_Size ) and
             ( The_Operand_2^.Column_Size = The_Result^.Column_Size ) )
          then
            Begin
              For Row_Index := 1 to The_Result^.Row_Size do
                For Column_Index := 1 to The_Result^.Column_Size do
                  Begin
                    Value := ( Get_Matrix( The_Operand_1^, Row_Index, Column_Index ) +
                               Get_Matrix( The_Operand_2^, Row_Index, Column_Index ) );
                    Put_Matrix( The_Result^, Row_Index, Column_Index, Value );
                  End;
              Add_Matrices := True;
            End
          else
            Add_Matrices := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Subtract matrices.
    As previously defined.

*************************************************)

    Function Subtract_Matrices( Var Operand1, Operand2, Result: Pointer ): Boolean;
      Var
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
        The_Operand_1: Matrix_Pointer absolute Operand1;
        The_Operand_2: Matrix_Pointer absolute Operand2;
        The_Result: Matrix_Pointer absolute Result;
      Begin
        If ( ( Operand1 = Nil ) or ( Operand2 = Nil ) or ( Result = Nil ) )
          then
            Write_Error( 204, 'Subtract_Matrices: Variable not initialize' );
        If ( ( The_Operand_1^.Row_Size = The_Result^.Row_Size ) and
             ( The_Operand_1^.Column_Size = The_Result^.Column_Size ) and
             ( The_Operand_2^.Row_Size = The_Result^.Row_Size ) and
             ( The_Operand_2^.Column_Size = The_Result^.Column_Size ) )
          then
            Begin
              For Row_Index := 1 to The_Result^.Row_Size do
                For Column_Index := 1 to The_Result^.Column_Size do
                  Begin
                    Value := ( Get_Matrix( The_Operand_1^, Row_Index, Column_Index ) -
                               Get_Matrix( The_Operand_2^, Row_Index, Column_Index ) );
                    Put_Matrix( The_Result^, Row_Index, Column_Index, Value );
                  End;
              Subtract_Matrices := True;
            End
          else
            Subtract_Matrices := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply with matrix.
    As previously defined.

*************************************************)

    Function Multiply_With_Matrix( Scalar: Matrix_Number; Var Operand, Result: Pointer ): Boolean;
      Var
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
        The_Operand: Matrix_Pointer absolute Operand;
        The_Result: Matrix_Pointer absolute Result;
      Begin
        If ( ( Operand = Nil ) or ( Result = Nil ) )
          then
            Write_Error( 204, 'Multiply_With_Matrices: Variable not initialize' );
        If ( ( The_Operand^.Row_Size = The_Result^.Row_Size ) and
             ( The_Operand^.Column_Size = The_Result^.Column_Size ) )
          then
            Begin
              For Row_Index := 1 to The_Result^.Row_Size do
                For Column_Index := 1 to The_Result^.Column_Size do
                  Begin
                    Value := ( Get_Matrix( The_Operand^, Row_Index, Column_Index ) * Scalar );
                    Put_Matrix( The_Result^, Row_Index, Column_Index, Value );
                  End;
              Multiply_With_Matrix := True;
            End
          else
            Multiply_With_Matrix := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Multiply matrices.
    As previously defined.

*************************************************)

    Function Multiply_Matrices( Var Operand1, Operand2, Result: Pointer ): Boolean;
      Var
        Index,
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
        The_Operand_1: Matrix_Pointer absolute Operand1;
        The_Operand_2: Matrix_Pointer absolute Operand2;
        The_Result: Matrix_Pointer absolute Result;
      Begin
        If ( ( Operand1 = Nil ) or ( Operand2 = Nil ) or ( Result = Nil ) )
          then
            Write_Error( 204, 'Multiply_Matrices: Variable not initialize' );
        If ( ( The_Operand_1^.Row_Size = The_Result^.Row_Size ) and
             ( The_Operand_1^.Column_Size = The_Operand_2^.Row_Size ) and
             ( The_Operand_2^.Column_Size = The_Result^.Column_Size ) )
          then
            Begin
              For Row_Index := 1 to The_Result^.Row_Size do
                For Column_Index := 1 to The_Result^.Column_Size do
                  Begin
                    Value := 0;
                    For Index := 1 to The_Operand_2^.Row_Size do
                      Value := Value + ( Get_Matrix( The_Operand_1^, Row_Index, Index ) *
                                         Get_Matrix( The_Operand_2^, Index, Column_Index ) );
                    Put_Matrix( The_Result^, Row_Index, Column_Index, Value );
                  End;
              Multiply_Matrices := True;
            End
          else
            Multiply_Matrices := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Transpose matrix.
    As previously defined.

*************************************************)

    Function Transpose_Matrix( Var Operand, Result: Pointer ): Boolean;
      Var
        Row_Index,
        Column_Index: Byte;
        The_Operand: Matrix_Pointer absolute Operand;
        The_Result: Matrix_Pointer absolute Result;
      Begin
        If ( ( Operand = Nil ) or ( Result = Nil ) )
          then
            Write_Error( 204, 'Transpose_Matrix: Variable not initialize' );
        If ( ( The_Operand^.Row_Size = The_Result^.Column_Size ) and
             ( The_Operand^.Column_Size = The_Result^.Row_Size ) )
          then
            Begin
              For Row_Index := 1 to The_Result^.Row_Size do
                For Column_Index := 1 to The_Result^.Column_Size do
                  Put_Matrix( The_Result^, Column_Index, Row_Index, Get_Matrix( The_Operand^, Row_Index, Column_Index ) );
              Transpose_Matrix := True;
            End
          else
            Transpose_Matrix := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read in the matrix internal.
    This procedure reads in the matrix from the
    given file.

*************************************************)

    Procedure Read_In_Matrix_Internal( Var InFile: Text; Var Matrix: Matrix_Type );
      Var
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
      Begin
        For Row_Index := 1 to Matrix.Row_Size do
          For Column_Index := 1 to Matrix.Column_Size do
            Begin
              Read( InFile, Value );
              Put_Matrix( Matrix, Row_Index, Column_Index, Value );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write out the matrix internal.
    This procedure writes out the matrix to the
    given file.

*************************************************)

    Procedure Write_Out_Matrix_Internal( Var OutFile: Text; Var Matrix: Matrix_Type );
      Var
        Row_Index,
        Column_Index: Byte;
      Begin
        For Row_Index := 1 to Matrix.Row_Size do
          Begin
            For Column_Index := 1 to Matrix.Column_Size do
              Write( OutFile, ' ', Get_Matrix( Matrix, Row_Index, Column_Index ), ' ' );
            WriteLn( OutFile );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Interchange rows
    This procedure exchanges the values in the
    two given rows.

*************************************************)

    Procedure Interchange_Row( Var Matrix: Matrix_Type; Row_1, Row_2: Byte );
      Var
        Column_Index: Byte;
        Value: Matrix_Number;
      Begin
        If ( ( Row_1 < 1 ) or ( Row_1 > Matrix.Row_Size ) or
             ( Row_2 < 1 ) or ( Row_2 > Matrix.Row_Size ) or ( Row_1 = Row_2 ) )
          then
            Write_Error( 201, 'Interchange_Row: Row_1 or Row_2 out of range' );
        For Column_Index := 1 to Matrix.Column_Size do
          Begin
            Value := Get_Matrix( Matrix, Row_1, Column_Index );
            Put_Matrix( Matrix, Row_1, Column_Index, Get_Matrix( Matrix, Row_2, Column_Index ) );
            Put_Matrix( Matrix, Row_2, Column_Index, Value );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Multiply with row.
    This procedure multiplies all the values in
    the given row by the given value.

*************************************************)

    Procedure Multiply_Row( Var Matrix: Matrix_Type; Row: Byte; Scalar: Matrix_Number );
      Var
        Column_Index: Byte;
        Value: Matrix_Number;
      Begin
        If ( ( Row < 1 ) or ( Row > Matrix.Row_Size ) or ( Scalar = 0.0 ) )
          then
            Write_Error( 201, 'Multiply_Row: Row or Scalar out of range' );
        For Column_Index := 1 to Matrix.Column_Size do
          Begin
            Value := ( Scalar * Get_Matrix( Matrix, Row, Column_Index ) );
            Put_Matrix( Matrix, Row, Column_Index, Value );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Add to row.
    This procedure multiplies all the values in
    the first row by the scalar value, then adds
    the results to the values in the second row.

*************************************************)

    Procedure Add_Row( Var Matrix: Matrix_Type; Row_1, Row_2: Byte; Scalar: Matrix_Number );
      Var
        Column_Index: Byte;
        Value: Matrix_Number;
      Begin
        If ( ( Row_1 < 1 ) or ( Row_1 > Matrix.Row_Size ) or
             ( Row_2 < 1 ) or ( Row_2 > Matrix.Row_Size ) or ( Row_1 = Row_2 ) )
          then
            Write_Error( 201, 'Add_Row: Row_1 or Row_2 out of range' );
        For Column_Index := 1 to Matrix.Column_Size do
          Begin
            Value := ( ( Scalar * Get_Matrix( Matrix, Row_1, Column_Index ) ) + Get_Matrix( Matrix, Row_2, Column_Index ) );
            Put_Matrix( Matrix, Row_2, Column_Index, Value );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Are they all zeros?
    This function returns true if all the values
    in the given row are zero.

*************************************************)

    Function All_Zeros( Var Matrix: Matrix_Type; Column, Top: Byte ): Boolean;
      Var
        Row_Index: Byte;
      Begin
        If ( ( Column < 1 ) or ( Column > Matrix.Column_Size ) )
          then
            Write_Error( 201, 'All_Zeros: Column out of range' );
        Row_Index := Top;
        While ( Row_Index <= Matrix.Row_Size ) and ( Get_Matrix( Matrix, Row_Index, Column ) = 0.0 ) do
          Inc( Row_Index );
        All_Zeros := ( Row_Index > Matrix.Row_Size );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Find the pivot column.
    This function returns the column that is best
    suited for the pivot column, otherwise it
    returns zero.

*************************************************)

    Function Find_Pivot_Column( Var Matrix: Matrix_Type; Top: Byte ): Byte;
      Var
        Done: Boolean;
        Column_Index: Byte;
      Begin
        Done := False;
        Column_Index := 1;
        Repeat
          If All_Zeros( Matrix, Column_Index, Top )
            then
              Inc( Column_Index )
            else
              Done := True;
        Until ( ( Column_Index > Matrix.Column_Size ) or Done );
        If Done
          then
            Find_Pivot_Column := Column_Index
          else
            Find_Pivot_Column := 0;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Find the pivot row.
    This function returns the row that is best
    suited for the pivot row, otherwise it
    returns zero.

*************************************************)

    Function Find_Pivot_Row( Var Matrix: Matrix_Type; Column, Top: Byte ): Byte;
      Var
        Result,
        Row_Index: Byte;
      Begin
        If ( ( Column < 1 ) or ( Column > Matrix.Column_Size ) )
          then
            Write_Error( 201, 'Find_Pivot_Row: Column out of range' );
        Result := 0;
        For Row_Index := Matrix.Row_Size downto Top do
          Begin
            If ( Get_Matrix( Matrix, Row_Index, Column ) <> 0.0 )
              then
                Result := Row_Index;
          End;
        Find_Pivot_Row := Result;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Solve the matrix.
    This procedure solves the matrix using the
    matrix manipulation method.  It uses
    recursion so should have a large stack area
    for large matrices.

*************************************************)

    Procedure Solve( Var Matrix: Matrix_Type; Top: Byte );
      Var
        Look,
        Pivot_Row,
        Pivot_Column: Byte;
        Value: Matrix_Number;
      Begin
        If ( ( Matrix.Row_Size > 1 ) or ( Matrix.Column_Size > 1 ) )
          then
            Begin
              Pivot_Column := Find_Pivot_Column( Matrix, Top );
              If ( Pivot_Column <> 0 )
                then
                  Begin
                    Pivot_Row := Find_Pivot_Row( Matrix, Pivot_Column, Top );
                    If ( Pivot_Row = 0 )
                      then
                        Write_Error( 201, 'Solve: Pivot_Row equals zero' );
                    If ( Pivot_Row > Top )
                      then
                        Interchange_Row( Matrix, Top, Pivot_Row );
                    Pivot_Row := Top;
                    Value := ( 1 / Get_Matrix( Matrix, Pivot_Row, Pivot_Column ) );
                    Multiply_Row( Matrix, Pivot_Row, Value );
                    For Look := 1 to Matrix.Row_Size do
                      If ( Look <> Top )
                        then
                          Begin
                            Value := - Get_Matrix( Matrix, Look, Pivot_Column );
                            If ( Value <> 0.0 )
                              then
                                Add_Row( Matrix, Top, Look, Value );
                          End;
                    Solve( Matrix, Succ( Top ) );
                  End;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Solve the matrix.
    As previously defined.

*************************************************)

    Procedure Solve_Matrix( Var Matrix: Pointer );
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Write_Error( 204, 'Solve_Matrix: Variable not initialize' );
        Solve( The_Matrix^, 1 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Copy the matrix internal.
    This function copies the given matrix.

*************************************************)

    Function Copy_Matrix_Internal( Var Operand, Result: Matrix_Type ): Boolean;
      Var
        Row_Index,
        Column_Index: Byte;
      Begin
        If ( ( Operand.Row_Size <= Result.Row_Size ) and ( Operand.Column_Size <= Result.Column_Size ) )
          then
            Begin
              For Row_Index := 1 to Result.Row_Size do
                For Column_Index := 1 to Result.Column_Size do
                  Put_Matrix( Result, Row_Index, Column_Index, Get_Matrix( Operand, Row_Index, Column_Index ) );
              Copy_Matrix_Internal := True;
            End
          else
            Copy_Matrix_Internal := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Inverse the matrix.
    This function tries to find the inverse of
    the given matrix.  If it isn't found, it
    returns false.

*************************************************)

    Function Inverse_Matrix( Var Operand, Result: Matrix_Type ): Boolean;
      Var
        Look,
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
        Large_Matrix: Matrix_Pointer;
      Begin
        Inverse_Matrix := False;
        If ( ( Operand.Row_Size = Operand.Column_Size ) and
             ( Result.Column_Size = Result.Row_Size ) and
             ( Operand.Row_Size = Result.Row_Size ) )
          then
            Begin
              Large_Matrix := Allocate_Matrix( Operand.Row_Size, ( 2 * Operand.Row_Size ) );
              If ( Large_Matrix <> Nil )
                then
                  If Copy_Matrix_Internal( Operand, Large_Matrix^ )
                    then
                      Begin
                        Look := Operand.Column_Size;
                        For Row_Index := 1 to Operand.Row_Size do
                          Begin
                            Inc( Look );
                            For Column_Index := Succ( Operand.Column_Size ) to ( 2 * Operand.Column_Size ) do
                              If ( Column_Index <> Look )
                                then
                                  Put_Matrix( Large_Matrix^, Row_Index, Column_Index, 0.0 )
                                else
                                  Put_Matrix( Large_Matrix^, Row_Index, Column_Index, 1.0 )
                          End;
                        Solve( Large_Matrix^, 1 );
                        For Row_Index := 1 to Result.Row_Size do
                          For Column_Index := 1 to Result.Column_Size do
                            Begin
                              Value := Get_Matrix( Large_Matrix^, Row_Index, ( Column_Index + Operand.Column_Size ) );
                              Put_Matrix( Result, Row_Index, Column_Index, Value );
                            End;
                        For Column_Index := 1 to Result.Column_Size do
                          If ( Get_Matrix( Large_Matrix^, Result.Row_Size, Column_Index ) <> 0.0 )
                            then
                              Inverse_Matrix := True;
                        Deallocate_Matrix( Pointer( Large_Matrix ) );
                      End
            End
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Fill the small matrix.
    This procedure takes one matrix and puts it
    into the second, eliminating the last row
    and column.

*************************************************)

    Procedure Fill_Small_Matrix( Var Matrix, Small_Matrix: Matrix_Type; Size, Point: Byte );
      Var
        Row_2,
        Row_Index,
        Column_Index: Byte;
      Begin
        For Row_Index := 1 to Pred( Size ) do
          Begin
            Row_2 := Succ( Row_Index );
            For Column_Index := 1 to Pred( Point ) do
              Put_Matrix( Small_Matrix, Row_Index, Column_Index, Get_Matrix( Matrix, Row_2, Column_Index ) );
            For Column_Index := Point to Pred( Size ) do
              Put_Matrix( Small_Matrix, Row_Index, Column_Index, Get_Matrix( Matrix, Row_2, Succ( Column_Index ) ) );
          End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Default determinate.
    This procedure calculates the determinate
    for a two by two matrix.

*************************************************)

    Function Default_Determinate( Var Matrix: Matrix_Type ): Matrix_Number;
      Begin
        Case Matrix.Row_Size of
          1: Default_Determinate := Get_Matrix( Matrix, 1, 1 );
          2: Default_Determinate := ( ( Get_Matrix( Matrix, 1, 1 ) * Get_Matrix( Matrix, 2, 2 ) ) -
                                      ( Get_Matrix( Matrix, 1, 2 ) * Get_Matrix( Matrix, 2, 1 ) ) );
          else Write_Error( 201, 'Default_Determinate: Matrix size is invalid' );
        End;  { Case }
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Slow determinate.
    This procedure calculates the determinate
    using the slower method.

*************************************************)

    Function Slow_Determinate( Var Matrix: Matrix_Type ): Matrix_Number;
      Var
        Size,
        Point: Byte;
        Second_Matrix: Matrix_Pointer;
        Calculate,
        Determinate_Value,
        Determinate_Result: Matrix_Number;
      Begin
        If ( Matrix.Row_Size > Matrix.Column_Size )
          then
            Write_Error( 201, 'Slow_Determinate: Row_Size is greater than Column_Size' );
        Size := Matrix.Row_Size;
        If ( Size < 3 )
          then
            Determinate_Value := Default_Determinate( Matrix )
          else
            Begin
              Second_Matrix := Allocate_Matrix( Pred( Size ), Pred( Size ) );
              If ( Second_Matrix = Nil )
                then
                  Write_Error( 203, 'Slow_Determinate: Out of heap space' );
              Determinate_Value := 0;
              For Point := 1 to Size do
                Begin
                  Fill_Small_Matrix( Matrix, Second_Matrix^, Size, Point );
                  Determinate_Result := Slow_Determinate( Second_Matrix^ );
                  Calculate := ( Get_Matrix( Matrix, 1, Point ) * Determinate_Result );
                  If Odd( Point )
                    then
                      Determinate_Value := Determinate_Value + Calculate
                    else
                      Determinate_Value := Determinate_Value - Calculate;
                End;
              Deallocate_Matrix( Pointer( Second_Matrix ) );
            End;
        Slow_Determinate := Determinate_Value;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Fast determinate.
    This function finds the determinate of the
    matrix using the faster method.

*************************************************)

    Function Fast_Determinate( Var Matrix: Matrix_Type ): Matrix_Number; Forward;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Slow determinate Fast.
    This procedure calculates the determinate
    using the slower method then calls the faster
    method.

*************************************************)

    Function Slow_Determinate_Fast( Var Matrix: Matrix_Type ): Matrix_Number;
      Var
        Size,
        Point: Byte;
        Second_Matrix: Matrix_Pointer;
        Calculate,
        Determinate_Value,
        Determinate_Result: Matrix_Number;
      Begin
        If ( Matrix.Row_Size > Matrix.Column_Size )
          then
            Write_Error( 201, 'Slow_Determinate: Row_Size is greater than Column_Size' );
        Size := Matrix.Row_Size;
        If ( Size < 3 )
          then
            Determinate_Value := Default_Determinate( Matrix )
          else
            Begin
              Second_Matrix := Allocate_Matrix( Pred( Size ), Pred( Size ) );
              If ( Second_Matrix = Nil )
                then
                  Write_Error( 203, 'Slow_Determinate: Out of heap space' );
              Determinate_Value := 0;
              For Point := 1 to Size do
                Begin
                  Fill_Small_Matrix( Matrix, Second_Matrix^, Size, Point );
                  Determinate_Result := Fast_Determinate( Second_Matrix^ );
                  Calculate := ( Get_Matrix( Matrix, 1, Point ) * Determinate_Result );
                  If Odd( Point )
                    then
                      Determinate_Value := Determinate_Value + Calculate
                    else
                      Determinate_Value := Determinate_Value - Calculate;
                End;
              Deallocate_Matrix( Pointer( Second_Matrix ) );
            End;
        Slow_Determinate_Fast := Determinate_Value;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Prepare.
    This function prepares the matrix for solving
    the determinate using the fast method.

*************************************************)

    Procedure Prepare( Var Number: Matrix_Number; Var Matrix: Matrix_Type );
      Var
        Pivot_Row,
        Pivot_Column: Byte;
      Begin
        If ( ( Matrix.Row_Size > 1 ) or ( Matrix.Column_Size > 1 ) )
          then
            Begin
              Pivot_Column := Find_Pivot_Column( Matrix, 1 );
              If ( Pivot_Column <> 0 )
                then
                  Begin
                    Pivot_Row := Find_Pivot_Row( Matrix, Pivot_Column, 1 );
                    If ( Pivot_Row = 0 )
                      then
                        Write_Error( 201, 'Prepare: Pivot_Row equals zero' );
                    If ( Pivot_Row > 1 )
                      then
                        Begin
                          Interchange_Row( Matrix, 1, Pivot_Row );
                          Number := - Number;
                        End;
                  End;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function Reduce the matrix.
    This function takes the matrix and reduces
    it.

*************************************************)

    Function Reduced_Matrix( Var Matrix: Matrix_Type; Size: Byte; Var Smallest: Matrix_Number ): Matrix_Pointer;
      Var
        Row_2,
        Column_2,
        Row_Index,
        Column_Index: Byte;
        Small_Matrix: Matrix_Pointer;
        Value: Matrix_Number;
      Begin
        Smallest := Maximum_Matrix_Number;
        Small_Matrix := Allocate_Matrix( Pred( Size ), Pred( Size ) );
        If ( Small_Matrix = Nil )
          then
            Write_Error( 203, 'Reduced_Matrix: Out of heap space' );
        For Row_Index := 1 to Pred( Size ) do
          Begin
            Row_2 := Succ( Row_Index );
            For Column_Index := 1 to Pred( Size ) do
              Begin
                Column_2 := Succ( Column_Index );
                Value := ( ( Get_Matrix( Matrix, 1, 1 ) * Get_Matrix( Matrix, Row_2, Column_2 ) ) -
                           ( Get_Matrix( Matrix, 1, Column_2 ) * Get_Matrix( Matrix, Row_2, 1 ) ) );
                If ( Value < Smallest )
                  then
                    Smallest := Value;
                Put_Matrix( Small_Matrix^, Row_Index, Column_Index, Value );
              End;
          End;
        Reduced_Matrix := Small_Matrix;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Originate.
    This function takes the matrix and reduces it
    so that numeric overflows don't happen as
    often.

*************************************************)

    Procedure Originate( Var Matrix: Matrix_Type; Var Smallest, Number: Matrix_Number );
      Const
        Adjust_Value = 512;  { Can be any number greater than one, but powers of two work best. }
      Var
        Row,
        Column: Byte;
        Value: Matrix_Number;
      Begin
        If ( Abs( Smallest ) > Adjust_Value )
          then
            Begin
              Smallest := Maximum_Matrix_Number;
              For Row := 1 to Matrix.Row_Size do
                Begin
                  For Column := 1 to Matrix.Column_Size do
                    Begin
                      Value := ( Get_Matrix( Matrix, Row, Column ) / Adjust_Value );
                      If ( Value < Smallest )
                        then
                          Smallest := Value;
                      Put_Matrix( Matrix, Row, Column, Value );
                    End;
                  Number := Number * Adjust_Value;
                End;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Fast determinate.
    As previously defined.

*************************************************)

    Function Fast_Determinate( Var Matrix: Matrix_Type ): Matrix_Number;
      Var
        Size: Byte;
        Value,
        Number,
        Smallest,
        Determinate_Result: Matrix_Number;
        Small_Matrix: Matrix_Pointer;
      Begin
        If ( Matrix.Row_Size > Matrix.Column_Size )
          then
            Write_Error( 201, 'Fast_Determinate: Row_Size is greater than Column_Size' );
        Size := Matrix.Row_Size;
        If ( Size < 3 )
          then
            Fast_Determinate := Default_Determinate( Matrix )
          else
            Begin
              Number := 1.0;
              Prepare( Number, Matrix );
              If ( Get_Matrix( Matrix, 1, 1 ) <> 0.0 )
                then
                  Begin
                    Small_Matrix := Reduced_Matrix( Matrix, Size, Smallest );
                    While ( Abs( Smallest ) > 1000 ) do
                      Originate( Small_Matrix^, Smallest, Number );
                    Value := Power( ( 1 / Get_Matrix( Matrix, 1, 1 ) ), ( Size - 2 ) );
                    Determinate_Result := Fast_Determinate( Small_Matrix^ );
                    Fast_Determinate := ( Determinate_Result * Number * Value );
                    Deallocate_Matrix( Pointer( Small_Matrix ) );
                  End
                else
                  Fast_Determinate := ( Number * Slow_Determinate_Fast( Matrix ) );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Determinate fast internal.
    As previously defined.

*************************************************)

    Function Determinate_Fast_Internal( Var Matrix: Matrix_Type ): Matrix_Number;
      Var
        New_Matrix: Matrix_Pointer;
      Begin
        New_Matrix := Allocate_Matrix( Matrix.Row_Size, Matrix.Column_Size );
        If Copy_Matrix_Internal( Matrix, New_Matrix^ )
          then
            Determinate_Fast_Internal := Fast_Determinate( New_Matrix^ );
        Deallocate_Matrix( Pointer( New_Matrix ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Solve the matrix.
    This function solves the matrix using the
    matrix determinate method.

*************************************************)

    Function Solve_The_Matrix( Var Matrix: Matrix_Type ): Boolean;
      Var
        Size,
        Point,
        Row_Index,
        Column_Index: Byte;
        Numerator: Array[ 1 .. Max_Matrix_Size ] of Matrix_Number;
        Denominator: Matrix_Number;
        Second_Matrix: Matrix_Pointer;
      Begin
        If ( Succ( Matrix.Row_Size ) = Matrix.Column_Size )
          then
            Begin
              Size := Matrix.Row_Size;
              Denominator := Determinate_Fast_Internal( Matrix );
              Second_Matrix := Allocate_Matrix( Size, Size );
              If ( Second_Matrix = Nil )
                then
                   Write_Error( 203, 'Slow_Determinate: Out of heap space' );
              For Point := 1 to Size do
                Begin
                  For Row_Index := 1 to Size do
                    Begin
                      For Column_Index := 1 to Pred( Point ) do
                        Put_Matrix( Second_Matrix^, Row_Index, Column_Index, Get_Matrix( Matrix, Row_Index, Column_Index ) );
                      For Column_Index := Point to Size do
                        Put_Matrix( Second_Matrix^, Row_Index, Column_Index, Get_Matrix( Matrix, Row_Index,
                                                                                         Succ( Column_Index ) ) );
                    End;
                  If Odd( Point )
                    then
                      Numerator[ Point ] := ( Determinate_Fast_Internal( Second_Matrix^ ) / Denominator )
                    else
                      Numerator[ Point ] := - ( Determinate_Fast_Internal( Second_Matrix^ ) / Denominator );
                  If Odd( Size )
                    then
                      Numerator[ Point ] := - Numerator[ Point ];
                End;
              Deallocate_Matrix( Pointer( Second_Matrix ) );
              For Point := 1 to Size do
                Put_Matrix( Matrix, 1, Point, Numerator[ Point ] );
              Solve_The_Matrix := True;
            End
          else
            Solve_The_Matrix := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Determinate_Fast.
    As previously defined.

*************************************************)

    Function Determinate_Fast( Var Matrix: Pointer ): Matrix_Number;
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Write_Error( 204, 'Determinate_Fast: Variable not initialize' );
        Determinate_Fast := Determinate_Fast_Internal( The_Matrix^ );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Determinate_Slow.
    As previously defined.

*************************************************)

    Function Determinate_Slow( Var Matrix: Pointer ): Matrix_Number;
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Write_Error( 204, 'Determinate_Slow: Variable not initialize' );
        Determinate_Slow := Slow_Determinate( The_Matrix^ );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read in the matrix.
    As previously defined.

*************************************************)

    Procedure Read_In_Matrix( Var InFile: Text; Var Matrix: Pointer );
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Write_Error( 204, 'Read_In_Matrix: Variable not initialize' );
        Read_In_Matrix_Internal( InFile, The_Matrix^ );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Write out the matrix.
    As previously defined.

*************************************************)

    Procedure Write_Out_Matrix( Var OutFile: Text; Var Matrix: Pointer );
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Write_Error( 204, 'Write_Out_Matrix: Variable not initialize' );
        Write_Out_Matrix_Internal( OutFile, The_Matrix^ );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Copy the matrix.
    As previously defined.

*************************************************)

    Function Copy_Matrix( Var Operand, Result: Pointer ): Boolean;
      Var
        The_Operand: Matrix_Pointer absolute Operand;
        The_Result: Matrix_Pointer absolute Result;
      Begin
        If ( ( Operand = Nil ) or ( Result = Nil ) )
          then
            Write_Error( 204, 'Copy_Matrix: Variable not initialize' );
        Copy_Matrix := Copy_Matrix_Internal( The_Operand^, The_Result^ );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Generate matrix internal.
    This procedure generates a matrix.

*************************************************)

    Procedure Generate_Matrix_Internal( Var Matrix: Matrix_Type );
      Var
        Row_Index,
        Column_Index: Byte;
        Value: Matrix_Number;
      Begin
        For Row_Index := 1 to Matrix.Row_Size do
          For Column_Index := 1 to Matrix.Column_Size do
            Begin
              Value := -1000 + Random( 2000 );
              { Turning the above calculation the other way will, for some
                very strange reason, only product incorrect results }
              Put_Matrix( Matrix, Row_Index, Column_Index, Value );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Generate the matrix.
    As previously defined.

*************************************************)

    Procedure Generate_Matrix( Var Matrix: Pointer );
      Var
        The_Matrix: Matrix_Pointer absolute Matrix;
      Begin
        If ( Matrix = Nil )
          then
            Write_Error( 204, 'Generate_Matrix: Variable not initialize' );
        Generate_Matrix_Internal( The_Matrix^ );
      End;

{-----------------------------------------------------------------------------}

  End.

