{ SLLLIB.PAS : Single Linked List object library

  Title    : SLLLIB
  Version  : 2.1
  Date     : Nov 14,1996
  Author   : J R Ferguson
  Language : Borland Pascal v7.0 (all targets)
  Usage    : Unit
}

UNIT SllLib;

INTERFACE
Uses ObjLib;


{ --- Private definitions --- }

type
  P_Sll_Node = ^T_Sll_Node;
  T_Sll_Node = record
                 Item: Pointer;
                 Next: P_Sll_Node;
               end;


{ --- Public definitions --- }

type
  P_Sll_List    = ^T_Sll_List;
  P_Sll_Stack   = ^T_Sll_Stack;
  P_Sll_OrdList = ^T_Sll_OrdList;

  T_Sll_List    = object(T_Object)
  { Single linked list, acting as a First In First Out queue. }
    Head,Tail   : P_Sll_Node;
    constructor Init;
    { Create an empty list.
      Override: sometimes. }
    destructor  Done; virtual;
    { Discard list after deleting and disposing of all nodes in it by
      calling DeleteAll.
      Override: sometimes. }
    procedure   FreeItem(V_Item: Pointer); virtual;
    { Disposes of the specified item, freeing any memory allocated by it.
      The default FreeItem does nothing. In most cases you will override
      this method. When V_Item is an object pointer of type P_Item, the
      code could be:
        Dispose(P_Item(V_Item),Done);
      FreeItem is calles by DeleteAll for each node item that is not nil,
      before that node itself is deallocated. It should never be called
      directly, because that would leave a node with an invalid Item
      pointer in the list.
      Override: often. }
    procedure   DeleteAll; virtual;
    { Delete and dispose of all nodes in the list. Before a node is
      deleted, FreeItem is called with its Item pointer as a parameter,
      unless the Item pointer is nil.
      The result is an empty list.
      Override: seldom. }
    function    Empty: boolean; virtual;
    { Test if the list is empty.
      Override: never. }
    procedure   Insert(V_Item: Pointer); virtual;
    { Insert V_Item into the list as the last element. }
    function    Retrieve: Pointer; virtual;
    { Retrieve a pointer to the first item in the list, and delete the
      node containing that item from the list, without disposing of the
      Item.
      Returns nil if the list is empty.
      Override: never. }
    procedure   ForEach(V_ProcessProc: T_ProcessProc); virtual;
    { Process all items with the specified Procedure in the order they were
      inserted.
      Override: never.
    }
  end;


  T_Sll_Stack   = object(T_Sll_List)
  { Last In First Out Stack. }
    procedure   Insert(V_Item: Pointer); virtual;
    { Insert V_Item into the list as the first element. }
  end;

  T_Sll_OrdList = object(T_Sll_List)
  { Ordered Linked List. }
    CompareFunc : T_CompareFunc;
    constructor Init(V_CompareFunc: T_CompareFunc);
    { Create new (empty) list, setting the compare function on which to
      base its ordering. }
    procedure   Insert(V_Item: Pointer); virtual;
    { Insert V_Item into the list at a position that keeps the list in a
      non-descending order, based on CompareFunc. }
    procedure   Merge(V_OrdList: P_Sll_OrdList); virtual;
    { Merge V_OrdList into the list, deleting (but not disposing of) all
      items from V_OrdList. }
  end;


IMPLEMENTATION


{ --- T_Sll_List methods --- }

constructor T_Sll_List.Init;
begin inherited Init; Head:= nil; Tail:= nil end;

destructor  T_Sll_List.Done;
begin DeleteAll; inherited Done; end;

procedure   T_Sll_List.FreeItem(V_Item: Pointer);
begin end;

procedure   T_Sll_List.DeleteAll;
var Item: Pointer;
begin while not Empty do FreeItem(Retrieve); end;

function    T_Sll_List.Empty: boolean;
begin Empty:= Head=nil end;

procedure   T_Sll_List.Insert(V_Item: Pointer);
var elm: P_Sll_Node;
begin
  new(elm); elm^.Item:= V_Item; elm^.Next:= nil;
  if Empty then Head:= elm else Tail^.Next:= elm;
  Tail:= elm;
end;

function    T_Sll_List.Retrieve: Pointer;
var h: P_Sll_Node; item: Pointer;
begin
  if Empty then Retrieve:= nil
  else begin
    item:= Head^.Item; if Head=Tail then Tail:= nil;
    h:= Head; Head:= Head^.Next; dispose(h);
    Retrieve:= item;
  end;
end;

procedure   T_Sll_List.ForEach(V_ProcessProc: T_ProcessProc);
var elm: P_Sll_Node;
begin if @V_ProcessProc <> nil then begin
  elm:= Head;
  while elm <> nil do begin
    V_ProcessProc(elm^.Item);
    elm:= elm^.Next;
  end;
end end;


{ --- T_Sll_Stack methods --- }

procedure   T_Sll_Stack.Insert(V_Item: Pointer);
var elm: P_Sll_Node;
begin
  new(elm); elm^.Item:= V_Item; elm^.Next:= Head;
  Head:= elm; if Tail = nil then Tail:= elm;
end;


{ --- T_Sll_OrdList methods --- }


constructor T_Sll_OrdList.Init(V_CompareFunc: T_CompareFunc);
begin inherited Init; CompareFunc:= V_CompareFunc; end;

procedure   T_Sll_OrdList.Insert(V_Item: Pointer);
var p,elm: P_Sll_Node;
begin
  new(elm); elm^.Item:= V_Item;
  if Empty then begin Head:= elm; Tail:= elm; end
  else begin
    Tail^.Next:= elm; elm^.Next:= Head; p:= elm;
    while CompareFunc(p^.Next^.Item,V_Item) < 0 do p:= p^.Next;
    if      p=elm  then Head:= elm
    else if p=Tail then Tail:= elm
    else                begin elm^.Next:= p^.Next; p^.Next:= elm end;
  end;
  Tail^.Next:= nil;
end;

procedure   T_Sll_OrdList.Merge(V_OrdList: P_Sll_OrdList);
begin while not V_OrdList^.Empty do Insert(V_OrdList^.Retrieve); end;


END.
