/* TCAP.C: implementation of the C/C++ TCAP API.
   Copyright(c) 1998 Tim Kannel

   See API.TXT for information about the API,
   See TCAP.H for function descriptions.

   Use of this source code in a commercial or shareware program,
   with or without modification, requires permission from the author.
   Permission to use this source code in freeware programs, with or without
   modification, is granted.  Distribution of a modified version of this
   source code requires permission from the author, regardless of the
   program distribution type.
*/
#include <dos.h>
#include <mem.h>
#include <time.h>
#include "tcap.h"

/* Function Numbers */
#define MPLEX_INST       0           /* installation check */
#define MPLEX_VER        1           /* get version */
#define MPLEX_GETCAPF    2           /* get capture flag */
#define MPLEX_SETCAPF    3           /* set capture flag */
#define MPLEX_BYTES      4           /* bytes in buffer */
#define MPLEX_FLUSH      5           /* attempt flush */
#define MPLEX_OVERFLOW   6           /* get overflow status */

/* Public Data */
char _tcapID=0;

/* Private Data */
static char tcap_sig[] = { "\0\1\2TCAP\2\1\0" };

/* Functions */

char tcapGetID(int* API)
{
  union REGS regs;
  struct SREGS sregs;
  int id;
  int installed = 0;

  for (id=128; id < 256; id++) {
     regs.h.ah = (char)id;
     regs.h.al = MPLEX_INST;
     int86x(0x2F, &regs, &regs, &sregs);

     if (regs.h.al == 0xFF)
       if (_fmemcmp(tcap_sig, MK_FP(sregs.es, regs.x.di),10)==0) {
          installed = 1;
          break;
       }
  }

  if (installed) {
    _tcapID = (char)id;
    if (API) {
      if (regs.h.ah==0)
        *API = 1;
      else
        *API = 0;
    }
  }
  else {
    _tcapID = 0;
    if (API)
      *API = 0;
  }

  return _tcapID;
}

void tcapGetVersion(char id, int* major, int* minor, int* reserved)
{
  union REGS regs;
  regs.h.ah = id;
  regs.h.al = MPLEX_VER;
  int86(0x2F, &regs, &regs);
  if (major)
    *major = regs.h.ah;
  if (minor)
    *minor = regs.h.al;
  if (reserved)
    *reserved = regs.x.bx;
}

unsigned short tcapGetCaptureFlag(char id)
{
  union REGS regs;
  regs.h.ah = id;
  regs.h.al = MPLEX_GETCAPF;
  int86(0x2F, &regs, &regs);
  return regs.x.ax;
}

void tcapSetCaptureFlag(char id, unsigned short flag)
{
  union REGS regs;
  regs.h.ah = id;
  regs.h.al = MPLEX_SETCAPF;
  regs.x.bx = flag;
  int86(0x2F, &regs, &regs);
}

unsigned tcapGetBufferBytes(char id)
{
  union REGS regs;
  regs.h.ah = id;
  regs.h.al = MPLEX_BYTES;
  int86(0x2F, &regs, &regs);
  return regs.x.ax;
}

unsigned tcapFlush(char id)
{
  union REGS regs;
  regs.h.ah = id;
  regs.h.al = MPLEX_FLUSH;
  int86(0x2F, &regs, &regs);
  return regs.x.ax;
}

int tcapWaitForFlush(char id, int seconds)
{
  clock_t canceltime,current_time;
  int flushed;

  if (seconds <= 0)
    return tcapFlush(id);

  flushed = 0;
  canceltime = clock() + (clock_t)(seconds * CLK_TCK);

  while (!flushed)
  {
    current_time = clock();
    if (current_time >= canceltime)
       break;
    else
    if (current_time == -1 )
       break;
    if (tcapFlush(id)==0)
       flushed = 1;
  }

  return !flushed;
}

int tcapGetOverflowStatus(char id, int* last_cmd_ovr)
{
  union REGS regs;
  regs.h.ah = id;
  regs.h.al = MPLEX_OVERFLOW;
  int86(0x2F, &regs, &regs);
  if (last_cmd_ovr)
     *last_cmd_ovr = regs.x.bx;
  return regs.x.ax;
}
