/*
 *  Routines for sampling from Pro-Audio Spectrum 16-bit soundcards
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "specgram.h"

#ifdef SC_PAS16

#include <stdlib.h>
#include <stdio.h>
#include <mem.h>
#include <math.h>
#include <string.h>
#include <fcntl.h>
#include <io.h>
#include <pcmio.h>        // Standard PCM record functions
#include <state.h>        // Need this for Mixer code
#include <binary.h>       // Hardware values

#include "display.h"
#include "extern.h"       // Global variable definitions

/* Function prototypes */
void reset_pas16(void);
void halt_pas16(void);
void cleanup_pas16(void);
void recordblock_pas16(void *);
void set_mixer_pas16(int mix,int level);
int GetOutputMixer(char *name,int channel);
int GetInputMixer(char *name,int channel);
void SetOutputMixer(char *name,int level);
void SetInputMixer(char *name,int level);
void CalibrateMixer(void);

// Use default DMA and IRQ values, 4k DMA buffer divided into 4 parts
#define DMA          -1
#define IRQ          -1
#define DMASize       8
#define DMADivisions  8
#define Compression   0
#define SampleSize   16

int pas_LeftChannel;        // Variables for saving/restoring mixer values
int pas_RightChannel;
int pas_FilterVal;
int mv;           /* Handle for the mixer device */
double mixer_offset=0,mixer_slope=1;

int pas_record_buffer;
short *pas_buffer[2];

// Turn off stack checking for this routine
#pragma option -N-
/*
 *  Callback function.  This function is called every time a buffer has
 *  been filled.  It sets a flag so the main loop recognises and processes
 *  the buffer.
 */
void far callback1()
{
   buffer_full=1;
}
// Restore stack checking to the command-line specified state
#pragma option -N.

void init_pas16(void)
{
   /* Allocate buffers for the sampling */
   pas_buffer[0]=(short *)malloc(MAX_LEN*(stereo+1)*sizeof(short));
   pas_buffer[1]=(short *)malloc(MAX_LEN*(stereo+1)*sizeof(short));

   if(!pas_buffer[0] || !pas_buffer[1])
   {
      puts("Error allocating memory in init_pas16()");
      exit(1);
   }

   reset_soundcard=reset_pas16;
   halt_soundcard=halt_pas16;
   cleanup_soundcard=cleanup_pas16;
   recordblock=recordblock_pas16;
   set_mixer=set_mixer_pas16;
   sample_size=16;

   /*
    *  Initialize link to Mixer control routines
    *  Then save current mixer settings and turn off the PCM output
    *  mixer to stop feedback and clicking noises.
    */
   DOUT("PAS: Initializing mixer connection");
   if ((mv = open ("MVPROAS",O_RDWR )) == -1)
   {
      mixers=0;
      puts("PAS: Unable to establish mixer connection.");
      puts("Press any key to continue ...");
      getch();
   }
   else
   {
      int in,out;
      long sumx=0,sumy=0,sumxy=0,sumxx=0,sumyy=0,count=0;

      mixers=1;
      DOUT("PAS: Getting mixer values");
      mic_level=GetInputMixer("MIC",1);
      ext_level=GetInputMixer("EXT",1);
      int_level=GetInputMixer("INT",1);
      if(mic_level>100) mic_level=100;
      if(ext_level>100) ext_level=100;
      if(int_level>100) int_level=100;
      if(mic_level<0) mic_level=0;
      if(ext_level<0) ext_level=0;
      if(int_level<0) int_level=0;
      pas_LeftChannel=GetInputMixer("PCM",1);
      pas_RightChannel=GetInputMixer("PCM",2);

      /* Get proportionality factor for mixer settings */
      for(in=1,out=0;(in<100) && (out<100);in++)
      {
         SetOutputMixer("PCM",in);
         out=GetOutputMixer("PCM",1);
         count++;
         sumy+=in;
         sumx+=out;
         sumxy+=in*out;
         sumyy+=in*in;
         sumxx+=out*out;
      }
      mixer_offset=(double)(sumy*sumxx-sumx*sumxy)/(double)(count*sumxx-sumx*sumx);
      mixer_slope=(double)(sumxy-mixer_offset*sumx)/(double)sumxx;
      SetOutputMixer("PCM",0);
   }
}

void reset_pas16(void)
{
   int i;
   if(OpenPCMBuffering(DMA,IRQ,DMASize,DMADivisions)!=0)
   {
      cleanup_pas16();
      draw_cleanup_graphics();
      puts("Error trying to open PCM buffering.");
      exit(1);
   }
   /* Round sampling rate to a valid value for the PAS card */
   DOUT("PAS: Setting PCM state");
   i=floor(1193180.0/SampleRate+0.5);
   if(i<1) i=1;
   SampleRate=1193180L/i;
   if(PCMState(SampleRate,stereo,Compression,SampleSize)!=0)
   {
      cleanup_pas16();
      draw_cleanup_graphics();
      puts("Error setting sample rate.");
      exit(1);
   }
   DOUT("PAS: Setting up the buffers");
   pas_record_buffer=0;    /* Pointer to the buffer being filled */
   buffer_full=0;          /* Flag indicating that a buffer has been filled */

   /* This function starts the DMA process. */
   DOUT("Starting the recording process");
   RecordThisBlock((char *)pas_buffer[0],fftlen*2*(stereo+1),callback1);
   RecordThisBlock((char *)pas_buffer[1],fftlen*2*(stereo+1),callback1);
}


void halt_pas16(void)
{
   /* Shut down the DMA system. */
   DOUT("PAS: Closing the PCM buffering");
   ClosePCMBuffering();
}

void cleanup_pas16(void)
{
   /* Shut down the DMA system. */
   DOUT("PAS: Closing the PCM buffering");
   ClosePCMBuffering();

   if(mixers)
   {
      /* Restore mixers to their original value. */
      DOUT("PAS: Restoring mixer levels");
      SetOutputMixer("LEFT PCM",pas_LeftChannel);
      SetOutputMixer("RIGHT PCM",pas_RightChannel);

      close(mv);
   }
}

void recordblock_pas16(void *buffer)
{
   while(!buffer_full && !done);
   buffer_full=0;
   memcpy(buffer,pas_buffer[pas_record_buffer],fftlen*2*(stereo+1));
   /* Re-queue the block up for the next pass */
   RecordThisBlock((char *)pas_buffer[pas_record_buffer],fftlen*2*(stereo+1),callback1);
   pas_record_buffer=1-pas_record_buffer;
}

void set_mixer_pas16(int mix,int level)
{
   /* Set a mixer level on the PAS16 card */
   if(mix==MIXER_EXT)
      SetInputMixer("EXT",level);
   else if(mix==MIXER_INT)
      SetInputMixer("INT",level);
   else if(mix==MIXER_MIC)
      SetInputMixer("MIC",level);
}

void SetInputMixer(char *name,int level)
{
   int i;
   char buffer[80];

   /* Scale the values to fit the actual readings */
   level=(int)floor(level*mixer_slope+mixer_offset+0.5);
   if(level<0) level=0;
   if(level>100) level=100;

   DOUT("Setting input mixer level");
   sprintf(buffer,"SET INPUT MIXER %s TO %d\n",name,level);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
}

void SetOutputMixer(char *name,int level)
{
   int i;
   char buffer[80];

   /* Scale the values to fit the actual readings */
   level=(int)floor(level*mixer_slope+mixer_offset+0.5);
   if(level<0) level=0;
   if(level>100) level=100;

   DOUT("Setting output mixer level");
   sprintf(buffer,"SET OUTPUT MIXER %s TO %d\n",name,level);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
}

int GetInputMixer(char *name,int channel)
{
   char buffer[80];
   int left=0,right=0,i,index;

   DOUT("Reading input mixer level");
   sprintf(buffer,"GET INPUT MIXER %s\n",name);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
   index=0;
   for(i=0;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      left=atoi(buffer+index);
      if(left<0) left=0;
      if(left>100) left=100;
      i++;
   }
   /*sw1=buffer[i];    +/- flag for mixer active */
   i+=2;
   index=i;
   for(;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      right=atoi(buffer+index);
      if(right<0) right=0;
      if(right>100) right=100;
      i++;
   }
   if(channel==1)
      return left;
   else
      return right;
   /*sw1=buffer[i];    +/- flag for mixer active */
}

int GetOutputMixer(char *name,int channel)
{
   char buffer[80];
   int left=0,right=0,i,index;

   DOUT("Reading output mixer level");
   sprintf(buffer,"GET OUTPUT MIXER %s\n",name);
   write(mv,buffer,strlen(buffer));
   read (mv,buffer,80);
   for(i=0;(buffer[i]!='\n') && (i<80);i++);
   buffer[i]=0;
   DOUT(buffer);
   index=0;
   for(i=0;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      left=atoi(buffer+index);
      if(left<0) left=0;
      if(left>100) left=100;
      i++;
   }
   /* sw1=buffer[i];    +/- flag for mixer active */
   i+=2;
   index=i;
   for(;buffer[i] && buffer[i]!='\t';i++);
   if(buffer[i]=='\t')
   {
      buffer[i]=0;
      right=atoi(buffer+index);
      if(right<0) right=0;
      if(right>100) right=100;
      i++;
   }
   /* sw2=buffer[i];    +/- flag for mixer active */
   if(channel==1)
      return left;
   else
      return right;
}

#endif

