/***********************************************************************
    DOWGJ2 -  Day_Of_the_Week Gregorian or Julian calendar
************************************************************************

  Author: Roberto R. Viscogliosi
  Date Written : 10/92
  Release: 1.1a

  Modified by Douglas Boling

  Program's Purpose:
      This program computes the Day_Of_the_Week for a given date
      using the Gregorian calendar as default, or the Julian calendar
      when an extra argument is given at the command line.

  Input Arguments:
      mm dd yyyy [c]
      where:
        mm   = month
        dd   = day
        yyyy = year
        [c]  = any value forces the use of the Julian calendar;
               no value uses the Gregorian calendar.

  Example:
      The input, "DOWGJ2  10 12 1492 j" shows that Columbus discovered
      the New World on a Friday (Julian calendar).

  Returns:
      0    OK
      1    Invalid calling syntax/Invalid Input Data

   Last change:  14 Jan 93       2:34 pm
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>

#define  OK   0
#define  GRG  1

long dow (int, int, int, int);
void help (char *);

//====================================================================
// MAIN process
//====================================================================
int main (ac, av)
int ac;
char *av[];
{
   int day, month, year, calendar = GRG, ret = OK;

   static char *dowstr[7] = {
      "Sunday", "Monday", "Tuesday", "Wednesday",  
      "Thursday", "Friday", "Saturday"
   };

   for (;;) {
       // check arguments passed
       if  (ac < 4) {         // not enough args. 
           ret = !OK;
           break;
       }
       // get/verify the input values
       if  (((month = atoi (av[1])) < 1) ||
            ((day   = atoi (av[2])) < 1) ||
            ((year  = atoi (av[3])) < 1)   
           ) {                // invalid values 
           ret = !OK;
           break;
       }
       // check if Julian calendar requested
       if  (ac > 4) {         // reset the calendar indicator 
           calendar = !GRG;
       }
       // prevent overflow by check year size
       if  (year > 9999) {    // year too large
           ret = !OK;
           break;
       }
       // compute and show the day_of_the_week
       printf ("%02d/%02d/%04d is on %s %s\n",    
                month, day, year,                 
                dowstr [dow (month, day, year, calendar)],    
                (calendar == GRG) ? "" : "(Julian calendar)"  
              );
       break;
   }
   // check for errors
   if  (ret) {
       help (av[0]);
   }
   exit (ret);
}

//======================================================================
// DOW - Return the day of the week value 0-6 (0 = Sunday)
//          based on the date (month, day, year) and calendar
//======================================================================
long dow (month, day, year, calendar)
int       month, day, year, calendar;
{
   static char month_val[12] = {
     // J F M A M J J A S O N D
	6,2,2,5,0,3,5,1,4,6,2,4
   };
   long DayOfWeek;

   DayOfWeek = day + month_val[month - 1];

   if (month < 3) {
      if (calendar == GRG) {
	 if (!(year % 4) && (year % 100) || !(year % 400))
	    DayOfWeek--;
      } else
	 if (!year %4)
	    DayOfWeek--;
   }

   DayOfWeek += year + (year / 4);
   if (calendar == GRG)
      DayOfWeek += (year / 400) - (year / 100);
   else
      DayOfWeek += 5;

   DayOfWeek %= 7;
   return DayOfWeek;
}

//====================================================================
// HELP - Print the help info
//====================================================================
void help (fn)
char      *fn;
{
 printf ("\nWelcome to DOWGJ2 (DOW Gregorian or Julian) Version 1.1\n");
 printf ("                                     by R. Viscogliosi\n");
 printf ("                            modified by Douglas Boling\n");
 printf ("\nTo use it, please enter:\n");
 printf ("\n  %s  mm dd yyyy [c]\n", fn);
 printf ("\n  (If [c] is specified then the date is in Julian,\n");
 printf ("   else the date defaults to the Gregorian calendar)\n");
 printf ("\n  (e.g. 'DOWGJ2  10 12 1492 j' was on Friday (Julian))\n");
}


