//-------------------------------------------------------------------//

// Synopsis:	print and plot section info

// Syntax:	show_prop ( P , s, coord )

// Description:

//	The properties of polygonal sections may be calculated using rfile
//  section.  These section properties can be shown on the screen by 
//  this rfile.    
//  
//  The (x,y) coordinates of the vertices of the polygon are stored 
//  in P.  P is a two column matrix,  its first column stores xs and
//  its second column stores ys.   The coordinates of the vertces must
//  be stored sequentially for a complete clockwise path around the 
//  polygon.

//  s stores section properties (returned from section.r)

//  If coord is "polar", then the coordinates in P are in polar 
//  coordinates (r,theta), where theta is in degree.
//
// Dependencies
//   rfile plplot.r
//
//  Tzong-Shuoh Yang  8/10/94   (tsyang@ce.berkeley.edu)
//-------------------------------------------------------------------//

show_prop = function(P,s,coord)
{
   local(P,mx,mn,pp,x,y,xp,yp,R,t,T,ps)
   global (pi)

   // if the polygon is not closed, let the last point = 1st point
   if (P[P.nr;1] != P[1;1] || P[P.nr;2] != P[1;2] )
   {
       P = [P;P[1;1],P[1;2]];
   }
   // convert polar coordinates to cartesian coordinates
   if (exist(coord)) 
   {  if (coord == "polar")
      {
         P[;2] = P[;2]*(pi/180);
         P = [P[;1].*cos(P[;2]),P[;1].*sin(P[;2])];
      }
   }
   
   if (s.area < 0) 
   {
     printf(" Area = %g (hole ?)\n\n",s.area);
   else
     printf(" Area = %g\n\n",s.area);
   }
   printf(" Centroid = (%g,%g)\n\n",s.cgx,s.cgy);
   printf(" Moment of Inertia: (Original axis)\n");
   printf("                    Ixx = %g\n",s.Ixx);
   printf("                    Iyy = %g\n",s.Iyy);
   printf("                    Ixy = %g\n\n",s.Ixy);
   printf(" Moment of Inertia: (Centroid axis)\n");
   printf("                    Ixx = %g\n",s.Ixx0);
   printf("                    Iyy = %g\n",s.Iyy0);
   printf("                    Ixy = %g\n\n",s.Ixy0);
   printf(" Moment of Inertia: (Principal axis)\n");
   printf("                    Ixx = %g\n",s.pIxx);
   printf("                    Iyy = %g\n",s.pIyy);
   printf("                  angle = %g (deg)\n\n",s.angle);
   printf(" Polar Moment of Inertia:\n");
   printf("                      J = %g\n\n",s.J0);
   sprintf(ps,"Properties: A=%.3g I#+XX#+=%.3g I#+YY#+=%.3g I#+XY#+=%.3g",...
           s.area,s.Ixx0,s.Iyy0,s.Ixy0);
   
   mx = max(P);
   mn = min(P);
   pp = abs(mx-mn)*0.2;
   plimits(mn[1;1]-pp[1;1],mx[1;1]+pp[1;1],mn[1;2]-pp[1;2],mx[1;2]+pp[1;2]);
   // plaspect(1);
   // plgrid("abcnst","abcnst");
   ptitle(ps);
   xlabel("X");
   ylabel("Y");
   plegend();
   x = [mx[1;1]+pp[1;1]/2,s.cgy;mn[1;1]-pp[1;1]/2,s.cgy];
   y = [s.cgx,mx[1;2]+pp[1;2]/2;s.cgx,mn[1;2]-pp[1;2]/2];
   t = s.angle*pi/180;
   R = [cos(t),sin(t);-sin(t),cos(t)];
   T = [s.cgx,s.cgy;s.cgx,s.cgy];
   xp = (x-T)*R+T;
   yp = (y-T)*R+T;
   if (abs(s.angle) > 0.001) {
      plot(<<P;x;y;xp;yp>>);   
   else
      plot(<<P;x;y>>);
   }
   sprintf(ps," CG(%.3g,%.3g)",s.cgx,s.cgy);
   plptex(ps,s.cgx,s.cgy+pp[1;2]/5,1,0,0);
   plptex(" #+Y",s.cgx,mx[1;2]+pp[1;2]/2,1,0,0);
   plptex(" #+X",mx[1;1]+pp[1;1]/2,s.cgy,1,0,0); 
   if (abs(s.angle) > 0.001) {
      plptex(" #+Y#+'",yp[1;1],yp[1;2],1,0,0);
      plptex(" #+X#+'",xp[1;1],xp[1;2],1,0,0); 
      sprintf(ps," #gh=%.3g#uo",s.angle);
      plptex(ps,s.cgx,s.cgy+pp[1;2],1,0,0);   
   }
};
