/* $ vflat.cpp 12/07/99 21:03 $
 * Virtual Flat Frame Buffer. Revision 1.1
 *
 * Copyright (C) 1999  Dmitry Uvarov <mit@pc2o205a.pnpi.spb.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Revision history:
 *    1.0 - 1.1  - Integrated VFlat Software Edition (SE) version.
 */
#include <mem.h>
#include <stdlib.h>
#include <conio.h>
#include <stdio.h>
#include <assert.h>
#include "..\include\portio.h"
#include "..\include\vesa.h"
#include "..\include\pcibios.h"
#include "..\include\pmode.h"
#include "..\include\vflat.h"

#ifdef __VF__DEBUG__
  #define debug_msg(msg)  printf(msg)
#else
  #define debug_msg(msg)
#endif

#define  PCI_VENDOR_ID_ACER    0x1025  /* is this right code? */
#define  SOFTWARE_LFB_BASE     0xF0000000  /* software lfb base linear address */

typedef void (*lfb_init_proc)();

static dword lfb_base;   /* physical LFB base address  */
static dword lin_base;   /* mapped linear base address */
static word  vendor_id;
static word  device_id;

#include "vfchips.cpp"   /* VFlat chipsets database */

static lfb_init_proc InitProc = NULL; /* Pointer to LFB initialization
                                       * procedure for current chipset
                                       */
static word  mode_set = 0;            /* current mode number */
static word  prev_mode = 0;
static word  maptype = 0;                   /* base address mapping type */

#ifdef __VF__SE__
 /* VFlat Software Edition */
 static void* BankCode;           /* Bank switch code for current chipset */
 static void* HandlerCode;        /* builded exeption handler             */
#endif

/* procedures from software emulation library vfasm.asm */

extern "C" {
#ifdef __VF__SE__
  /* External procedures for VFlat Software Edition */
  int  __cdecl InitVFlat_SE();
  void __cdecl DoneVFlat_SE();
  void __cdecl InitPageTable_SE(dword);
  void __cdecl DonePageTable_SE();
  void __cdecl InstallVFHandler_SE(dword,void*);
  void __cdecl RemoveVFHandler_SE();

  void __cdecl HND_Int10();
  void __cdecl HND_S3();
  void __cdecl HND_Trident();
  void __cdecl HND_TSeng();
  void __cdecl HND_SiS();
  void __cdecl HND_Cirrus();
  void __cdecl HND_ATI();
  void __cdecl HND_ARK();

  dword __cdecl GetHandlerSize(void *driver, dword banksize);
  void  __cdecl BuildHandler(void *driver, dword banksize, void *dst);

  #define InitVFlat        InitVFlat_SE
  #define DoneVFlat        DoneVFlat_SE
  #define InitPageTable    InitPageTable_SE
  #define DonePageTable    DonePageTable_SE
  #define InstallVFHandler InstallVFHandler_SE
  #define RemoveVFHandler  RemoveVFHandler_SE
#else
  int  __cdecl InitVFlat();
  void __cdecl DoneVFlat();
  void __cdecl InitPageTable(dword);
  void __cdecl DonePageTable();
  void __cdecl InstallVFHandler(dword);
  void __cdecl RemoveVFHandler();
#endif
}


byte GetMode();
#pragma aux GetMode = \
    " mov ah,0xF "    \
    " int 10h    "    \
    parm nomemory modify nomemory [ax bx] value [al];

void SetMode(byte);
#pragma aux SetMode = \
    " xor ah,ah  "    \
    " int 10h    "    \
    parm nomemory [al] modify nomemory;

/*
 * Status: Internal procedure
 * Search for installed VGA compatible devices on PCI bus.
 * Returns TRUE if successful
 */
static ibool vfFindHardware() {
   byte bus,dev;
   /* scan all pci busses for VGA compatible device */
   for (int i = 0; i<511; i++) {
      bus = (i>>5) & 0xF;
      dev = (i<<3) & 0xF8;
      /* read vendor and device id from device configuration space */
      dword tmp = PCIBIOS_ReadConfigDWord(dev,bus,0);
      /* some broken devices return 0 if slot is empty */
      if (tmp==0 || tmp==0xFFFFFFFF) continue;
          /* we found installed device. Read it's information */
          vendor_id = tmp & 0xFFFF;
          device_id = tmp >> 16;
          /* read device class */
          tmp = PCIBIOS_ReadConfigWord(dev,bus,PCI_CLASS_DEVICE);
          if (tmp==PCI_CLASS_DISPLAY_VGA) {
              /* device is VGA compatible. try to determine video surface */
              /* base address                                             */
              for (int j = 0,tmp = 1; (tmp&1) && (j<6); j++)
              /* if bit 0 is set then the region is in I/O space       */
              /* if clear then it is in memory space. (we need memory) */
                 tmp = PCIBIOS_ReadConfigDWord(dev,bus,PCI_BASE_ADDRESS_0+(j<<2));
              if (tmp && ((tmp&1)==0)) {
                 lfb_base = tmp & 0xFFFFFF00;
                 return TRUE;
              }
          }
   }
   return FALSE;
 }

/*
 * Status: Internal procedure
 * Search for initialization procedure for given chipset vendor id
 * Returns TRUE if successful
 */
#ifdef __VF__SE__
  #define SetBankCode(name)   BankCode = &##name;
#else
  #define SetBankCode(name)
  /* mask out drivers names */
  #define HND_S3
  #define HND_Trident
  #define HND_TSeng
  #define HND_Cirrus
  #define HND_ARK
  #define HND_ATI
  #define HND_SiS
  #define HND_Int10
#endif

static ibool vfFindDriver(word vendor_id) {
   switch (vendor_id) {
      case PCI_VENDOR_ID_S3      : InitProc = &S3_enable;      SetBankCode(HND_S3);      break;
      case PCI_VENDOR_ID_TRIDENT : InitProc = &Trident_enable; SetBankCode(HND_Trident); break;
      case PCI_VENDOR_ID_TSENG   : InitProc = &TSeng_enable;   SetBankCode(HND_TSeng);   break;
      case PCI_VENDOR_ID_CIRRUS  : InitProc = &Cirrus_enable;  SetBankCode(HND_Cirrus);  break;
      case PCI_VENDOR_ID_ATI     : InitProc = &ATI_enable;     SetBankCode(HND_ATI);     break;
           /* here is goes not tested video chipsets */
      case PCI_VENDOR_ID_ARK     : InitProc = &ARK_enable;     SetBankCode(HND_ARK);     break;
      case PCI_VENDOR_ID_COMPAQ  : InitProc = &COMPAQ_enable;  SetBankCode(HND_Int10);   break;
      case PCI_VENDOR_ID_OAK     : InitProc = &OAK_enable;     SetBankCode(HND_Int10);   break;
      case PCI_VENDOR_ID_SIERRA  : InitProc = &Sierra_enable;  SetBankCode(HND_Int10);   break;
      case PCI_VENDOR_ID_ACER    : InitProc = &ACER_enable;    SetBankCode(HND_Int10);   break;
      case PCI_VENDOR_ID_SI      : InitProc = &SiS_enable;     SetBankCode(HND_SiS);     break;

      case PCI_VENDOR_ID_NCR     :
      case PCI_VENDOR_ID_WEITEK  : /* we don't have initialization code for those chipsets */
                                   VBE_setVideoMode(0x03);
                                   printf("unsupported chipset detected! please report to <mit@pc2o205a.pnpi.spb.ru>\n"
                                          "now press any key to continue...\n");
                                   getch();
   }
   if ((vendor_id == PCI_VENDOR_ID_ARK) ||
       (vendor_id == PCI_VENDOR_ID_COMPAQ) ||
       (vendor_id == PCI_VENDOR_ID_OAK) ||
       (vendor_id == PCI_VENDOR_ID_SIERRA) ||
       (vendor_id == PCI_VENDOR_ID_ACER) ||
       Video7_detect()) {
         /* we have untested chipset here */
         VBE_setVideoMode(0x03);
         printf("untested chipset detected! please report to <mit@pc2o205a.pnpi.spb.ru>\n"
                "graphics initialization can fail if you will continue with hardware LFB\n"
                "support.\n"
                "Press S key now to use software emulation, or press any other key to continue\n");
         char ch = getch();
         if ((ch=='s') || (ch=='S')) {
            InitProc = NULL;
         #ifdef __VF__SE__
            BankCode = NULL;
         #endif
            return FALSE;
         }
   }
   if (InitProc) return TRUE;

   /* perform additional checking for all known devices */
   if (Video7_detect()) { InitProc = &Video7_enable; SetBankCode(HND_Int10); }
   return InitProc!=0;
 }

/*
 * Status: Internal procedure
 * Tryes to initiate software emulation
 * Returns TRUE if successful
 */
ibool vfTrySoftware() {
   int res;
   if ((res = InitVFlat())==0) {
         debug_msg("vfInit(): software emulation avaible. Using LFB base at 0xF0000000\n");
         /* set fixed value for LFB base address */
         lin_base = SOFTWARE_LFB_BASE;
         lfb_base = 0;
         maptype = vfSoftware;
         return TRUE;
   }
#ifdef __VF__DEBUG__
     else {
         printf("vfInit(): software emulation failed: %s",
                res==1 ? "ring0 access not avaible\n":"paging mode not avaible\n");
   }
#endif

   return FALSE;
 }

#if defined(__PCIBIOS__DEVICE_LIST__) && defined(__VF__DEBUG__)
  static pci_dev_info UNKNOWN_DEVICE = {0,0,"UNKNOWN",0xFF};
#endif



#ifndef __VF__SE__
/***************************************************************************
 *                                                                         *
 * Procedure: vfInit()                                                     *
 * Params:    forcemode - detection mode to be forced                      *
 *                 can be vfSoftware or vfHardware.                        *
 *                 by default VBE usage forced.                            *
 * Description: Try's to detect avaible hardware configuration(LFB support)*
 *              If hardware fails, try's to initiate internal VFlat        *
 *              software emulation module.                                 *
 *                                                                         *
 **************************************************************************/
int   vfInit(int forcemode) {
   if (maptype) return TRUE; /* check if already initiated */

   if (!VBE_init()) {
      debug_msg("vfInit(): VBE not detected. Initialization failed\n");
      return vferrNoVBE;
   }

   /*
    * if software emulation was forced, try to initiate it first...
    */
   if (forcemode==vfSoftware) {
      debug_msg("vfInit(): trying to force software emulation...\n");
      if (vfTrySoftware()) {
/* register our exit function so it will be called on program termination */
         atexit(vfDone);
         return vferrOK;
      }
   }

   if (VBEVersion >= 0x200) {
   /*
    * VBE with version greater than 2.0 must provide hardware LFB support by
    * itself. So we don't need to initiate external support modules.
    */
      maptype = vfVBESupport;
      if (forcemode==vfVBESupport) {
#ifdef __VF__DEBUG__
         printf("vfInit(): detected VESA version %d.%d. Using hardware LFB provided by VBE\n",
                VBEVersion>>8, VBEVersion & 0xF);
#endif
         /*
          * If autodetection enabled (forcemode == vfVBESupport) then return
          */
         atexit(vfDone);
         return vferrOK;
      }
   }
   /*
    * Try to detect PCI bios presence and find VGA compatible device
    */
   if (PCIBIOS_init()) {
      debug_msg("vfInit(): searching for VGA compatible device on PCI bus...\n");
      if (vfFindHardware()) {
#ifdef __VF__DEBUG__
   #ifdef __PCIBIOS__DEVICE_LIST__
         pci_dev_info *dev_info;
         /* find information table for this device */
         dev_info = PCIBIOS_LookUpDevice(vendor_id,device_id);
         if (!dev_info) {
            UNKNOWN_DEVICE.vendor = vendor_id;
            UNKNOWN_DEVICE.device = device_id;
            dev_info = &UNKNOWN_DEVICE;
         }
     #ifndef __PCIBIOS__VENDOR_LIST__
         printf("vfInit(): detected %s with hardware LFB at 0x%x\n",dev_info->name,lfb_base);
     #else
         char *vendor_name = PCIBIOS_GetStrVendor(vendor_id);
         if (!vendor_name) vendor_name = "UNKNOWN";
         printf("vfInit(): detected %s %s with hardware LFB at 0x%x\n",vendor_name,dev_info->name,lfb_base);
     #endif
   #else
     #ifdef __PCIBIOS__VENDOR_LIST__
         char *vendor_name = PCIBIOS_GetStrVendor(vendor_id);
         if (!vendor_name) vendor_name = "UNKNOWN";
         printf("vfInit(): detected %s video adapter with hardware LFB at 0x%x\n",lfb_base);
     #else
         printf("vfInit(): found hardware LFB at %u\n",lfb_base);
     #endif
   #endif
#endif
         /*
          * Search for initialization driver for this device
          */
         if (!vfFindDriver(vendor_id)) {
            debug_msg("vfInit(): driver for this device not founded. device not supported.\n");
         } else {
            maptype = vfHardware;
            /*
             * map physical address to linear. If mapping failed, we can
             * treat this as global fatal error.
             */
            assert((lin_base = PM_MapPhysicalToLinear(lfb_base,VBEMemory))!=0);

            atexit(vfDone);
            return vferrOK;
         }
      }
  #ifdef __VF__DEBUG__
        else {
         printf("vfInit(): VGA compatible device not founded. Possibly used old ISA device\n");
      }
  #endif
   }
   debug_msg("vfInit(): hardware detection fails. ");
   /*
    * if hardware detection was forced, we ignore VBE hardware support,
    * but if our hardware detection routine fails, we continue with
    * VBE support.
    */
   if (maptype == vfVBESupport) {
      debug_msg("Using hardware LFB support provided by VBE\n");
      atexit(vfDone);
      return vferrOK;
   }

   debug_msg(" Trying software emulation\n");
   /*
    * hardware detection was failed. Try to initiate software emulation
    */
   if (vfTrySoftware()) {
     atexit(vfDone);
     return vferrOK;
   } else {
     debug_msg("vfInit(): could not find supported mapping type. Initialization failed\n");
     return vferrEmulationFailed;
   }
 }

#else

/***************************************************************************
 *                                                                         *
 * Procedure: vfInit()                                                     *
 * Params:    forcemode - detection mode to be forced                      *
 *                 can be vfSoftware or vfHardware.                        *
 *                 by default VBE usage forced.                            *
 * Description: VFlat Software Edition of this function. Really messed up. *
 *                                                                         *
 **************************************************************************/
int   vfInit(int forcemode) {
   if (maptype) return TRUE; /* check if already initiated */

   if (!VBE_init()) {
      debug_msg("vfInit(): VBE not detected. Initialization failed\n");
      return vferrNoVBE;
   }

   ibool VesaFail = VBEVersion<0x200;
   ibool HardwareFail = FALSE;
   /* check flag and then clear it */
   ibool useDrv = forcemode & vfUseStandartDriver;
   forcemode &= ~vfUseStandartDriver;

   if (forcemode==vfVBESupport && !VesaFail) {
#ifdef __VF__DEBUG__
      printf("vfInit(): detected VESA version %d.%d. Using hardware LFB provided by VBE\n",
             VBEVersion>>8, VBEVersion & 0xF);
#endif
_label_TryVBE:
      maptype = vfVBESupport;
      atexit(vfDone);
      return vferrOK;
   }

   /*
    * Detect hardware if hardware or software emulation was forced
    */
   if (PCIBIOS_init()) {
      debug_msg("vfInit(): searching for VGA compatible device on PCI bus...\n");
      if (vfFindHardware()) {
#ifdef __VF__DEBUG__
   #ifdef __PCIBIOS__DEVICE_LIST__
           pci_dev_info *dev_info;
           /* find information table for this device */
           dev_info = PCIBIOS_LookUpDevice(vendor_id,device_id);
           if (!dev_info) {
              UNKNOWN_DEVICE.vendor = vendor_id;
              UNKNOWN_DEVICE.device = device_id;
              dev_info = &UNKNOWN_DEVICE;
           }
      #ifndef __PCIBIOS__VENDOR_LIST__
           printf("vfInit(): detected %s with hardware LFB at 0x%x\n",dev_info->name,lfb_base);
      #else
          char *vendor_name = PCIBIOS_GetStrVendor(vendor_id);
          if (!vendor_name) vendor_name = "UNKNOWN";
          printf("vfInit(): detected %s %s with hardware LFB at 0x%x\n",vendor_name,dev_info->name,lfb_base);
      #endif
   #else
      #ifdef __PCIBIOS__VENDOR_LIST__
          char *vendor_name = PCIBIOS_GetStrVendor(vendor_id);
          if (!vendor_name) vendor_name = "UNKNOWN";
          printf("vfInit(): detected %s video adapter with hardware LFB at 0x%x\n",lfb_base);
      #else
          printf("vfInit(): found hardware LFB at %u\n",lfb_base);
      #endif
   #endif
#endif
           /*
            * Search for initialization driver for this device
            */
           if (!vfFindDriver(vendor_id)) {
              HardwareFail = TRUE;
              if (forcemode==vfSoftware) goto _label_TrySoftware;
              debug_msg("vfInit(): driver for this device not founded. device not supported.\n");
           } else {
              if (forcemode==vfSoftware) {
                /* initiate software */
_label_TrySoftware:
                  debug_msg("vfInit(): trying to force software emulation...\n");
                  if (!vfTrySoftware()) {
                     if (HardwareFail) {
                        if (VesaFail) {
_label_InitError:
                           debug_msg("vfInit(): could not find supported mapping type. Initialization failed\n");
                           return vferrEmulationFailed;
                        } else {
                           goto _label_TryVBE;
                        }
                     } else {
                        goto _label_TryHardware;
                     }
                  }
                  if (BankCode==0 || useDrv) BankCode = &HND_Int10;
                  atexit(vfDone);
                  return vferrOK;
              }

_label_TryHardware:
              maptype = vfHardware;
              /*
               * map physical address to linear. If mapping failed, we can
               * treat this as global fatal error.
               */
              assert((lin_base = PM_MapPhysicalToLinear(lfb_base,VBEMemory))!=0);
              atexit(vfDone);
              return vferrOK;
           }
      }
  #ifdef __VF__DEBUG__
        else {
         printf("vfInit(): VGA compatible device not founded. Possibly used old ISA device\n");
      }
  #endif
   }
   debug_msg("vfInit(): hardware detection fails. ");
   if (!VesaFail && forcemode!=vfSoftware) {
_label_BackToVBE:
      debug_msg("Using hardware LFB support provided by VBE\n");
      goto _label_TryVBE;
   }
   debug_msg("Trying software emulation...\n");
   if (vfTrySoftware()) {
      if (BankCode==0 || useDrv) BankCode = &HND_Int10;
      atexit(vfDone);
      return vferrOK;
   } else
      if (!VesaFail) {
         debug_msg("vfInit(): ");
         goto _label_BackToVBE;
      }
   goto _label_InitError;
 }
#endif

/***************************************************************************
 *                                                                         *
 * Procedure: vfDone()                                                     *
 * Description: Deinitiates VFlat. Frees all allocated memory.             *
 *                                                                         *
 **************************************************************************/
void  vfDone() {
   if (mode_set) vfCloseMode();
   if (lin_base) PM_FreePhysicalMapping(lin_base);
   if (maptype == vfSoftware) DoneVFlat();
   maptype = mode_set = lin_base = lfb_base = device_id = vendor_id = 0;
 }


/***************************************************************************
 *                                                                         *
 * Procedure: vfSetMode()                                                  *
 * Description: Sets video mode using VBE call. Returns error code         *
 *              Note! You *must* set LFB flag to set video mode with LFB.  *
 *                                                                         *
 **************************************************************************/
int   vfSetMode(word mode) {
   /* If mode was previously initiated, first close it */
   if (mode_set) vfCloseMode();
   /* store current video mode. mode number obtained via standart bios services */
   prev_mode = GetMode();
   /*
    * if LFB flag does not set, just set video mode with VBE set mode
    * function and return error code.
    */
   if ((mode & vbeLinearBuffer)==0)
      return VBE_setVideoMode(mode) ? (mode_set = mode,vferrOK) : vferrErrorModeSet;
   /*
    * get information about this video mode
    */
   VBE_modeInfo modeInfo;
   if (!VBE_getModeInfo(mode & ~vbeLinearBuffer,&modeInfo)) return vferrModeNotSupported;
   /*
    * if VBE support avaible, just set mode and map physical address to
    * linear
    */
   if (maptype == vfVBESupport) {
      /* check if mode support LFB addressing */
      if ((modeInfo.ModeAttributes & vbeModeLinear)==0 ||
           (lfb_base = modeInfo.PhysBasePtr)==0) return vferrLFBNotSupported;

      if (!VBE_setVideoMode(mode)) return vferrErrorModeSet;
      /* map physical address to linear */
      assert((lin_base = PM_MapPhysicalToLinear(lfb_base,VBEMemory))!=0);
      mode_set = mode;
      return vferrOK;
   } else
   if (maptype == vfHardware)  {
      /* clear LFB flat and set mode using VBE call */
      if (!VBE_setVideoMode(mode & ~vbeLinearBuffer)) return vferrErrorModeSet;
      /* enable LFB */
      InitProc();
      mode_set = mode;
      return vferrOK;
   } else
   if (maptype == vfSoftware)  {
      if (modeInfo.WinGranularity != 64 && modeInfo.WinGranularity!=4) {
#ifdef __VF__DEBUG__
         printf("vfSetMode(): unsupported window granularity %d\n",modeInfo.WinGranularity);
#endif
         return vferrModeNotSupported;
      }
      /* clear LFB flat and set mode using VBE call */
      if (!VBE_setVideoMode(mode & ~vbeLinearBuffer)) return vferrErrorModeSet;
#ifdef __VF__SE__
      /* VFlat Software Edition initialization code */
      /* Create exception handler */
      int size = GetHandlerSize(BankCode,modeInfo.WinGranularity);
      HandlerCode = malloc(size);
      if (!HandlerCode) {
        VBE_setVideoMode(0x3);
        return vferrErrorModeSet;
      }
      BuildHandler(BankCode,modeInfo.WinGranularity,HandlerCode);
      InitPageTable(modeInfo.WinGranularity);
      InstallVFHandler(modeInfo.WinGranularity,HandlerCode);
#else
      /* VFlat initialization code */
      InitPageTable(modeInfo.WinGranularity);
      InstallVFHandler(modeInfo.WinGranularity);
#endif
      mode_set = mode;
      return vferrOK;
   }
   return vferrErrorModeSet;
}

/***************************************************************************
 *                                                                         *
 * Procedure: vfCloseMode()                                                *
 * Description: Restore video mode that was before calling vfSetMode()     *
 *              Note! you *must* call this function or vfDone() before     *
 *              program termination                                        *
 *                                                                         *
 **************************************************************************/
void  vfCloseMode() {
   /* restore previous mode */
   if (!mode_set) return;
   SetMode(prev_mode);
   if (maptype == vfSoftware) {
      RemoveVFHandler();
      DonePageTable();
   }
   if (maptype == vfVBESupport) {
      PM_FreePhysicalMapping(lin_base);
      lin_base = lfb_base = 0;
   }
   mode_set = 0;
 }


/***************************************************************************
 *                                                                         *
 * Procedure: vfReturnStatus()                                             *
 * Description: Returns VFlat status structure.                            *
 *                                                                         *
 **************************************************************************/
void  vfReturnStatus(vfStatus *st) {
   st->mode = mode_set;
   st->maptype = maptype;
   st->physbase = (mode_set & vbeLinearBuffer) == 0 ? (char*)0xA0000:(char*)lfb_base;
   st->linbase  = (mode_set & vbeLinearBuffer) == 0 ? (char*)0xA0000:(char*)lin_base;
   st->vendor_id = vendor_id;
   st->device_id = device_id;
   st->lfb_enable = (void*)InitProc;
#ifdef __VF__SE__
   st->bankswitch = (void*)BankCode;
#endif
 }

