/* Version 1.00
   January 15, 1996
   setid.c
   Written by Jeremiah W. James
   Compiled with:
      Borland C++ 3.0 (small memory model)
      GCC 2.6.3 (DJGPP 1.12.maint4)

   This program sets the ID field of the individuals in a PAF database.
   It can set them according to an ancestor numbering scheme
   (Sosa-Stradonitz or ahnentafel system), a descendant numbering scheme
   (Modified Henry system), or a combined scheme (previous two separated
   by a period).  It can also be set to overwrite previously existing
   IDs or skip people with previously existing IDs.

   The program accepts the following command-line options:
      -r# = The RIN of the base individual
      -m# = The MRIN of the base couple
      -a  = Don't number ancestors (descendant scheme only)
      +a  = Number ancestors
      -d  = Don't number descendants (ancestor scheme only)
      +d  = Number descendants
      -o  = Do not overwrite pre-existing IDs.
      +o  = Overwrite pre-existing IDs.
   The default command-line is: -r1 +a +d +o.  If conflicting switches
   are present, then the last one to appear on the command line will
   take effect.
*/

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "idtypes.h"


/**************************** Function prototypes ***************************/

static BOOL ID_less(char * const id1, const char FARPTR const id2),
   Label_relatives(const WORD ind);
static FASTFUNC int read_int(const char * const num);
static void Label_ancestors(const WORD ind, const WORD my_value);


/***************************** Global variables *****************************/

static BOOL Anc = TRUE, Desc = TRUE, Overwrite = TRUE;
static const char description[] =
"SETID - automatically set the ID fields in a PAF database.\n\
Usage: setid [options], where options is any combination of the following:\n\
       -r# = The RIN of the base individual\n\
       -m# = The MRIN of the base couple\n\
       -a  = Don't number ancestors (descendant scheme only)\n\
       +a  = Number ancestors\n\
       -d  = Don't number descendants (ancestor scheme only)\n\
       +d  = Number descendants\n\
       -o  = Do not overwrite pre-existing IDs.\n\
       +o  = Overwrite pre-existing IDs.\n\
The default command-line is: -r1 +a +d +o.  If conflicting switches are\n\
present, then the last one to appear on the command line will take effect.";
static individual FARPTR indivs;
static marriage FARPTR marrs;
static union {
  char buffer[BUFFER_SIZE];
  pafindividual ibuf;
  pafmarriage mbuf;
} b;

/*************************** Function definitions ***************************/

/* To give the GCC compiler a chance to inline some functions, the
   definitions here are in "reverse" order; i.e. each function is
   defined before its first use.  This means, of course, that main() is
   last.
*/

/* read_int -- atoi() carries too much baggage with it when we link with
   Borland C, and provides more functionality that we really need.  This
   function is a replacement.  It reads characters from src, and
   interprets them as the ASCII representation of an unsigned decimal
   integer.  It assumes that the ASCII representation is no more than
   ID_SIZE characters in length.
*/
static FASTFUNC int read_int(const char * const src)
{
  int val = 0;
  register int i;

  for (i = 0; (src[i] < '0' || src[i] > '9') && i < ID_SIZE; i++)
    if (src[i] == '\0')
      return 0;
  for ( ; src[i] >= '0' && src[i] <= '9' && i < ID_SIZE; i++)
    val = val * 10 + src[i] - '0';
  return val;
}

/* ID_less: This function compares two ID strings, the first of which is near
   and the second far.  If the first, id1, is "less than" the second, id2,
   TRUE is returned; otherwise, FALSE is returned.  We say that id1 is
   less than id2 if id1 represents a person "more closely related" to
   the root person than does id2.

   Each ID potentially has three parts: an ancestor part, a dot, and a
   descendant part.  If descendant numbering is in effect, then the ID
   with the shortest descendant part is "less".  If descendant numbering
   is not being used or the descendant parts have equal length, then the
   ID with the numerically smallest ancestor part is "less".  Note that
   this function is only called during descendant numbering, and is only
   called on two IDs with descendant parts of equal length.  Therefore,
   we merely test the ancestor portions.
*/
static BOOL ID_less(char * const id1, const char FARPTR const id2)
{
  char idtemp[ID_SIZE + PAD_SIZE];

  if (Anc) {
    memset(idtemp, '\0', ID_SIZE + PAD_SIZE);
    FARSTRNCPY((char FARPTR)idtemp, id2, ID_SIZE);
    return (read_int(id1) < read_int(idtemp));
  }
  else return FALSE;
}

/* Label_relatives -- In the descendant labelling phase, this function
   is called on an individual 'ind' to label his/her spouse(s) and
   child(ren).  This function is NOT recursive since our graph traversal
   algorithm takes care of propagating IDs for us.
*/
static BOOL Label_relatives(const WORD ind)
{
  BOOL changed = FALSE;
  char sex, id1[ID_SIZE + PAD_SIZE], id2[ID_SIZE + PAD_SIZE];
  char child_num, old_child_num;
  short len1, len2, spouse_num, num_children;
  WORD child, spouse, mar, status;

  /* Does 'ind' have any spouses or children? */
  if (indivs[ind].marriage == 0)
    return FALSE;
  sex = (marrs[indivs[ind].marriage].husband == ind) ? 'M' : 'F';

  /* Prepare my ID for modification */
  memset(id1, '\0', ID_SIZE + PAD_SIZE);
  memset(id2, '\0', ID_SIZE + PAD_SIZE);
  FARSTRNCPY((char FARPTR)id1, indivs[ind].id, ID_SIZE);
  len1 = strlen(id1);
  if (Anc && strchr(id1, '.') == NULL) id1[len1++] = '.';
  strcpy(id2, id1);
  len2 = len1;

  /* Check my relatives, by spouse */
  child_num = '0';
  for (spouse_num = 0, mar = indivs[ind].marriage; mar != 0; spouse_num++,
       mar = (sex=='M') ? marrs[mar].husband_other : marrs[mar].wife_other) {
    spouse = (sex == 'M') ? marrs[mar].wife : marrs[mar].husband;
    if (spouse > 0) {
      status = indivs[spouse].status;
      id1[len1] = spouse_num + 'A';
      if (status == st_UNSET ||
	  (status == st_SET_THIS && ID_less(id1, indivs[spouse].id))) {
	FARSTRNCPY(indivs[spouse].id, id1, ID_SIZE);
	indivs[spouse].status = st_SET_THIS;
	changed = TRUE;
      }
    }

    /* Check the children of this marriage/relationship */
    for (num_children = 0, child = marrs[mar].child; child != 0;
	 num_children++, child = indivs[child].older_sibling)
      /* PAF stores children in youngest-to-oldest order, so we have to
	 get a count of the children before we begin */
      ;

    if (child_num <= '9' && child_num + num_children > '9')
      child_num += 'a' - '9' - 1;
    child_num += num_children;
    old_child_num = child_num;
    for (child = marrs[mar].child; child != 0;
	 child_num--, child = indivs[child].older_sibling) {
      if (child_num == 'a' - 1) child_num = '9';
      status = indivs[child].status;
      id2[len2] = child_num;
      if (status == st_UNSET ||
	  (status == st_SET_THIS && ID_less(id2, indivs[child].id))) {
	FARSTRNCPY(indivs[child].id, id2, ID_SIZE);
	indivs[child].status = st_SET_THIS;
	changed = TRUE;
      }
    }
    child_num = old_child_num;
  }
  return changed;
}

/* Label_ancestors -- A simple recursive algorithm to set IDs of all
   ancestors.  It labels the parents of 'ind' and then calls itself on
   those parents, if they exist.  The 'my_value' parameter is used to
   keep track of the value to set into each ID.  Note that the stack
   size needs to be proportional to the maximum distance from the
   starting person to an ancestor.
*/
static void Label_ancestors(const WORD ind, const WORD my_value)
{
  const WORD mar = indivs[ind].parent_marriage;
  const WORD father = marrs[mar].husband;
  const WORD mother = marrs[mar].wife;

  if (mar == 0) return;
  if (father > 0 && indivs[father].status == st_UNSET) {
    sprintf(b.buffer, "%d", my_value * 2);
    FARSTRNCPY(indivs[father].id, (char FARPTR)b.buffer, ID_SIZE);
    indivs[father].status = st_SET_LAST;
    Label_ancestors(father, my_value * 2);
  }
  if (mother > 0 && indivs[mother].status == st_UNSET) {
    sprintf(b.buffer, "%d", my_value * 2 + 1);
    FARSTRNCPY(indivs[mother].id, (char FARPTR)b.buffer, ID_SIZE);
    indivs[mother].status = st_SET_LAST;
    Label_ancestors(mother, my_value * 2 + 1);
  }
}

/* main -- Here we process command-line arguments, read in the necessary
   data from INDIV2.DAT and MARR2.DAT, set the appropriate ID fields,
   and write the changed IDs back out to INDIV2.DAT.
*/
int main(int argc, char *argv[])
{
  BOOL changed, startrin = TRUE;
  FILE *data;
  WORD start = 1, husband = 0, wife = 0, num_indiv, num_marr;
  register int i;


   /**** Get command-line arguments ****/

  for (i = 1; i < argc; i++) {
    BOOL plus;

    if (argv[i][0] == '-')
      plus = FALSE;
    else if (argv[i][0] == '+')
      plus = TRUE;
    else {
      fputs(description, stderr);
      return EXIT_FAILURE;
    }
    switch(toupper(argv[i][1])) {
    case 'A':
      Anc = plus;
      break;
    case 'D':
      Desc = plus;
      break;
    case 'O':
      Overwrite = plus;
      break;
    case 'M':
      startrin = FALSE;
      start = read_int(&argv[i][2]);
      break;
    case 'R':
      startrin = TRUE;
      start = read_int(&argv[i][2]);
      break;
    default:
      fputs(description, stderr);
      return EXIT_FAILURE;
    }
  }


  /**** Read in data from PAF files ****/

  /* Extract needed info from INDIV2.DAT */
  puts("Reading INDIV2.DAT.");
  if ((data = fopen("INDIV2.DAT", "rb")) == NULL) {
    fprintf(stderr, "Error #%d trying to open \"INDIV2.DAT\"!", errno);
    return EXIT_FAILURE;
  }

  /* Read the header info from INDIV2.DAT */
  if (fread(b.buffer, sizeof(pafindividual), 1, data) != 1) {
    fprintf(stderr,
            "Error #%d trying to read the header of \"INDIV2.DAT\"!", errno);
    return EXIT_FAILURE;
  }
  num_indiv = read_int(&b.buffer[1]);

  /* Set up the individual information array */
  indivs = (individual FARPTR)FARMALLOC((num_indiv + 1) * sizeof(individual));
  if (indivs == NULL) {
    fputs("Insufficient memory for INDIV2.DAT!  You'll have to use the slow version.",
	  stderr);
    return EXIT_FAILURE;
  }

  /* Read in the information from INDIV2.DAT */
  for (i = 1; i <= num_indiv; i++) {
    if (fread(b.buffer, sizeof(pafindividual), 1, data) != 1) {
      fprintf(stderr, "Error #%d trying to read \"INDIV2.DAT\"!", errno);
      return EXIT_FAILURE;
    }
    FARMEMCPY(&indivs[i], &b.ibuf.older_sibling,
	      3 * sizeof(WORD) + 10 * sizeof(char));
    indivs[i].status = (b.ibuf.sex == 'D') ? st_FINAL : st_UNSET;
  }
  fclose(data);

  /* Set the initial status of each individual record */
  if (!Overwrite) {
    for (i = 1; i <= num_indiv; i++)
      if (indivs[i].id[0] != '\0')
	indivs[i].status = st_FINAL;
  }

  /* Extract needed info from MARR2.DAT */
  puts("Reading MARR2.DAT.");
  if ((data = fopen("MARR2.DAT", "rb")) == NULL) {
    fprintf(stderr, "Error #%d trying to open \"MARR2.DAT\"!", errno);
    return EXIT_FAILURE;
  }

  /* Read the header info from MARR2.DAT */
  if (fread(b.buffer, sizeof(pafmarriage), 1, data) != 1) {
    fprintf(stderr,
            "Error #%d trying to read the header of \"MARR2.DAT\"!", errno);
    return EXIT_FAILURE;
  }
  num_marr = read_int(&b.buffer[1]);

  /* Set up the marriage information array */
  marrs = (marriage FARPTR)FARMALLOC((num_marr + 1) * sizeof(marriage));
  if (marrs == NULL) {
    fputs("Insufficient memory for MARR2.DAT!  You'll have to use the slow version.",
	  stderr);
    return EXIT_FAILURE;
  }

  /* Read in the information from MARR2.DAT */
  for (i = 1; i <= num_marr; i++) {
    if (fread(b.buffer, sizeof(pafmarriage), 1, data) != 1) {
      fprintf(stderr, "Error #%d trying to read \"MARR2.DAT\"!", errno);
      return EXIT_FAILURE;
    }
    marrs[i].husband = b.mbuf.husband;
    marrs[i].wife = b.mbuf.wife;
    marrs[i].child = b.mbuf.child;
    marrs[i].husband_other = b.mbuf.husband_other;
    marrs[i].wife_other = b.mbuf.wife_other;
  }
  fclose(data);


  /* Set the ID(s) of the starting person or couple */

  if (startrin) {
    FARSTRNCPY(indivs[start].id, (const char FARPTR)"1", ID_SIZE);
    indivs[start].status = st_SET_LAST;
  }
  else {
    if ((husband = marrs[start].husband) > 0) {
      FARSTRNCPY(indivs[husband].id, (const char FARPTR)"2", ID_SIZE);
      indivs[husband].status = st_SET_LAST;
    }
    if ((wife = marrs[start].wife) > 0) {
      FARSTRNCPY(indivs[wife].id, (const char FARPTR)"3", ID_SIZE);
      indivs[wife].status = st_SET_LAST;
    }
  }


  /**** Set IDs for ancestors ****/

  if (Anc) {
    puts("Setting ancestor IDs.");
    if (startrin)
      Label_ancestors(start, 1);
    else {
      Label_ancestors(husband, 2);
      Label_ancestors(wife, 3);
    }
  }


  /**** Set IDs for descendants ****/

  /* We use a graph traversal algorithm to ensure that all applicable
     ID fields are eventually set to the lowest valid ID. */

  if (Desc) {
    printf("Setting descendant IDs.");
    do {
      putchar('.');
      changed = FALSE;
      for (i = 1; i <= num_indiv; i++)
	if (indivs[i].status == st_SET_LAST)
	  changed |= Label_relatives(i);
      for (i = 1; i <= num_indiv; i++) {
	if (indivs[i].status == st_SET_LAST)
	  indivs[i].status = st_FINAL;
	else if (indivs[i].status == st_SET_THIS)
	  indivs[i].status = st_SET_LAST;
      }
    } while (changed);
  }


  /**** Store the new IDs into INDIV2.DAT ****/

  puts("\nWriting INDIV2.DAT.");
  if ((data = fopen("INDIV2.DAT", "r+b")) == NULL) {
    fprintf(stderr, "Error #%d trying to open \"INDIV2.DAT\"!", errno);
    return EXIT_FAILURE;
  }


  /* Set the file pointer to the ID field of RIN 1. */

  fseek(data, 172, SEEK_SET);


  /* Write out the IDs */

  b.buffer[ID_SIZE] = '\0';     /* Just in case.... */
  for (i = 1; i <= num_indiv; i++) {
    FARSTRNCPY((char FARPTR)b.buffer, indivs[i].id, ID_SIZE);
    fwrite(b.buffer, 1, ID_SIZE, data);
    fseek(data, 82, SEEK_CUR);
  }

  return EXIT_SUCCESS;
}
