
MODULE xasc10;  (* Ascii table, JPI TopSpeed M-2 V1.16, Ron Buelow 5/90 *)

 FROM IO     IMPORT  WrCard, WrChar, WrCharRep, WrHex, WrLn, WrStr, WrStrAdj;
 FROM Window IMPORT  Black, LightGray, Clear, Close, DoubleFrame, GotoXY,
                     Hide, Open, PutOnTop, Use, WinDef, WinType;

 IMPORT Lib, Str, SYSTEM;

 TYPE
   DefType = ARRAY [0..32] OF ARRAY [0..2] OF CHAR;
   AttType = ARRAY [0..15] OF ARRAY [0..1] OF CHAR;

 CONST
   xDef = DefType('NUL','SOH','STX','ETX','EOT','ENQ','ACK','BEL',' BS',
                  ' HT',' LF',' VT',' FF',' CR',' SO',' SI','DLE','DC1',
                  'DC2','DC3','DC4','NAK','SYN','ETB','CAN',' EM','SUB',
                  'ESC',' FS',' GS',' RS',' US','SPC');

   xAtt = AttType('bk','bu','gn','cn','rd','vt','bn','wh','GY','BU','GN',
                  'CN','RD','VT','YW','WH');
   ScrWidth = 80;
   ScrDepth = 25;

   Depth = 23;  (* window *)
   Width = 36;
   Left = ScrWidth - Width;
   Top = (ScrDepth - Depth) -2;
   Right = Left + Width-1;
   Bottom = Top + Depth;

   pointup = 36C;  (* char *)
   pointdn = 37C;  (* char *)
   rev = 160C;  (* attr *)
   nor = 7C;  (* attr *)
   cursor = 17C;  (* attr *)

   esc = 33C;     f1 = 273C;     f2 = 274C;
   home = 307C;   up = 310C;     pgup = 311C;
   left = 313C;   right = 315C;  end = 317C;
   dn = 320C;     pgdn = 321C;   ctrlL = 363C;
   ctrlR = 364C;

 VAR
   screen [0B800H:0]: ARRAY [1..ScrDepth], [1..ScrWidth] OF
                        RECORD
                          char : CHAR;
                          attr : CHAR
                        END;

   x, y, x1, y1, x2, y2, z : CARDINAL;
   c, d, e, i : SHORTCARD;
   scnNum, chrNum : CARDINAL;
   helpon, kbmode : BOOLEAN;
   sc, ch : CHAR;

   WD, HELPWD : WinDef;
   AscWin, HelpWin : WinType;

 PROCEDURE GetScanKey(VAR sc:CHAR): CHAR;
 VAR r: SYSTEM.Registers;
 BEGIN
   WITH r DO
     AH := 0;
     Lib.Intr(r,16H);
     IF AH # 0 THEN
       sc := CHR(AH)
     ELSE
       sc := CHR(AH-128)
     END(*IF*);
   IF sc > CHR(127) THEN
     AL := AH
   END(*IF*);
     IF AL # 0 THEN
       RETURN CHR(AL)
     ELSE
       RETURN CHR(AH+128)
     END(*IF*)
   END(*WITH*)
 END GetScanKey;

 PROCEDURE WrOct(n:CARDINAL; Length:INTEGER);
 VAR
   s : ARRAY[0..80] OF CHAR;
   OK : BOOLEAN;
 BEGIN
   Str.CardToStr(VAL(LONGCARD,n),s,8,OK);
   IF OK THEN
     WrStrAdj(s,Length)
   END(*IF*)
 END WrOct;

 PROCEDURE MoveCursor(newX, newY : INTEGER);
 BEGIN
   IF newX < Left+3 THEN  (* wrap cursor *)
     newX := Right-1;
     newY := newY-2
   END;(*IF*)
   IF newX > Right-1 THEN
     newX := Left+3;
     newY := newY+2
   END;(*IF*)
   IF newY < Top+3 THEN
     newY := Bottom-6
   END;(*IF*)
   IF newY > Bottom-6 THEN
     newY := Top+3
   END;(*IF*)
   screen[y-1,x].char := ' ';  (* erase & move cursor *)
   screen[y,x].attr := rev;
   screen[y+1,x].char := ' ';
   x := newX;
   y := newY;
   screen[y-1,x].char := pointdn;
   screen[y,x].attr := cursor;
   screen[y+1,x].char := pointup
 END MoveCursor;

 PROCEDURE WrMode;
 BEGIN
   GotoXY(18,22);
   IF kbmode = TRUE THEN
     WrStr(" Keybd ")
   ELSE
     WrStr(" Table ")
   END;(*IF*)
   x1 := Left+19;
   y1 := Top+23;
   FOR i := 1 TO 7 DO
     screen[y1,x1].attr := nor;
     INC(x1)
   END(*FOR*)
 END WrMode;

 PROCEDURE WrTable;
 BEGIN
   PutOnTop(AscWin);
   c := 0;
   d := Width-5;
   x := Left+3;
   y := Top+3;
   FOR i := 1 TO 8 DO  (* write table *)
     FOR e := c TO c + d DO
       screen[y,x].attr := rev;
       screen[y,x].char := CHR(e);
       INC(x)
     END;(*FOR*)
     INC(y);
     INC(y);
     x := Left+3;
     c := e+1
   END;(*FOR*)
   x := Left+1;  (* adjust borders *)
   y := Bottom-4;
   FOR i := 1 TO 2 DO
     screen[y,x].char := CHR(199);
     INC(x);
       FOR e := 1 TO 34 DO
         screen[y,x].char := CHR(196);
         INC(x)
       END;(*FOR*)
     screen[y,x].char := CHR(182);
     x := Left+1;
     y := Bottom-1
   END;(*FOR*)           (* write text *)
   GotoXY(1,20);
   WrStr(" CHR DEC HEX OCT DEF SCD SCH ATTR");
   GotoXY(1,22);
   WrStr(" F1-Help F2-Mode  Table  Esc-Exit ");
   x := Left+3;
   y := Top+3;
   MoveCursor(x,y);  (* mark cursor *)
   WrMode;
   IF helpon = TRUE THEN
     MoveCursor(x2,y2)  (* return cursor *)
   END;(*IF*)
   helpon := FALSE
 END WrTable;

 PROCEDURE Help;
 VAR ch : CHAR;
 PROCEDURE Help1;
 BEGIN
   GotoXY(1,2);
   WrStr("         Table Navigation");
   WrLn; WrLn;
   WrStr("           Cursor Keys:  ");
   WrLn; WrLn;
   WrStr("         Home       PgUp");
   WrLn; WrLn;
   WrStr("         -          -");
   WrLn; WrLn;
   WrStr("         End        PgDn");
   WrLn; WrLn;
   WrStr("    Ctrl -  (Edges)   Ctrl -");
   WrLn; WrLn;
   WrStr("        (-- Keys Wrap)");
   WrLn; WrLn;
   WrStr("     <Esc> Key = Exit Program")
 END Help1;


 PROCEDURE Help2;
 BEGIN
   GotoXY(3,2);
   WrStr(" Character at cursor"); WrLn;
   WrStr("  "); WrLn;
   WrStr("      Decimal ordinal number    ");
   WrStr("     "); WrLn;
   WrStr("         Hexadecimal ordinal   ");
   WrStr("        "); WrLn;
   WrStr("            Octal ordinal     ");
   WrStr("           "); WrLn;
   WrStr("               Definition    ");
   WrStr("              "); WrLn;
   WrStr("                Ctrl-Char or ");
   WrStr("                Scan Code F2 ");
   WrStr("                           ");
   WrStr("                      Attr ");
   WrStr("                          ");
   WrStr("                       ")
 END Help2;

 PROCEDURE Help3;
 BEGIN
   GotoXY(2,2);
   WrStr("Attribute of character ordinal #  ");
   WrLn;
   WrStr("   # 8-15 blink when background   ");
   WrStr("   & become same color as # 0-7   ");
   WrLn;
   WrStr(" 0 bk/black    8 GY/Gray"); WrLn;
   WrStr(" 1 bu/blue     9 BU/Bright Blue   ");
   WrStr(" 2 gn/green   10 GN/Bright Green  ");
   WrStr(" 3 cn/cyan    11 CN/Bright Cyan   ");
   WrStr(" 4 rd/red     12 RD/Bright Red    ");
   WrStr(" 5 vt/violet  13 VT/Bright Violet ");
   WrStr(" 6 bn/brown   14 YW/Yellow        ");
   WrStr(" 7 wh/white   15 WH/Bright White  ");
   WrLn;
   WrStr(" Color attribute fore & back Ŀ");
   GotoXY(31,17);
   WrChar(CHR(31))
 END Help3;

 PROCEDURE Help4;
 BEGIN
   GotoXY(1,2);
   WrStr("  Two modes are available, Table  ");
   WrStr("  and Keyboard. The current mode  ");
   WrStr("  is indicated at window bottom.  ");
   WrLn;
   WrStr("  Changing modes - from >>> to:   ");
   WrLn;
   WrStr("  Table >>> Keybd"); WrLn;
   WrStr("   press the <F2> key once       ");
   WrStr("   the mode changes to Keyboard  ");
   WrStr("   values reflect pressed keys   ");
   WrLn;
   WrStr("  Keybd >>> Table"); WrLn;
   WrStr("   press <F2> or <Esc> key twice ");
   WrStr("   the mode changes to Table     ");
   WrStr("   values reflect selected char")
 END Help4;

 BEGIN
   GotoXY(1,22);
   WrStr("  Page ...      Esc = Exit Help ");
   x2 := Left+2;
   y2 := Bottom;
   FOR i := 1 TO 34 DO
     screen[y2,x2].attr := nor;
     INC(x2)
   END;(*FOR*)
   PutOnTop(HelpWin);
   z := 1;
   Help1;
     REPEAT
       Use(AscWin);
       GotoXY(9,0); WrStr(" Help Page   of 4 ");
       GotoXY(20,0); WrCard(z,1); (* page number *)
       Use(HelpWin);
       ch := GetScanKey(sc);
       CASE ch OF
         pgup, up : Clear; DEC(z)
       | pgdn, dn : Clear; INC(z)
       END;(*CASE*)
       IF z > 4 THEN z := 1
         ELSIF z < 1 THEN z := 4
       END;(*IF*)
       CASE z OF
         1 : Help1
       | 2 : Help2
       | 3 : Help3
       | 4 : Help4
       END(*CASE*)
     UNTIL ch = esc;
   x2 := x;
   y2 := y;
   Clear;
   Hide(HelpWin);
   PutOnTop(AscWin);
   GotoXY(9,0);
   WrCharRep('',18);
   WrTable
 END Help;

 PROCEDURE WrOrd;
 BEGIN
   IF scnNum = 3 THEN  (* tag ^2 nul *)
     IF chrNum = 131 THEN
       chrNum := 0;
       x1 := Left+3;
       y1 := Top+3;
       MoveCursor(x1,y1)
     END(*IF*)
   END(*IF*);

   screen[Bottom-3,Left+4].char := CHR(chrNum);  (* write character *)

   GotoXY(6,19);  (* write ordinals *)
   WrCard(chrNum,3);
   WrHex(chrNum,3);
   WrStr('h ');
   WrOct(chrNum,3);
   WrChar(' ');

   IF chrNum < 33 THEN  (* write definition *)
     WrStr(xDef[chrNum])
   ELSIF chrNum = 127 THEN
     WrStr("DEL")
   ELSE
     WrStr("   ")
   END;(*IF*)

   IF kbmode = FALSE THEN  (* write control char *)
     GotoXY(22,20);
     WrStr("       ");
     GotoXY(22,19);
     CASE chrNum OF
     0..31   : WrChar('^');
               WrChar(CHR(chrNum+64));
               WrStr("     ")
     ELSE
       WrStr("      ")
     END;(*CASE*)
   ELSE
     GotoXY(22,20);
     WrStr("SCD SCH")
   END;(*IF*)

   GotoXY(30,19);  (* write attribute *)
   WrStr(xAtt[chrNum MOD 16]);
   WrStr(xAtt[chrNum DIV 16]);
   x1 := Right-4;
   y1 := Bottom-2;
   FOR i := 1 TO 4 DO
     screen[y1,x1].attr := CHR(chrNum);
     INC(x1)
   END(*FOR*)
 END WrOrd;

 PROCEDURE WrCode;
 BEGIN
   scnNum := ORD(sc);
   chrNum := ORD(ch);
   MoveCursor((Left+3)+chrNum MOD 32,
             ((chrNum DIV 32)*2)+Top+3);
   WrOrd;
   GotoXY(22,19);
   WrCard(scnNum,3);  (* write scancode *)
   WrHex(scnNum,3);
   WrChar('h');
   ch := GetScanKey(sc)
 END WrCode;

 PROCEDURE RunAsc;
 BEGIN
   WrTable;
   WrMode;
   kbmode := FALSE;
   LOOP
     chrNum := ORD(screen[y,x].char);
     WrOrd;
     WHILE kbmode = TRUE DO
       CASE ch OF
         f1 : WrCode;
              IF ch = f1 THEN
                Help;
                sc := CHR(59)
              END(*IF*)
       | f2 : WrCode;
              IF ch = f2 THEN
                MoveCursor(Left+3,Top+3);
                kbmode := FALSE
              END;(*IF*)
              WrMode
      | esc : WrCode;
              IF ch = esc THEN
                MoveCursor(Left+3,Top+3);
                kbmode := FALSE
              END;(*IF*)
              WrMode
       ELSE
         WrCode
       END(*CASE*)
     END;(*WHILE*)

     chrNum := ORD(screen[y,x].char);
     WrOrd;

     ch := GetScanKey(sc);

     CASE ch OF  (* handle input *)
       f1    : helpon := TRUE ; Help
     | f2    : IF kbmode = TRUE THEN
                 kbmode := FALSE;
               ELSE
                 kbmode := TRUE
               END;
               WrMode
     | esc   : EXIT (* end program *)
     | left  : MoveCursor(x-1,y)  (* cursor controls *)
     | up    : MoveCursor(x,y-2)
     | right : MoveCursor(x+1,y)
     | dn    : MoveCursor(x,y+2)
     | home  : MoveCursor(Left+3,Top+3)
     | end   : MoveCursor(Right-1,Bottom-6)
     | pgup  : MoveCursor(x,Top+3)
     | pgdn  : MoveCursor(x,Bottom-6)
     | ctrlL : MoveCursor(Left+3,y)
     | ctrlR : MoveCursor(Right-1,y)
     ELSE
       IF kbmode = FALSE THEN
         WrChar(7C)  (* honk if invalid key is hit *)
       END(*IF*)
     END(*CASE*)
   END;(*LOOP*)
   Close(HelpWin);
   Close(AscWin)
 END RunAsc;

 (* run program *)

 BEGIN
   WITH HELPWD DO
     X1 := Left+1;
     Y1 := Top+1;
     X2 := Right-1;
     Y2 := Bottom-6;
     Foreground := LightGray;
     Background := Black;
     CursorOn := FALSE;
     WrapOn := TRUE;
     Hidden := TRUE;
     FrameOn := FALSE
   END;(*WITH*)

   WITH WD DO
     X1 := Left;
     Y1 := Top;
     X2 := Right;
     Y2 := Bottom;
     Foreground := Black;
     Background := LightGray;
     CursorOn := FALSE;
     WrapOn := FALSE;
     Hidden := TRUE;
     FrameOn := TRUE;
     FrameDef := DoubleFrame;
     FrameFore := Black;
     FrameBack := LightGray
   END;(*WITH*)

   HelpWin := Open(HELPWD);
   AscWin := Open(WD);
   RunAsc
 END xasc10.