/* Copyright 1996  Jonathan Griffiths. See "readme.jon" for detailed terms. */
#include <jx_font.h>

void generate_RLE (sprite_data_rec * srec); /* from source/sprite/spriteio.c */

void do_font_draw_underlineNC(font *fnt,buffer_rec *buff,int x,int y,char c,UBYTE col);
void do_font_draw_underline(font *fnt,buffer_rec *buff,int x,int y,char c,UBYTE col);
void do_font_draw_underline_buffNC(buffer_rec *src,font *fnt,buffer_rec *buff,int x,int y,char c);
void do_font_draw_underline_buff(buffer_rec *src,font *fnt,buffer_rec *buff,int x,int y,char c);


/*+------------------------------------------------------------------------+*/
/*| Load a font into memory.                                               |*/
/*+------------------------------------------------------------------------+*/
font *font_load(char *filename)
{
 unsigned long outlen;
 int i,position;
 font *newfont;
 FILE *fp;
 ULONG thirtytwo;
 UBYTE *bit_buffer;

 JLIB_ENTER("font_load");

 /* malloc space for record */
 if((newfont=(font *)malloc(sizeof(font)))==NULL){
    JLIB_SPRINTF("malloc failed at line %d.", __LINE__);
    jlib_exit(jlib_msg(JLIB_EMALLOC));
 }

 /* open font file */
 if((fp=fopen(filename,"rb"))== NULL){
        JLIB_SPRINTF("Couldn't open file %s.", filename);
        free(newfont);
        JLIB_LEAVE;
        return NULL;
 }

 /* load font details */
 newfont->magic=fgetc(fp);

 if(newfont->magic!=FONT_MAGIC){
    JLIB_SPRINTF("File %s not a font file.",filename);
    free(newfont);
    JLIB_LEAVE;
    return NULL;
 }

 newfont->reserved=fgetc(fp);
 newfont->color=fgetc(fp);
 newfont->proportional=fgetc(fp);
 newfont->width=fgetc(fp);
 newfont->height=fgetc(fp);
 newfont->underline=fgetc(fp);
 newfont->first_char=fgetc(fp);
 newfont->last_char=fgetc(fp);

 if(newfont->color){
    newfont->palette=pal_init();

    /* load in font palette */
    for(i=0;i<SCREEN_NUM_COLORS*3;i++){
       newfont->palette[i]=fgetc(fp);
    }
 }

 /* init sprite system to hold font data */
 newfont->font_data=sprite_init(1,newfont->last_char-newfont->first_char+1);

 /* load widths and set heights */
 for(i=0;i<(newfont->last_char-newfont->first_char+1);i++){
     if((newfont->font_data->sprite_data[i]=
        (sprite_data_rec *)calloc(sizeof(sprite_data_rec),1)) == NULL){
        jlib_exit(jlib_msg(JLIB_EMALLOC));
     }

     newfont->font_data->sprite_data[i]->width=fgetc(fp);
     newfont->font_data->sprite_data[i]->height=newfont->height;

     if((newfont->font_data->sprite_data[i]->data=(UBYTE *)calloc(newfont->font_data->sprite_data[i]->width*newfont->height,1))==NULL){
       jlib_exit(jlib_msg(JLIB_EMALLOC));
     }
 }

 /* get size of font data */
 jio_read_elementary_type(fp,&thirtytwo,sizeof(ULONG));
 outlen=thirtytwo;

 /* process non color font as a packed bit array */
 if(!newfont->color){
   /* malloc space for bit buffer */
   if((bit_buffer=(UBYTE *)malloc(thirtytwo*sizeof(UBYTE)))==NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }

   /* load in bit buffer */
   for(i=0;i<thirtytwo;i++){
      bit_buffer[i]=fgetc(fp);
   }
 
   fclose(fp);
 
   /* unpack bit_buffer (bit array) into the sprite structure */
   for(i=0,position=0;i<(newfont->last_char-newfont->first_char+1);i++){
       int j,size;
 
       size=newfont->font_data->sprite_data[i]->width*newfont->font_data->sprite_data[i]->height;
 
       /* unpack bit array into sprite frame buffer */
       for(j=0;j<size;j++){
           if((bit_buffer[position>>3] & (1 << (position & 7)))!=0){
             newfont->font_data->sprite_data[i]->data[j]=1;
           }
           ++position;           
      }
  
       generate_RLE(newfont->font_data->sprite_data[i]);
       newfont->font_data->sprite_data[i]->no_rects=0;
       newfont->font_data->sprite_data[i]->rect_coords=NULL;
   }

   free(bit_buffer);
 }
 else{
   jlib_exit("Colour Fonts not supported");
 }

 newfont->font_data->number_loaded = newfont->last_char-newfont->first_char+1;
 
 JLIB_LEAVE;
 return newfont;
}


/*+------------------------------------------------------------------------+*/
/*| Free the memory used by a font.                                        |*/
/*+------------------------------------------------------------------------+*/
font *font_free(font *font)
{
 JLIB_ENTER("font_free");

 if(font!=NULL){
    if(font->font_data!=NULL){
       sprite_free(font->font_data);
    }
    free(font);
 }

 JLIB_LEAVE;
 return NULL;
}


/*+------------------------------------------------------------------------+*/
/*| Return the width of a letter in the font.                              |*/
/*+------------------------------------------------------------------------+*/
int font_letter_width(font *fnt,char c)
{
 JLIB_ENTER("font_letter_width");

#ifndef JLIB_PRODUCTION
 if(fnt==NULL){
    jlib_exit(jlib_msg(JLIB_ENULL));
 }
#endif

 if(FONT_HAS_CHAR(fnt,c)){
    JLIB_LEAVE;
    return fnt->font_data->sprite_data[c-(fnt->first_char)]->width;
 }

 JLIB_LEAVE;
 return 0;
}


/*+------------------------------------------------------------------------+*/
/*| Return the height of a letter in the font.                             |*/
/*+------------------------------------------------------------------------+*/
int font_letter_height(font *fnt,char c)
{
 JLIB_ENTER("font_letter_height");

#ifndef JLIB_PRODUCTION
 if(fnt==NULL){
    jlib_exit(jlib_msg(JLIB_ENULL));
 }
#endif

 if(FONT_HAS_CHAR(fnt,c)){
    JLIB_LEAVE;
    return fnt->font_data->sprite_data[c-(fnt->first_char)]->height;
 }

 JLIB_LEAVE;
 return FONT_HEIGHT(fnt);
}


/*+------------------------------------------------------------------------+*/
/*| Return the width of a string in the font.                              |*/
/*+------------------------------------------------------------------------+*/
int font_string_width(font *fnt,char *c)
{
 int count=0;

 JLIB_ENTER("font_string_width");

#ifndef JLIB_PRODUCTION
 if((fnt==NULL)||(c==NULL)){
    jlib_exit(jlib_msg(JLIB_ENULL));
 }
#endif

 while((*c) != '\0'){
    if(FONT_HAS_CHAR(fnt,*c)){
       count+= fnt->font_data->sprite_data[*c-(fnt->first_char)]->width;
    }
    c++;
 }

 JLIB_LEAVE;
 return count;
}


/*+------------------------------------------------------------------------+*/
/*| Draw a solid underline for a character with clipping.                  |*/
/*+------------------------------------------------------------------------+*/
void do_font_draw_underline(font *fnt,buffer_rec *buff,int x,int y,char c,UBYTE col)
{
 sprite_data_rec *temp;

 JLIB_ENTER("font_draw_underline");

 y+=fnt->underline;
 temp=fnt->font_data->sprite_data[c-(fnt->first_char)];
 buff_draw_rect(buff,x,y,x+temp->width,y+(temp->height/16),col);

 JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| Draw a solid underline for a character without clipping.               |*/
/*+------------------------------------------------------------------------+*/
void do_font_draw_underlineNC(font *fnt,buffer_rec *buff,int x,int y,char c,UBYTE col)
{
 sprite_data_rec *temp;

 JLIB_ENTER("font_draw_underlineNC");

 y+=fnt->underline;
 temp=fnt->font_data->sprite_data[c-(fnt->first_char)];
 buff_draw_rectNC(buff,x,y,x+temp->width,y+(temp->height/16),col);

 JLIB_LEAVE;
}



/*+------------------------------------------------------------------------+*/
/*| Draw a textured underline for a character with clipping.               |*/
/*+------------------------------------------------------------------------+*/
void do_font_draw_underline_buff(buffer_rec *src,font *fnt,buffer_rec *buff,int x,int y,char c)
{
 sprite_data_rec *temp;

 JLIB_ENTER("font_draw_underline_buff");

 y+=fnt->underline;
 temp=fnt->font_data->sprite_data[c-(fnt->first_char)];
 buff_blit_buff_to(buff,x,y,src,0,0,temp->width,(temp->height/16));

 JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| Draw a textured underline for a character without clipping.            |*/
/*+------------------------------------------------------------------------+*/
void do_font_draw_underline_buffNC(buffer_rec *src,font *fnt,buffer_rec *buff,int x,int y,char c)
{
 sprite_data_rec *temp;

 JLIB_ENTER("font_draw_underline_buffNC");

 y+=fnt->underline;
 temp=fnt->font_data->sprite_data[c-(fnt->first_char)];
 buff_blit_buff_toNC(buff,x,y,src,0,0,temp->width,(temp->height/16));

 JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| General purpose non-clipping character drawing function.               |*/
/*+------------------------------------------------------------------------+*/
void font_charNC(font *fnt,buffer_rec *dest,int x,int y,char c,UBYTE col,buffer_rec *src,int style)
{
 JLIB_ENTER("font_charNC");
 
 if(FONT_HAS_CHAR(fnt,c)){
   if(style & FONT_TEXTURE){
     if(style & FONT_STAMP){
        buff_stamp_sprite_buffNC(fnt->font_data,c-(fnt->first_char),dest,x,y,src);
     }
     else{
        buff_stencil_sprite_buffNC(fnt->font_data,c-(fnt->first_char),dest,x,y,src);
     }
   }
   else{
     if(style & FONT_STAMP){
       buff_stamp_sprite_colorNC(fnt->font_data,c-(fnt->first_char),dest,x,y,col);
     }
     else{
       buff_stencil_sprite_colorNC(fnt->font_data,c-(fnt->first_char),dest,x,y,col);
     }
   }
  
   if(style & FONT_UNDERLINE){
      if(style & FONT_TEXTURE){
         do_font_draw_underline_buffNC(src,fnt,dest,x,y,c);
      }
      else{
         do_font_draw_underlineNC(fnt,dest,x,y,c,col);
      }
   }
 }
 JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| General purpose clipping character drawing function.                   |*/
/*+------------------------------------------------------------------------+*/
void font_char(font *fnt,buffer_rec *dest,int x,int y,char c,UBYTE col,buffer_rec *src,int style)
{
 JLIB_ENTER("font_char");

#ifndef JLIB_PRODUCTION
 if((fnt==NULL)||(dest==NULL)){
    jlib_exit(jlib_msg(JLIB_ENULL));
 }
#endif

 if(FONT_HAS_CHAR(fnt,c)){
   if(style & FONT_TEXTURE){
#ifndef JLIB_PRODUCTION
     if(src==NULL){
        jlib_exit(jlib_msg(JLIB_ENULL));
     }
#endif

     if(style & FONT_STAMP){
        buff_stamp_sprite_buff(fnt->font_data,c-(fnt->first_char),dest,x,y,src);
     }
     else{
        buff_stencil_sprite_buff(fnt->font_data,c-(fnt->first_char),dest,x,y,src);
     }
   }
   else{
     if(style & FONT_STAMP){
       buff_stamp_sprite_color(fnt->font_data,c-(fnt->first_char),dest,x,y,col);
     }
     else{
       buff_stencil_sprite_color(fnt->font_data,c-(fnt->first_char),dest,x,y,col);
     }
   }
  
   if(style & FONT_UNDERLINE){
      if(style & FONT_TEXTURE){
         do_font_draw_underline_buff(src,fnt,dest,x,y,c);
      }
      else{
         do_font_draw_underline(fnt,dest,x,y,c,col);
      }
   }
 }
 JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| General purpose non-clipping string drawing function.                  |*/
/*+------------------------------------------------------------------------+*/
void font_stringNC(font *fnt,buffer_rec *dest,int x,int y,char *s,UBYTE col,buffer_rec *src,int style)
{
 JLIB_ENTER("font_stringNC");

 while((*s) != '\0'){
   if(FONT_HAS_CHAR(fnt,*s)){
      font_charNC(fnt,dest,x,y,*s,col,src,style);
      x+=fnt->font_data->sprite_data[*s-(fnt->first_char)]->width;
   }
   s++;
 }

 JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| General purpose clipping string drawing function.                      |*/
/*+------------------------------------------------------------------------+*/
void font_string(font *fnt,buffer_rec *dest,int x,int y,char *s,UBYTE col,buffer_rec *src,int style)
{
 JLIB_ENTER("font_string");

#ifndef JLIB_PRODUCTION
 if((fnt==NULL)||(dest==NULL)||(s==NULL)){
    jlib_exit(jlib_msg(JLIB_ENULL));
 }
#endif

 while((*s) != '\0'){
   if(FONT_HAS_CHAR(fnt,*s)){
      font_char(fnt,dest,x,y,*s,col,src,style);
      x+=fnt->font_data->sprite_data[*s-(fnt->first_char)]->width;
   }
   s++;
 }

 JLIB_LEAVE;
}
