/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <jlib.h>
#include <stdlib.h>
#include <signal.h>

void          jlib_default_exit_function(char *message);
exit_function __jlib_error_handler = NULL;

/*+------------------------------------------------------------------------+*/
/*|set an exit function to be called on library errors                     |*/
/*+------------------------------------------------------------------------+*/
void jlib_set_exit_function(exit_function funct)
{
 JLIB_ENTER("jlib_set_exit_function");

 __jlib_error_handler = funct;

 JLIB_LEAVE
}


/*+------------------------------------------------------------------------+*/
/*|call the library fault routine.                                         |*/
/*+------------------------------------------------------------------------+*/
void jlib_exit(char *message)
{
 JLIB_ENTER("jlib_exit");
 
 if(__jlib_error_handler == NULL){
   JLIB_PRINT_DEBUG_INFO("Default Exit:");
   jlib_default_exit_function(message);
 }
 else{
   JLIB_PRINT_DEBUG_INFO ("Custom Exit:");
    __jlib_error_handler(message);
 }
 
 /* never arrives here */
}


/*+------------------------------------------------------------------------+*/
/*|Abolute minimim exit processing.                                        |*/
/*+------------------------------------------------------------------------+*/
void jlib_exit_details(char *message)
{
 kb_closedown();
 mouse_closedown();
 screen_restore_video_mode();

 printf("%s: Internal Error.\n %s\n",__jlib_app_title,message);

 JLIB_DUMP_STACK
}


/*+------------------------------------------------------------------------+*/
/*|default - Exit from jlib and print an error message.                    |*/
/*+------------------------------------------------------------------------+*/
void jlib_default_exit_function(char *message)
{
 jlib_exit_details(message);
  
 exit(0);
}


char __jlib_err_malloc[] =   "Memory Allocation Failure.";
char __jlib_err_nullparm[]=  "NULL Parameter.";
char __jlib_err_fileio[]=    "File I/O Failure.";
char __jlib_err_kbinit[]=    "Keyboard Initialisation Failure.";
char __jlib_err_unknown[]=   "Unrecognised Failure.";


/*+------------------------------------------------------------------------+*/
/*|Return a message corresponding to an error number.                      |*/
/*+------------------------------------------------------------------------+*/
char *jlib_msg(int error)
{
 switch(error){
 
   case JLIB_EMALLOC:
      return __jlib_err_malloc;

   case JLIB_ENULL:
      return __jlib_err_nullparm;

   case JLIB_EFILE:
      return __jlib_err_fileio;

   case JLIB_EKB:
      return __jlib_err_kbinit;
      
   default:
      return __jlib_err_unknown;
 }
} 


/*+------------------------------------------------------------------------+*/
/*|Signal handling: If the target supports it.                             |*/
/*+------------------------------------------------------------------------+*/
#ifdef JHANDLE_SIGNALS
/* robust signal processing with "sigaction" */
#define CHAIN_HANDLER(act, sig, func, oldact) \
        act.sa_handler = func; \
        memset(&act.sa_flags,0,sizeof(act.sa_flags)); \
        memset(&act.sa_mask,0,sizeof(act.sa_mask)); \
        sigaction(sig, &act, &oldact);

static struct sigaction __jlib_old_sigint;
static struct sigaction __jlib_old_sigfpe;
static struct sigaction __jlib_old_sigsegv;
static struct sigaction __jlib_old_sigill;


static void __jlib_sigint(int sig_number)
{
	jlib_exit_details("SIGINT Recieved.");
	sigaction(SIGINT, &__jlib_old_sigint, NULL);
	raise(SIGINT);
}

static void __jlib_sigfpe(int sig_number)
{
	jlib_exit_details("SIGFPE Recieved.");
	sigaction(SIGFPE, &__jlib_old_sigfpe, NULL);
	raise(SIGFPE);
}

static void __jlib_sigsegv(int sig_number)
{
	jlib_exit_details("SIGSEGV Recieved.");
	sigaction(SIGSEGV, &__jlib_old_sigsegv, NULL);
	raise(SIGSEGV);
}

static void __jlib_sigill(int sig_number)
{
	jlib_exit_details("SIGINT Recieved.");
	sigaction(SIGILL, &__jlib_old_sigill, NULL);
	raise(SIGILL);
}


void __jlib_setup_signals(void)
{
 struct sigaction siga;

 CHAIN_HANDLER(siga, SIGINT,  __jlib_sigint,  __jlib_old_sigint);
 CHAIN_HANDLER(siga, SIGFPE,  __jlib_sigfpe,  __jlib_old_sigfpe);
 CHAIN_HANDLER(siga, SIGSEGV, __jlib_sigsegv, __jlib_old_sigsegv);
 CHAIN_HANDLER(siga, SIGILL,  __jlib_sigill,  __jlib_old_sigill);
}

#else
#ifdef JHANDLE_SIMPLE_SIGNALS
/* simple signal handling */
#define CHAIN_HANDLER(sig, func)  signal(sig,func)
        
static void __jlib_sigint(int sig_number)
{
	jlib_exit_details("SIGINT Recieved.");
        signal(SIGINT,SIG_DFL);
	raise(SIGINT);
        exit(0);
}

static void __jlib_sigfpe(int sig_number)
{
	jlib_exit_details("SIGFPE Recieved.");
        signal(SIGFPE,SIG_DFL);
	raise(SIGFPE);
        exit(0);
}

static void __jlib_sigsegv(int sig_number)
{
	jlib_exit_details("SIGSEGV Recieved.");
        signal(SIGSEGV,SIG_DFL);
	raise(SIGSEGV);
        exit(0);
}

static void __jlib_sigill(int sig_number)
{
	jlib_exit_details("SIGINT Recieved.");
        signal(SIGILL,SIG_DFL);
	raise(SIGILL);
        exit(0);
}


void __jlib_setup_signals(void)
{
 CHAIN_HANDLER(SIGINT,  __jlib_sigint);
 CHAIN_HANDLER(SIGFPE,  __jlib_sigfpe);
 CHAIN_HANDLER(SIGSEGV, __jlib_sigsegv);
 CHAIN_HANDLER(SIGILL,  __jlib_sigill);
}

#endif
#endif

/*+----------------------------------------------------------------------+*/
/*|Exhaustively test the validity of a buffer.                           |*/
/*+----------------------------------------------------------------------+*/
void jlib_check_buffer(buffer_rec *buff)
{
#ifdef JDEBUG
#ifdef JLIB_PARANOID
   UBYTE test;
   UBYTE *test2;
#endif

   JLIB_ENTER("jlib_check_buffer");

   /* check buffer pointers */
   if (buff == NULL) {
      JLIB_PRINT_DEBUG_INFO("NULL buffer_rec ptr");
      jlib_exit(jlib_msg(JLIB_ENULL));
   }

   if (B_BUFF_PTR(buff)== NULL) {
      JLIB_PRINT_DEBUG_INFO("NULL B_BUFF_PTR()");
      jlib_exit(jlib_msg(JLIB_ENULL));
   }

   if (buff->offset == NULL) {
      JLIB_PRINT_DEBUG_INFO("NULL buff->offset");
      jlib_exit(jlib_msg(JLIB_ENULL));
   }

#ifdef JLIB_PARANOID
   /* check buffer size */
   if((buff->height < 1) || (buff->width < 1)) {
      jlib_exit("Invalid buffer size");
   }

   /* check access to first and last elements of malloc()'ed space */
   JLIB_ENTER("check_buffer_data");
   test = buff->buffer[0];
   test = buff->buffer[B_SIZE(buff)-1];
   JLIB_LEAVE;

   JLIB_ENTER("check_buffer_offset");
   test2 = buff->offset[0];
   test2 = buff->offset[B_MAX_Y(buff)];
   JLIB_LEAVE;
#endif

   JLIB_LEAVE;
#endif
}


/*+----------------------------------------------------------------------+*/
/*|Exhaustively test the validity of a sprite system.                    |*/
/*+----------------------------------------------------------------------+*/
void jlib_check_sprite_system(sprite_system *sys)
{
 #ifdef JDEBUG
#ifdef JLIB_PARANOID
   UBYTE test;
   sprite_record *test2;
   sprite_data_rec *test3;
#endif

   JLIB_ENTER("jlib_check_sprite_system");

   /* check simple pointers */
   if(sys == NULL) {
      JLIB_PRINT_DEBUG_INFO("sprite system is NULL");
      jlib_exit(jlib_msg(JLIB_ENULL));
   }

#ifdef JLIB_PARANOID
   /* check simple values */
   if((sys->no_sprites < 0) || (sys->no_frames < 0) || (sys->no_active < 0)
      || (sys->number_loaded < 0) || (sys->biggest_rle < 0)){
      JLIB_PRINT_DEBUG_INFO("Invalid value(s) for sprite system\n");
      JLIB_SPRINTF("no_sprites    = %d\n",sys->no_sprites);
      JLIB_SPRINTF("no_frames     = %d\n",sys->no_frames);
      JLIB_SPRINTF("no_active     = %d\n",sys->no_active);
      JLIB_SPRINTF("number_loaded = %d\n",sys->number_loaded);
      JLIB_SPRINTF("biggest_rle   = %d\n",sys->biggest_rle);
      jlib_exit("Invalid sprite system numbers");
   }

   if(sys->number_loaded > sys->no_frames) {
      jlib_exit("Inconsistent loaded frames.");
   }

   if(sys->no_sprites){
      int i,active=0;

      /* check active sprites array */
      JLIB_ENTER("check active sprite array");
      test = sys->active_sprites[0];
      test = sys->active_sprites[sys->no_sprites-1];
      JLIB_LEAVE;

      for(i=0;i<sys->no_sprites;i++) {
          active+=sys->active_sprites[i];
      }

      if(active != sys->no_active) {
         jlib_exit("Inconsistent active sprites");
      }

      /* check sprites array */
      JLIB_ENTER("check sprite array");
      test2 = sys->sprites[0];
      test2 = sys->sprites[sys->no_sprites-1];
      JLIB_LEAVE;
   }

   if(sys->no_frames){
      int i,frames=0;

      /* check loaded frames array */
      JLIB_ENTER("check sprite frame array");
      test3 = sys->sprite_data[0];
      test3 = sys->sprite_data[sys->no_frames-1];
      JLIB_LEAVE;

      for(i=0;i<sys->number_loaded;i++) {
          if(sys->sprite_data[i] != NULL) {
             frames++;
          }
      }

      if(frames != sys->number_loaded) {
         jlib_exit("Inconsistent loaded frames");
      }
   }

   if((sys->no_sprites == 0) && (sys->no_frames == 0)) {
      jlib_exit("sprite system has no frames or sprites");
   }
#endif

   JLIB_LEAVE;
#endif
}


/*+----------------------------------------------------------------------+*/
/*|Exhaustively test the validity of a single sprite.                    |*/
/*+----------------------------------------------------------------------+*/
void jlib_check_sprite(sprite_system *sys,int snum)
{
#ifdef JDEBUG
#ifdef JLIB_PARANOID
   UBYTE test;
#endif

   JLIB_ENTER("jlib_check_sprite");

   /* check sprite number */
   if((snum >= SPR_MAX_SPRITES(sys)) || (snum < 0)) {
      JLIB_SPRINTF("Sprite number %d invalid\n",snum);
      jlib_exit("Invalid sprite number");
   }

#ifdef JLIB_PARANOID
   if((sys->sprites == NULL) || (sys->sprites[snum] == NULL)) {
      JLIB_SPRINTF("Sprite %d,record is NULL",snum);
      jlib_exit(jlib_msg(JLIB_ENULL));
   }

   /* We don't check the sprite buffer, since it can legitimately be NULL */

   /* check frames */
   if(sys->no_frames) {
      JLIB_ENTER("check animframes");
      test = sys->sprites[snum]->animframes[0];
      test = sys->sprites[snum]->animframes[sys->no_frames-1];
      JLIB_LEAVE;
   }
#endif
   JLIB_LEAVE;
#endif
}

/*+----------------------------------------------------------------------+*/
/*|Exhaustively test the validity of a single sprite frame.              |*/
/*+----------------------------------------------------------------------+*/
void jlib_check_frame(sprite_system *sys,int frame)
{
#ifdef JDEBUG

   JLIB_ENTER("jlib_check_frame");

   /* check frame number */
   if((frame >= SPR_NUM_LOADED(sys)) || (frame < 0)) {
      JLIB_SPRINTF("Frame number %d invalid\n",frame);
      jlib_exit("Invalid frame number");
   }

#ifdef JLIB_PARANOID
   if((sys->sprite_data == NULL) || (sys->sprite_data[frame] == NULL)) {
      JLIB_SPRINTF("Frame %d,record is NULL",frame);
      jlib_exit(jlib_msg(JLIB_ENULL));
   }

   /* to do - check bounding rectangles,data,pattern */
#endif

   JLIB_LEAVE;
#endif
}
