/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */

#ifndef __JLIB_H__              /* only include this file once */

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stddef.h>
#include <limits.h>

/*+-----------------------------------------------------------------------+ */
/*|Target Specific Declarations                                           | */
/*+-----------------------------------------------------------------------+ */
#include <jconfig.h>
#define MOUSE_PRESENT 1
#define MOUSE_ABSENT  0
#include <jtarget.h>

/*+-----------------------------------------------------------------------+ */
/*|Types                                                                  | */
/*+-----------------------------------------------------------------------+ */
#ifdef JDEFAULT_TYPES

#ifndef UBYTE
#define UBYTE   unsigned char
#endif

#ifndef BYTE
#define BYTE    signed char
#endif

#ifndef USHORT
#define USHORT  unsigned short int
#endif

#ifndef SHORT
#define SHORT   short int
#endif

#ifndef ULONG
#define ULONG   unsigned long
#endif

#ifndef LONG
#define LONG    long
#endif

#endif


/*+-----------------------------------------------------------------------+ */
/*|Sprite Functions                                                       | */
/*+-----------------------------------------------------------------------+ */
/* return codes of function 'sprite_load' */
#define COULDNT_OPEN      1
#define TOO_MANY_IN_FILE  2
#define OUT_OF_MEMORY     3
#define SUCCESS           0

/* macros to get information about a sprite system */
#define SPR_MAX_X (64)          /* biggest possible sprite x */
#define SPR_MAX_Y (64)          /* biggest possible sprite y */
#define SPR_MAX_SPRITES(sys)    ((sys)->no_sprites)
#define SPR_MAX_FRAMES(sys)     ((sys)->no_frames)
#define SPR_NUM_LOADED(sys)     ((sys)->number_loaded)
#define SPR_NUM_ACTIVE(sys)     ((sys)->no_active)

/* macros to get information about a sprite */
#define SPR_X_SIZE(spr)         ((spr)->width)
#define SPR_Y_SIZE(spr)         ((spr)->height)
#define SPR_IS_ON(sys,num)      ((sys)->active_sprites[(num)])
#define SPR_GET_XPOS(sys,num)   ((sys)->sprites[(num)]->x)
#define SPR_GET_YPOS(sys,num)   ((sys)->sprites[(num)]->y)
#define SPR_GET_FRAME(ssys,spr) ssys->sprites[spr]->frame

/* macros to set information for a sprite */
#define SPR_SET_WIDTH(spr,w)    ((spr)->width=(w))
#define SPR_SET_HEIGHT(spr,h)   ((spr)->height=(h))
#define SPR_SET_XPOS(spr,x)     ((spr)->(x)=(x))
#define SPR_SET_YPOS(spr,y)     ((spr)->(y)=(y))

/* modes for sprite animation/movement */
#define SPR_MODE_WHOLE 0        /* moving in discrete steps */
#define SPR_MODE_FIXED 1        /* moving in fixed-point steps */

/* sprite data structures */
typedef struct{
     int x;                     /* current location */
     int y;

     UBYTE mode;                /* fixed point or whole steps */

     UBYTE speed;               /* movement speed (0 = not moving ) */
     UBYTE speed_counter;       /* internal counter */
     BYTE xinc;                 /* x and y increments when moving  */
     BYTE yinc;
     
     UBYTE animspeed;           /* animation speed (0 = not animating ) */
     UBYTE animspeed_counter;   /* internal counter */
     UBYTE noframes;            /* number of animation frames */
     int frame_count;           /* internal counter */
     int frame;                 /* current frame */
     int *animframes;           /* array of frames in animation */

     UBYTE *buffer;             /* holds the information under the sprite */
} sprite_record;


typedef struct{
     UBYTE width;               /* width of sprite */
     UBYTE height;              /* height of sprite */
     UBYTE *data;               /* x*y array of the data */
     UBYTE *pattern;            /* RLE storage pattern */
     UBYTE no_rects;            /* number of bounding rectangles */
     UBYTE *rect_coords;        /* coordinates of bounding rectangles */
} sprite_data_rec;


typedef struct{
     int no_sprites;            /* max number of sprites in system     */
     UBYTE *active_sprites;     /* array of flags for active sprites   */
     int no_active;             /* number of active sprites            */
     sprite_record **sprites;   /* array of the sprites themselves     */
     int no_frames;             /* num of sprite data frames in system */
     int number_loaded;         /* number of sprite frames loaded      */
     int biggest_rle;           /* biggest rle pattern used */
     sprite_data_rec **sprite_data;     /* array of frame data         */
} sprite_system;

/* this #define makes it easier to read the function declarations */
#define SS sprite_system

/* sprite functions */
SS *sprite_init (int max_sprites, int max_frames);
SS *sprite_free (SS * ssys);

UBYTE sprite_load (char *filename, SS * ssys);
UBYTE sprite_load_fp(FILE *fp, SS * ssys);

int sprite_save (char *filename, sprite_system * ssys);
int sprite_save_fp(FILE *sfile, sprite_system * ssys);

void sprite_turn_on (SS * ssys, int snum);
void sprite_turn_off (SS * ssys, int snum);
void sprite_set_xy (SS * ssys, int snum, int newx, int newy);
void sprite_set_an_frame (SS * ssys, int snum, int frame);

int sprite_find_first_free (SS * ssys);
int sprite_find_first_frame(SS * ssys);

void sprite_set_mode(SS *ssys,int snum,int mode);
void sprite_set_move_info(SS *ssys,int snum,UBYTE speed,BYTE xinc,BYTE yinc);
void sprite_set_anim_info(SS *ssys,int snum,UBYTE anspeed,UBYTE noframes,int *animpat);

void sprite_update_anim_and_move (SS * ssys, int snum);
void sprite_update_all_anim_and_move (SS * ssys);
void sprite_do_all_anim_and_move_n_times (SS * ssys, int n);

int  sprite_do_intersect (SS * ssys, int s1, int s2);
int  sprites_do_intersect (SS *sys1, int s1,SS *sys2, int s2);
void sprite_optimise_mem(sprite_system * ssys);
void generate_RLE (sprite_data_rec * srec);
void sprite_add_rect(sprite_system *ssys,int frame,UBYTE x1,UBYTE y1,UBYTE x2,UBYTE y2);
void sprites_kill_buffers(sprite_system * ssys);

extern int global_biggest_rle;


/*+-----------------------------------------------------------------------+ */
/*|Offscreen Buffer Functions                                             | */
/*+-----------------------------------------------------------------------+ */
/* macros to get information about a buffer */
#define B_X_SIZE(buff_ptr)      ((buff_ptr)->width)
#define B_Y_SIZE(buff_ptr)      ((buff_ptr)->height)
#define B_MAX_X(buff_ptr)       (((buff_ptr)->width)-1)
#define B_MAX_Y(buff_ptr)       (((buff_ptr)->height)-1)
#define B_SIZE(buff_ptr)        (((buff_ptr)->width)*((buff_ptr)->height))
#define B_BUFF_PTR(buff_ptr)    ((buff_ptr)->buffer)
#define B_OFFSET(buff_ptr,n)    ((buff_ptr)->offset[n])

/* buffer data structure */
typedef struct{
     int width;                 /* width of buffer  */
     int height;                /* height of buffer */
     UBYTE *buffer;             /* the buffer data  */
     UBYTE **offset;            /* line offsets */
} buffer_rec;

/* this #define makes it easier to read the functions in this header */
#define BR buffer_rec

/* buffer functions */
BR *buff_init(int width, int height);
BR *buff_free(BR *buff);

void buff_clear(BR *buff);
void buff_fill(BR *buff, UBYTE colour);
void buff_remap_colors (buffer_rec * buff, UBYTE *remap);

void buff_blit_buff_to(BR *dbuff,int dbfx,int dbfy,BR* sbuff,int sbx1,int sby1,int sbx2,int sby2);
void buff_blit_buff_toNC(BR *dbuff,int dbfx,int dbfy,BR *sbuff,int sbx1,int sby1,int sbx2,int sby2);

void buff_stencil_buff_to (BR * dbuff, int dbfx, int dbfy,
    BR *sbuff, int sbx1, int sby1, int sbx2, int sby2);

void buff_stencil_buff_toNC (BR * dbuff, int dbfx, int dbfy,
    BR *sbuff, int sbx1, int sby1, int sbx2, int sby2);


void buff_draw_point   (BR *buff, int x, int y, UBYTE c);
void buff_draw_pointNC (BR *buff, int x, int y, UBYTE c);
UBYTE buff_get_point   (BR *buff, int x, int y);
UBYTE buff_get_pointNC (BR *buff, int x, int y);

void buff_draw_box  (BR *buff, int x1, int y1, int x2, int y2, UBYTE c);
void buff_draw_boxNC(BR *buff, int x1, int y1, int x2, int y2, UBYTE c);

void buff_draw_rect  (BR *buff, int x1, int y1, int x2, int y2, UBYTE c);
void buff_draw_rectNC(BR *buff, int x1, int y1, int x2, int y2, UBYTE c);

void buff_draw_char  (BR *buff, UBYTE letter, int x, int y, UBYTE colour);
void buff_draw_charNC(BR *buff, UBYTE letter, int x, int y, UBYTE colour);

void buff_draw_string  (BR *buff, char *string, int x, int y, UBYTE colour);
void buff_draw_stringNC(BR *buff, char *string, int x, int y, UBYTE colour);

void buff_draw_line  (BR *buff, int x, int y, int x2, int y2, UBYTE c);
void buff_draw_lineNC(BR *buff, int x, int y, int x2, int y2, UBYTE c);

void buff_draw_h_line  (BR *buff, int x1, int y1, int x2, UBYTE c);
void buff_draw_h_lineNC(BR *buff, int x1, int y1, int x2, UBYTE c);

void buff_draw_v_line  (BR *buff, int x1, int y1, int y2, UBYTE c);
void buff_draw_v_lineNC(BR *buff, int x1, int y1, int y2, UBYTE c);

void buff_draw_ellipse  (BR *buff, int x0, int y0, int a, int b, UBYTE c);
void buff_draw_ellipseNC(BR *buff, int x0, int y0, int a, int b, UBYTE c);

void buff_filled_ellipse  (BR *buff, int x0, int y0, int a, int b, UBYTE c);
void buff_filled_ellipseNC(BR *buff, int x0, int y0, int a, int b, UBYTE c);

#define buff_draw_circle(b,x0,y0,d,c) buff_draw_ellipse(b,x0,y0,d,d,c);
#define buff_draw_circleNC(b,x0,y0,d,c) buff_draw_ellipseNC(b,x0,y0,d,d,c);

#define buff_filled_circle(b,x0,y0,d,c) buff_filled_ellipse((b),(x0),(y0),(d),(d),(c));
#define buff_filled_circleNC(b,x0,y0,d,c) buff_filled_ellipseNC((b),(x0),(y0),(d),(d),(c));

void buff_draw_triangle  (BR *buff,int x1,int y1,int x2,int y2,int x3,int y3,UBYTE col);
void buff_draw_triangleNC(BR *buff,int x1,int y1,int x2,int y2,int x3,int y3,UBYTE col);

void buff_filled_triangle  (BR *buff,int x1,int y1,int x2,int y2,int x3,int y3,UBYTE col );
void buff_filled_triangleNC(BR *buff,int x1,int y1,int x2,int y2,int x3,int y3,UBYTE col);

void buff_convex_poly  (BR *buff, int n, int *x, int *y,UBYTE col);
void buff_convex_polyNC(BR *buff, int n, int *x, int *y,UBYTE col);

void buff_hollow_poly  (BR *buff, int n, int *x, int *y,UBYTE col);
void buff_hollow_polyNC(BR *buff, int n, int *x, int *y,UBYTE col);

void buff_scale_full_buff_toNC(BR *dest,int x1,int y1,int x2,int y2,BR *src);
void buff_scale_full_buff_to  (BR *dest,int x1,int y1,int x2,int y2,BR *src);

void buff_scale_buff_toNC(BR *dest,int dx1,int dy1,int dx2,int dy2,BR *src,int x1,int y1,int x2,int y2);
void buff_scale_buff_to  (BR *dest,int dx1,int dy1,int dx2,int dy2,BR *src,int x1,int y1,int x2,int y2);

void buff_draw_sprite     (SS *ssys, int snum, BR *obuff);
void buff_draw_spriteNC   (SS *ssys, int snum, BR *obuff);
void buff_draw_all_sprites(SS *ssys, BR *obuff);

void buff_save_sprite     (SS *ssys, int snum, BR *obuff);
void buff_save_spriteNC   (SS *ssys, int snum, BR *obuff);
void buff_save_all_sprites(SS *ssys, BR *obuff);

void buff_rest_sprite     (SS *ssys, int snum, BR *obuff);
void buff_rest_spriteNC   (SS *ssys, int snum, BR *obuff);
void buff_rest_all_sprites(SS *ssys, BR *obuff);

void buff_stencil_sprite  (SS *ssys, int frame, BR *obuff,int x, int y );
void buff_stencil_spriteNC(SS *ssys, int frame, BR *obuff,int x, int y );

void buff_stencil_sprite_color  (SS *ssys, int frame, BR *obuff,int x, int y, UBYTE col);
void buff_stencil_sprite_colorNC(SS *ssys, int frame, BR *obuff,int x, int y, UBYTE col);

void buff_stencil_sprite_buff  (SS *ssys,int frame,BR *obuff,int x, int y,BR * sbuff);
void buff_stencil_sprite_buffNC  (SS *ssys,int frame,BR *obuff,int x, int y,BR * sbuff);

void buff_stamp_sprite  (SS *ssys, int frame, BR *obuff,int x, int y );
void buff_stamp_spriteNC(SS *ssys, int frame, BR *obuff,int x, int y );

void buff_stamp_sprite_color  (SS *ssys, int frame, BR *obuff,int x, int y, UBYTE col);
void buff_stamp_sprite_colorNC(SS *ssys, int frame, BR *obuff,int x, int y, UBYTE col);

void buff_stamp_sprite_buff  (SS *ssys,int frame,BR *obuff,int x, int y,BR * sbuff);
void buff_stamp_sprite_buffNC  (SS *ssys,int frame,BR *obuff,int x, int y,BR * sbuff);

void sprite_build_from_buff(SS *ssys,int frame,BR *bf,int x1,int y1,int x2,int y2);

/*+-----------------------------------------------------------------------+ */
/*|Image Functions                                                        | */
/*+-----------------------------------------------------------------------+ */
#define IMG_WIDTH(img_ptr)          (img_ptr->width)
#define IMG_HEIGHT(img_ptr)         (img_ptr->height)
#define IMG_MAX_X(img_ptr)          (img_ptr->width-1)
#define IMG_MAX_Y(img_ptr)          (img_ptr->height-1)
#define IMG_PALETTE(img_ptr)        (img_ptr->palette)
#define IMG_DATA_PTR(img_ptr)       (img_ptr->data)


typedef struct{
     int width;
     int height;
     UBYTE *data;
     UBYTE *palette;
     buffer_rec *buff;
} image;

image *image_free (image *img);
image *image_load_pcx (char *filename);
image *image_load_pcx_fp (FILE *fp);
image *image_load_jlb (char *filename);
void   image_setup(image *image);

int image_save_jlb(char *filename,BR *buff,UBYTE *pal);
int image_save_pcx(char *filename,BR *buff,UBYTE *pal); 


void buff_blit_img_to (BR *dbuff, int dbfx, int dbfy,
	   image *img, int imx1, int imy1, int imx2, int imy2);
void buff_blit_img_toNC (BR *dbuff, int dbfx, int dbfy,
	   image *img, int imx1, int imy1, int imx2, int imy2);


/*+-----------------------------------------------------------------------+ */
/*|Screen Functions.                                                      | */
/*+-----------------------------------------------------------------------+ */
int  screen_set_video_mode(void);
void screen_restore_video_mode(void);

void screen_set_page(int page);
int  screen_get_page(void);
void screen_show_page(int page);

void screen_clear(void);
void screen_fill(UBYTE colour);
void screen_wait_vsync(void);

void screen_put_pal(UBYTE col, UBYTE red, UBYTE green, UBYTE blue);
void screen_block_set_pal(UBYTE *pal);
void screen_blank_pal(void);
void screen_fade_in_pal(UBYTE *pal,unsigned int delaytime);
void screen_fade_out_pal(UBYTE *pal,unsigned int delaytime);
void screen_fade_to_pal(UBYTE *pal1, UBYTE *pal2, unsigned int delaytime);

void screen_blit_buff_to(int x,int y,buffer_rec *sbuff,
			 int sbx1, int sby1,int sbx2, int sby2);
void screen_blit_buff_toNC(int x,int y,buffer_rec *sbuff,
			   int sbx1, int sby1,int sbx2, int sby2);
JINLINE void screen_blit_fs_buffer(buffer_rec *sbuff);


/*+-----------------------------------------------------------------------+ */
/*|Non-Screen palette functions.                                          | */
/*+-----------------------------------------------------------------------+ */
UBYTE *pal_init(void);
UBYTE *pal_load(char *fname);
UBYTE *pal_load_fp(FILE *fp);
UBYTE *pal_get_default(void);
void   pal_copy(UBYTE *pal1,UBYTE *pal2);
void   pal_to_grey(UBYTE *pal);
void   pal_to_red(UBYTE *pal);
void   pal_to_green(UBYTE *pal);
void   pal_to_blue(UBYTE *pal);
UBYTE  pal_closest_color_rgb(UBYTE *pal,UBYTE r,UBYTE g,UBYTE b);
UBYTE *pal_create_remap_index(UBYTE *pal1,UBYTE *pal2);


/*+-----------------------------------------------------------------------+ */
/*|Text Functions                                                         | */
/*+-----------------------------------------------------------------------+ */
#if SCREEN_WIDTH == 320
#define CHAR_WIDTH(chr)  4
#define CHAR_HEIGHT(chr) 8
#define HAS_FONT(chr)   (((UBYTE)chr>=32) && ((UBYTE)chr<=126))

#else
/* 640x480 and above modes */
#define CHAR_WIDTH(chr)  8
#define CHAR_HEIGHT(chr) 19
#define HAS_FONT(chr)   (((UBYTE)chr>=32) && ((UBYTE)chr<=126))
#endif

extern UBYTE __font_data[];


/*+-----------------------------------------------------------------------+ */
/*|Miscellanious Functions                                                | */
/*+-----------------------------------------------------------------------+ */
typedef void (*exit_function)();
extern exit_function __jlib_exit_function;
extern char *__jlib_app_title;
extern char version_string[];

/* return library version information */
float jlib_return_version_number(void);
char *jlib_return_version_string(void);

/* Set the title of the JLib application */
void screen_set_app_title(char *title);

/* pop up a screen detailing the jlib target,  version etc */
void popup_about(UBYTE back_col,UBYTE front_col);

/* pop up a message with an OK button */
void popup_info(char *message,UBYTE back_col,UBYTE front_col);

/* set up a user defined exit function for when a library error occurs */
void jlib_set_error_handler(exit_function funct);

/* exit due to error with a message */
void jlib_exit(char *message);

/* default exit processing */
void jlib_exit_details(char *message);

/* setup millisecond clock */
void uclock_init(void);

/* read millisecond timing */
unsigned int uclock_read(void);

/* these two are for elementary types (int,short,char) where size <= 4 */
int jio_read_elementary_type(FILE *fp,void *type,size_t size);
int jio_write_elementary_type(FILE *fp,void *type,size_t size);

/* edge table for poly/triangle drawing */
#define MAX_YRES 1280
#define MAX_XRES 16384
extern int left[MAX_YRES], right[MAX_YRES];

/*+-----------------------------------------------------------------------+ */
/*|Debugging Functions                                                    | */
/*+-----------------------------------------------------------------------+ */
#define JLIB_DEBUG_STACK_NUM 256
#define JLIB_EMALLOC 1                    /* error messages */
#define JLIB_ENULL   2 
#define JLIB_EFILE   3
#define JLIB_EKB     4

char *jlib_msg(int error);
void jlib_check_buffer(buffer_rec *buff);
void jlib_check_sprite_system(sprite_system *sys);
void jlib_check_sprite(sprite_system *sys,int snum);
void jlib_check_frame(sprite_system *sys,int fnum);

#ifdef JHANDLE_SIGNALS
void __jlib_setup_signals(void);
#define JLIB_SIGNAL_SETUP __jlib_setup_signals();
#else
#ifdef JHANDLE_SIMPLE_SIGNALS
void __jlib_setup_signals(void);
#define JLIB_SIGNAL_SETUP __jlib_setup_signals();
#else
#define JLIB_SIGNAL_SETUP
#endif
#endif

extern char JLIB_MESSAGE_STRING[81];      /* see misc/ident.c */
extern  char __jlib_debug_level;  /* in misc/ident.c */
extern  int  __jlib_stack_end;
extern  char *__jlib_stack[JLIB_DEBUG_STACK_NUM];

#ifndef JLIB_PROCESS
#define JLIB_PROCESS
#endif

#ifdef JDEFAULT_DEBUG
#ifdef JDEBUG
#define JLIB_DEBUG_OFF    {__jlib_debug_level=0;}
#define JLIB_DEBUG_ON     {__jlib_debug_level=1;}
#define JLIB_DEBUG_TRACE  {__jlib_debug_level=2;}
#define JLIB_PRINT_DEBUG_INFO(x) if(__jlib_debug_level){puts(x);fflush(stdout);}
#define JLIB_PRINT_MESSAGE_STRING if(__jlib_debug_level){puts(JLIB_MESSAGE_STRING);fflush(stdout);}
#define JLIB_SPRINTF(x,y) if(__jlib_debug_level){                      \
                             sprintf(JLIB_MESSAGE_STRING,x,y);         \
                             puts(JLIB_MESSAGE_STRING);fflush(stdout); \
                          }
#define JLIB_ENTER(x)  if(__jlib_stack_end>=(JLIB_DEBUG_STACK_NUM-1)){ \
                          jlib_exit("Debug Stack Overflow.");      \
                       }else{                                      \
                          __jlib_stack[__jlib_stack_end]=x;        \
                          __jlib_stack_end++;                      \
                          if(__jlib_debug_level>1){                \
                             int zx=__jlib_debug_level;            \
                             while(zx--){printf("  ");};           \
                             puts(x);fflush(stdout);               \
                          }                                        \
                       }                                           \
                       JLIB_PROCESS
		
#define JLIB_LEAVE     if(__jlib_stack_end<=0){                    \
                          jlib_exit("Debug Stack Underflow.");     \
                       }else{                                      \
                          __jlib_stack_end--;                      \
                          if(__jlib_debug_level>1){                \
                             int zx=__jlib_debug_level;            \
                             while(zx--){printf("  ");};           \
                             puts("Leave.");fflush(stdout);        \
                          }                                        \
                       }                                           \
                       JLIB_PROCESS                   

#define JLIB_DUMP_STACK {int __jlib_i=0;                           \
                         if(__jlib_stack_end==0){                  \
                            puts("Stack Is Empty.");               \
                         }else{                                    \
                            puts("Stack Dump:");                   \
                            for(;__jlib_i<__jlib_stack_end;__jlib_i++){ \
                             puts(__jlib_stack[__jlib_i]);         \
                            }                                      \
                            puts("End of Stack Dump.");            \
                         }}
#else
#define JLIB_DEBUG_ON
#define JLIB_DEBUG_OFF
#define JLIB_DEBUG_TRACE
#define JLIB_SPRINTF(x,y)
#define JLIB_SPRINTF_END
#define JLIB_PRINT_DEBUG_INFO(x)
#define JLIB_PRINT_MESSAGE_STRING
#define JLIB_ENTER(x) JLIB_PROCESS
#define JLIB_LEAVE JLIB_PROCESS
#define JLIB_DUMP_STACK
#endif /* ifdef JDEBUG */
#endif /* ifdef JDEFAULT_DEBUG */


/*+-----------------------------------------------------------------------+ */
/*|Input Functions.                                                       | */
/*+-----------------------------------------------------------------------+ */
int mouse_present (void);
void mouse_show_pointer (void);
void mouse_hide_pointer (void);

void mouse_get_status (int *x_pos, int *y_pos, int *b_status);
void mouse_set_status (int x, int y);
void mouse_closedown (void);

void   kb_init(void); 
void   kb_closedown(void);
int    kb_keydown(int key);         
int    kb_key_hit(void);            
int    kb_ext_key_hit(void);            
char   kb_get_next_key(void);
USHORT kb_get_next_ext_key(void);
USHORT kb_get_next_code(void);
void   kb_clear(void);

int joystick_count(void);
int joystick_init(int which);
void joystick_closedown(int which);
void joystick_get_status(int which,int *x_axis,int *y_axis,int *b_status);
void joystick_calibrate(int which,int l,int r,int t,int b,int x_cen,int y_cen);
int  joystick_get_direction(int which,int x_axis,int y_axis);


/*+-----------------------------------------------------------------------+ */
/*|Key Definitions                                                        | */
/*+-----------------------------------------------------------------------+ */
#define KEY_ESC         1
#define KEY_1           2
#define KEY_2           3
#define KEY_3           4
#define KEY_4           5
#define KEY_5           6
#define KEY_6           7
#define KEY_7           8
#define KEY_8           9
#define KEY_9           10
#define KEY_0           11
#define KEY_MINUS       12
#define KEY_EQUALS      13
#define KEY_BACKSPACE   14
#define KEY_TAB         15
#define KEY_Q           16
#define KEY_W           17
#define KEY_E           18
#define KEY_R           19
#define KEY_T           20
#define KEY_Y           21
#define KEY_U           22
#define KEY_I           23
#define KEY_O           24
#define KEY_P           25
#define KEY_OPENBRACE   26
#define KEY_CLOSEBRACE  27
#define KEY_ENTER       28
#define KEY_CONTROL     29
#define KEY_A           30
#define KEY_S           31
#define KEY_D           32
#define KEY_F           33
#define KEY_G           34
#define KEY_H           35
#define KEY_J           36
#define KEY_K           37
#define KEY_L           38
#define KEY_COLON       39
#define KEY_QUOTE       40
#define KEY_TILDE       41
#define KEY_SHIFT       42
#define KEY_LSHIFT      42
#define KEY_Z           44
#define KEY_X           45
#define KEY_C           46
#define KEY_V           47
#define KEY_B           48
#define KEY_N           49
#define KEY_M           50
#define KEY_COMMA       51
#define KEY_STOP        52
#define KEY_SLASH       53
#define KEY_RSHIFT      54
#define KEY_ASTERISK    55
#define KEY_ALT         56
#define KEY_SPACE       57
#define KEY_CAPSLOCK    58
#define KEY_F1          59
#define KEY_F2          60
#define KEY_F3          61
#define KEY_F4          62
#define KEY_F5          63
#define KEY_F6          64
#define KEY_F7          65
#define KEY_F8          66
#define KEY_F9          67
#define KEY_F10         68
#define KEY_NUMLOCK     69
#define KEY_SCRLOCK     70
#define KEY_HOME        71
#define KEY_UP          72
#define KEY_PGUP        73
#define KEY_LEFT        75
#define KEY_RIGHT       77
#define KEY_END         79
#define KEY_DOWN        80
#define KEY_PGDN        81
#define KEY_INSERT      82
#define KEY_DEL         83

/* Special values in the keyboard buffer */
#define KB_IS_SPECIAL_CHAR(x) ((x)>210)
#define KB_NO_CODE 16383 /* no code in kb buffer */
#define KB_WAS_RELEASED(x) ((x) & (1<<15)) /* was a key in the buffer released */

/* extended keys pressed in the input buffer */
#define KB_LSHIFT 211
#define KB_RSHIFT 212
#define KB_CTRL   213
#define KB_ALT    214
#define KB_LEFT   215
#define KB_RIGHT  216
#define KB_DOWN   217
#define KB_UP     218
#define KB_F1     219
#define KB_F2     220
#define KB_F3     221
#define KB_F4     222
#define KB_F5     223
#define KB_F6     224
#define KB_F7     225
#define KB_F8     226
#define KB_F9     227
#define KB_F10    228
#define KB_CAPS   229

#define KB_LSHIFT_UP (KB_LSHIFT | (1<<15))
#define KB_RSHIFT_UP (KB_RSHIFT | (1<<15))
#define KB_CTRL_UP   (KB_CTRL   | (1<<15))
#define KB_ALT_UP    (KB_ALT    | (1<<15))
#define KB_LEFT_UP   (KB_LEFT   | (1<<15))
#define KB_RIGHT_UP  (KB_RIGHT  | (1<<15))
#define KB_DOWN_UP   (KB_DOWN   | (1<<15))
#define KB_UP_UP     (KB_UP     | (1<<15))
#define KB_F1_UP     (KB_F1     | (1<<15))
#define KB_F2_UP     (KB_F2     | (1<<15))
#define KB_F3_UP     (KB_F3     | (1<<15))
#define KB_F4_UP     (KB_F4     | (1<<15))
#define KB_F5_UP     (KB_F5     | (1<<15))
#define KB_F6_UP     (KB_F6     | (1<<15))
#define KB_F7_UP     (KB_F7     | (1<<15))
#define KB_F8_UP     (KB_F8     | (1<<15))
#define KB_F9_UP     (KB_F9     | (1<<15))
#define KB_F10_UP    (KB_F10    | (1<<15))
#define KB_CAPS_UP   (KB_CAPS   | (1<<15))

/* extended attributes for kb_get_next_ext_key */
#define EXT_SHIFT (1<<15)
#define EXT_CTRL  (1<<14)
#define EXT_ALT   (1<<13)
#define EXT_UP    (1<<12)
#define EXT_DOWN  (1<<11)
#define EXT_LEFT  (1<<10)
#define EXT_RIGHT (1<<9)

/* joystick directions */
#define JOY_CENTER    0
#define JOY_NORTH     1
#define JOY_EAST      2
#define JOY_SOUTH     4
#define JOY_WEST      8
#define JOY_NORTHEAST (JOY_NORTH | JOY_EAST)
#define JOY_NORTHWEST (JOY_NORTH | JOY_WEST)
#define JOY_SOUTHEAST (JOY_SOUTH | JOY_EAST)
#define JOY_SOUTHWEST (JOY_SOUTH | JOY_WEST)

#undef BR

#ifdef JLIB_SAFE_SWAP
#define JLIB_SWAP(a,b) { int temp,x,y; x=(a);y=(b);temp=x;x=y;y=temp; }
#else
#define JLIB_SWAP(a,b) { a^=b;b^=a;a^=b; }
#endif

/* clipping codes */
#define C_LEFT  (1<<0)
#define C_RIGHT (1<<1)
#define C_UP    (1<<2)
#define C_DOWN  (1<<3)

/* Note: Assumes len <= 16 */
#define MEM_STORE_SHORT(d,v,l) { \
 switch(l){ \
     case 16: d[15] = v; \
     case 15: d[14] = v; \
     case 14: d[13] = v; \
     case 13: d[12] = v; \
     case 12: d[11] = v; \
     case 11: d[10] = v; \
     case 10: d[9]  = v; \
     case 9:  d[8]  = v; \
     case 8:  d[7]  = v; \
     case 7:  d[6]  = v; \
     case 6:  d[5]  = v; \
     case 5:  d[4]  = v; \
     case 4:  d[3]  = v; \
     case 3:  d[2]  = v; \
     case 2:  d[1]  = v; \
     case 1:  d[0]  = v; \
 } \
}

/* Note: assumes l > 16 and addresses fit into 32 bit ULONGS */
#define MEM_STORE_LONG(d,v,l) \
{ \
  ULONG *__dst=(ULONG *)((ULONG)d & 0xFFFFFFFC); \
  UBYTE *__end; \
  ULONG __len; \
  ULONG __val; \
  ULONG __rem = l; \
  __val = (ULONG)v + ((ULONG)v << 8) + ((ULONG)v << 16) + ((ULONG)v << 24); \
  \
  /* Take care of any non-aligned starting pixels */ \
  switch(((ULONG)d & 0x3)){ \
     case 1: *(UBYTE *)((ULONG)d+2) = v; __rem--; \
     case 2: *(UBYTE *)((ULONG)d+1) = v; __rem--; \
     case 3: *(UBYTE *) (ULONG)d    = v; __rem--; \
             ++__dst; \
  } \
  __len = __rem >> 2; \
  /* Main unrolled loop of ULONG stores */ \
  while(__len>16){ \
      __dst[0]  = __val; __dst[1]  = __val; __dst[2]  = __val; __dst[3]  = __val; \
      __dst[4]  = __val; __dst[5]  = __val; __dst[6]  = __val; __dst[7]  = __val; \
      __dst[8]  = __val; __dst[9]  = __val; __dst[10] = __val; __dst[11] = __val; \
      __dst[12] = __val; __dst[13] = __val; __dst[14] = __val; __dst[15] = __val; \
      __len-=16; __dst+=16; \
  } \
  /* write the remainder of ULONGS */ \
  switch(__len){ \
      case 16: __dst[15] = __val; \
      case 15: __dst[14] = __val; \
      case 14: __dst[13] = __val; \
      case 13: __dst[12] = __val; \
      case 12: __dst[11] = __val; \
      case 11: __dst[10] = __val; \
      case 10: __dst[9]  = __val; \
      case 9:  __dst[8]  = __val; \
      case 8:  __dst[7]  = __val; \
      case 7:  __dst[6]  = __val; \
      case 6:  __dst[5]  = __val; \
      case 5:  __dst[4]  = __val; \
      case 4:  __dst[3]  = __val; \
      case 3:  __dst[2]  = __val; \
      case 2:  __dst[1]  = __val; \
      case 1:  __dst[0]  = __val; \
  } \
  __end = (UBYTE *)((ULONG)(__dst += __len)); \
  /* Take care of any non-aligned ending pixels */ \
  switch(__rem & 0x3){ \
     case 3: __end[2] = v; \
     case 2: __end[1] = v; \
     case 1: __end[0] = v; \
  } \
}

/* Note: Assumes len <= 16 */
#define SPR_STAMP_COLOR(d,v,l) { \
 switch(l){ \
     case 31: d[14] = v; \
     case 30: d[13] = v; \
     case 29: d[12] = v; \
     case 28: d[11] = v; \
     case 27: d[10] = v; \
     case 26: d[9]  = v; \
     case 25: d[8]  = v; \
     case 24: d[7]  = v; \
     case 23: d[6]  = v; \
     case 22: d[5]  = v; \
     case 21: d[4]  = v; \
     case 20: d[3]  = v; \
     case 19: d[2]  = v; \
     case 18: d[1]  = v; \
     case 17: d[0]  = v; \
     case 16: break;     \
     case 15: d[14] = 0; \
     case 14: d[13] = 0; \
     case 13: d[12] = 0; \
     case 12: d[11] = 0; \
     case 11: d[10] = 0; \
     case 10: d[9]  = 0; \
     case 9:  d[8]  = 0; \
     case 8:  d[7]  = 0; \
     case 7:  d[6]  = 0; \
     case 6:  d[5]  = 0; \
     case 5:  d[4]  = 0; \
     case 4:  d[3]  = 0; \
     case 3:  d[2]  = 0; \
     case 2:  d[1]  = 0; \
     case 1:  d[0]  = 0; \
 } \
}

#define SPR_STENCIL_COLOR(d,v,l) { \
 switch(l){ \
     case 31: d[14] = v; \
     case 30: d[13] = v; \
     case 29: d[12] = v; \
     case 28: d[11] = v; \
     case 27: d[10] = v; \
     case 26: d[9]  = v; \
     case 25: d[8]  = v; \
     case 24: d[7]  = v; \
     case 23: d[6]  = v; \
     case 22: d[5]  = v; \
     case 21: d[4]  = v; \
     case 20: d[3]  = v; \
     case 19: d[2]  = v; \
     case 18: d[1]  = v; \
     case 17: d[0]  = v; \
 } \
}

/* Note: Assumes len <= 16 */
#define SPR_STAMP_COPY(d,s,l) { \
 switch(l){ \
     case 31: d[14] = s[14]; \
     case 30: d[13] = s[13]; \
     case 29: d[12] = s[12]; \
     case 28: d[11] = s[11]; \
     case 27: d[10] = s[10]; \
     case 26: d[9]  = s[9]; \
     case 25: d[8]  = s[8]; \
     case 24: d[7]  = s[7]; \
     case 23: d[6]  = s[6]; \
     case 22: d[5]  = s[5]; \
     case 21: d[4]  = s[4]; \
     case 20: d[3]  = s[3]; \
     case 19: d[2]  = s[2]; \
     case 18: d[1]  = s[1]; \
     case 17: d[0]  = s[0]; \
     case 16: break;     \
     case 15: d[14] = 0; \
     case 14: d[13] = 0; \
     case 13: d[12] = 0; \
     case 12: d[11] = 0; \
     case 11: d[10] = 0; \
     case 10: d[9]  = 0; \
     case 9:  d[8]  = 0; \
     case 8:  d[7]  = 0; \
     case 7:  d[6]  = 0; \
     case 6:  d[5]  = 0; \
     case 5:  d[4]  = 0; \
     case 4:  d[3]  = 0; \
     case 3:  d[2]  = 0; \
     case 2:  d[1]  = 0; \
     case 1:  d[0]  = 0; \
 } \
}

/* Note: Assumes len <= 16 */
#define SPR_STENCIL_COPY(d,s,l) { \
 switch(l){ \
     case 31: d[14] = s[14]; \
     case 30: d[13] = s[13]; \
     case 29: d[12] = s[12]; \
     case 28: d[11] = s[11]; \
     case 27: d[10] = s[10]; \
     case 26: d[9]  = s[9]; \
     case 25: d[8]  = s[8]; \
     case 24: d[7]  = s[7]; \
     case 23: d[6]  = s[6]; \
     case 22: d[5]  = s[5]; \
     case 21: d[4]  = s[4]; \
     case 20: d[3]  = s[3]; \
     case 19: d[2]  = s[2]; \
     case 18: d[1]  = s[1]; \
     case 17: d[0]  = s[0]; \
 } \
}

#ifdef __cplusplus
}
#endif 


#define __JLIB_H__
#endif /* #ifndef __JLIB_H__ */
