/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <jlib.h>


/*+------------------------------------------------------------------------+*/
/*|Draw part of src scaled into dest at a given pos without clipping.      |*/
/*+------------------------------------------------------------------------+*/
void buff_scale_buff_toNC(buffer_rec *dest,int dx1,int dy1,int dx2,int dy2,buffer_rec *src,int x1,int y1,int x2,int y2)
{
  int rxs,xx,yy,y_step,copy_method,width,height;
  int src_width,src_height,dst_width=B_X_SIZE(dest),src_x=B_X_SIZE(src);
  UBYTE *dst_ptr=B_BUFF_PTR(dest),*src_ptr=B_BUFF_PTR(src),*prev_src_ptr=NULL;

  JLIB_ENTER("buff_scale_buff_toNC");

  /* sort coordinates if needed */
  if (x2 < x1) {
     JLIB_SWAP(x1, x2);
  }
  if (y2 < y1) {
     JLIB_SWAP(y1, y2);
  }
  if (dx2 < dx1) {
     JLIB_SWAP(dx1, dx2);
  }
  if (dy2 < dy1) {
     JLIB_SWAP(dy1, dy2);
  }

  src_width=x2-x1+1;
  src_height=y2-y1+1;
  
  dst_ptr+=(dy1*dst_width+dx1);
  src_ptr+=(y1*src_x+x1);

  width=dx2-dx1+1;
  height=dy2-dy1+1;
  
  rxs=(width+src_width-1);

  if(src_height < height){
    y_step=src_height;
  }
  else{
    y_step=height-1;
  }

  copy_method=get_best_copy_method(src_ptr,dst_ptr,width);

  for(yy=height;yy>0;yy--){
     UBYTE *dst_sav=dst_ptr;

     y_step-=src_height;

     if( y_step >0 ){
       switch(copy_method){
          case ALIGNMENT_STATE_1:
               MEM_COPY_ALIGN_1(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_2:  
               MEM_COPY_ALIGN_2(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_3:
               MEM_COPY_ALIGN_3(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_4:
               MEM_COPY_ALIGN_4(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_5:
               MEM_COPY_ALIGN_5(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_6:
               MEM_COPY_ALIGN_6(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_7:
               MEM_COPY_ALIGN_7(prev_src_ptr,dst_ptr,width);
               break;
          case ALIGNMENT_STATE_8:
               MEM_COPY_ALIGN_8(prev_src_ptr,dst_ptr,width);
               break;
       }
     }
     else{
       int x_step=rxs;
       UBYTE *src_sav=src_ptr;

       prev_src_ptr=dst_ptr;

       for(xx=width;xx>0;xx--){
          x_step-=src_width;
          if(x_step< 0){
             do{
                src_ptr++;
                x_step+=width;
             }while ( x_step <0);
          }
          *dst_ptr= *src_ptr;
          dst_ptr++;
       }
       src_ptr=src_sav;
       do{
          src_ptr+=src_x;
          y_step+=height;
       }while (y_step<0);
    }
    dst_ptr=(dst_sav+dst_width);
  }

  JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*|Draw part of src scaled into dest at a given pos with clipping.         |*/
/*+------------------------------------------------------------------------+*/
void buff_scale_buff_to(buffer_rec *dest,int dx1,int dy1,int dx2,int dy2,buffer_rec *src,int x1,int y1,int x2,int y2)
{
  int src_width,src_height,dst_width,dst_height,diff;
  int src_max_x,src_max_y,dst_max_x,dst_max_y;
    
  JLIB_ENTER("buff_scale_buff_to");

#ifndef JLIB_PRODUCTION
   jlib_check_buffer(src);
   jlib_check_buffer(dest);
#endif

  /* sort coordinates if needed */
  if (x2 < x1) {
     JLIB_SWAP(x1, x2);
  }
  if (y2 < y1) {
     JLIB_SWAP(y1, y2);
  }
  if (dx2 < dx1) {
     JLIB_SWAP(dx1, dx2);
  }
  if (dy2 < dy1) {
     JLIB_SWAP(dy1, dy2);
  }

  dst_max_x=B_MAX_X(dest);
  dst_max_y=B_MAX_Y(dest);
  src_max_x=B_MAX_X(src);
  src_max_y=B_MAX_Y(src);
    
  src_width=x2-x1+1;
  src_height=y2-y1+1;
  dst_width=dx2-dx1+1;
  dst_height=dy2-dy1+1;

  /* handle trivial rejection cases */
  if((dx1>dst_max_x) || (dy1>dst_max_y) || (dx2<0)|| (dy2<0)){
     JLIB_LEAVE;
     return;
  }

  if((x1>src_max_x) || (y1>src_max_y) || (x2<0)|| (y2<0)){
     JLIB_LEAVE;
     return;
  }

  /* if src area is clipped, shrink dest area by the appropriate amount */
  if(x1<0){
     diff=(-x1);
        dx1+=((diff*dst_width)/src_width);
     x1=0;
  }
  if(x2>src_max_x){
     diff=x2-src_max_x;
        dx2-=((diff*dst_width)/src_width);
     x2=src_max_x;
  }
  if(y1<0){
     diff=(-y1);
        dy1+=((diff*dst_width)/src_width);
     y1=0;
  }
  if(y2>src_max_y){
     diff=y2-src_max_y;
        dy2-=((diff*dst_width)/src_width);
     y2=src_max_y;
  }


  /* if dest area is clipped, shrink src area by the appropriate amount */
  if(dx1<0){
     diff=(-dx1);
        x1+=((diff*src_width)/dst_width);
     dx1=0;
  }
  if(dx2>dst_max_x){
     diff=dx2-dst_max_x;
        x2-=((diff*src_width)/dst_width);
     dx2=dst_max_x;
  }
  if(dy1<0){
     diff=(-dy1);
        y1+=((diff*src_width)/dst_width);
     dy1=0;
  }
  if(dy2>dst_max_y){
     diff=dy2-dst_max_y;
        y2-=((diff*src_width)/dst_width);
     dy2=dst_max_y;
  }

  /* re-check rejection cases */
  if((dx1>dst_max_x) || (dy1>dst_max_y) || (dx2<0)|| (dy2<0)){
     JLIB_LEAVE;
     return;
  }

  if((x1>src_max_x) || (y1>src_max_y) || (x2<0)|| (y2<0)){
     JLIB_LEAVE;
     return;
  }
  
  /* do scale with clipped parameters */
  buff_scale_buff_toNC(dest,dx1,dy1,dx2,dy2,src,x1,y1,x2,y2);
      
  JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| Draw src scaled into dest in the given coordinates without clipping.   |*/
/*+------------------------------------------------------------------------+*/
void buff_scale_full_buff_toNC(buffer_rec *dest,int x1,int y1,int x2,int y2,buffer_rec *src)
{
  JLIB_ENTER("buff_scale_full_buff_toNC");

  buff_scale_buff_toNC(dest,x1,y1,x2,y2,src,0,0,B_MAX_X(src),B_MAX_Y(src));
 
  JLIB_LEAVE;
}


/*+------------------------------------------------------------------------+*/
/*| Draw src scaled into dest in the given coordinates with clipping.      |*/
/*+------------------------------------------------------------------------+*/
void buff_scale_full_buff_to(buffer_rec *dest,int x1,int y1,int x2,int y2,buffer_rec *src)
{
  JLIB_ENTER("buff_scale_full_buff_to");

  buff_scale_buff_to(dest,x1,y1,x2,y2,src,0,0,B_MAX_X(src),B_MAX_Y(src));
 
  JLIB_LEAVE;
}



