// ******************************************************************** //
//                                                                      //
//  UTILITY.CPP                                                         //
//  Copyright (c) 1994, Bob Flanders and Michael Holmes                 //
//                                                                      //
//  This file contains the following miscellaneous routines.            //
//      change_dir()     change drive and directory                     //
//      make_path()      make a target path and change to it            //
//      delete_file()    delete file regardless of status               //
//      extension_given  check for extension                            //
//      quit_with()      give an error message, then return to DOS      //
//      select_file()    determine if file matches against a pattern    //
//      parse_parms()    parse command line against a command table     //
//      clear_memory()   clear any block of memory, anywhere            //
//      malloc_chk()     get/clear a memory block with error check      //
//      huge_malloc()    ..for areas greater than 64k                   //
//      file_date()      convert file date and time to printable string //
//      overwrite_prompt ask user to overwrite existing file            //
//      ratio()          determine ratio between two numbers            //
//      touppers()       uppercase a string                             //
//      control_break()  control break intercept routine                //
//      critical_rtn()   DOS critical error handler                     //
//                                                                      //
// ******************************************************************** //


/* ******************************************************************** *
 *
 *  change_dir() -- change drive and directory
 *
 *  Returns: 0 = sucessfull
 *           1 = bad drive
 *           2 = path not found
 *
 * ******************************************************************** */

int     change_dir(char *s)                 // drive and directory
{
int     org_drive;                          // original drive


if (NOT *s)                                 // q. drive or path specified?
    return(0);                              // a. no .. just return ok

org_drive = _getdrive();                    // get original drive number

if (strlen(s) >= 2 && s[1] == ':')          // q. drive specified?
    {                                       // a. yes .. change drives
    if (_chdrive(toupper(*s) - '@'))        // q. change drive ok?
        return(1);                          // a. no .. return to user

    s += 2;                                 // point to pathname
    }

if (NOT *s)                                 // q. pathname given?
    return(0);                              // a. no .. just return ok

if (chdir(s))                               // q. change directory ok?
    {
    _chdrive(org_drive);                    // a. no .. back to original drive
    return(2);                              // ..and return to caller
    }

return(0);                                  // else .. return all ok

}



/* ******************************************************************** *
 *
 *  make_path() -- make a target path and change to it
 *
 * ******************************************************************** */

char    make_path(char *s)                  // target directory
                                            // returns:
                                            //  -3 = invalid drive
                                            //  -2 = drive not ready
                                            //  -1 = invalid pathname
                                            //   0 = all ok
{
UINT    i;                                  // work drive number
char    drive,                              // drive number
        current_path[MAX_PATH],             // starting path
        current_drive,                      // ..and drive
       *p,                                  // work pointer
       *fp;                                 // free pointer
union   REGS r;                             // cpu registers


p = fp = strdup(s);                         // get an extra copy

_getdcwd(0, current_path, MAX_PATH);        // get current path
current_drive = *current_path - '@';        // ..and drive

if (LAST(s) == '\\')                        // q. last char a backslash?
    LAST(s) = 0;                            // a. yes .. remove it

if (p[1] == ':')                            // q. drive specified?
    {
    drive = toupper(p[0]) - 'A' + 1;        // get name's drive number

    if ((_osmajor == 3 && _osminor >= 1) || // q. DOS 3.1 or higher?
            (_osmajor > 3))
        {
        r.x.ax = 0x4409;                    // a. yes .. do local test
        r.h.bl = drive;                     // bl = drive to test
        int86(0x21, &r, &r);                // test drive validity

        if (r.x.cflag)                      // q. bad drive?
            return(-3);                     // a. yes .. return w/error
        }

    if (_chdrive(drive))                    // q. chg to destination drive?
        {
        chdir(current_path);                // a. no .. change back ..
        _dos_setdrive(current_drive, &i);   // ..to path and drive
        return(-2);                         // ..and return w/error
        }

    p += 2;                                 // ..and move down the path
    }

if (*p == '\\')                             // q. start at root?
    {
    chdir("\\");                            // a. yes .. change to root
    p++;                                    // ..and move pointer
    }

for (; (p = strtok(p, "\\")) != 0;          // using each path segment..
            p = 0)                          // ..check/build a directory
    {
    if (chdir(p))                           // q. change ok?
        {                                   // a. no .. make a subdir
        if (mkdir(p))                       // q. make subdir ok?
            {
            chdir(current_path);            // a. no .. change back ..
            _dos_setdrive(current_drive, &i);  // ..to path and drive
            return(-1);                     // ..and return with error
            }

        chdir(p);                           // else .. change to new one
        }
    }

free(fp);                                   // release gotten memory..
return(0);                                  // ..and return .. all ok

}



/* ******************************************************************** *
 *
 *  delete_file() -- delete file any way necessary
 *
 * ******************************************************************** */

int     delete_file(char *s)                // filename to delete
{
int     rc;                                 // return code, 0=ok


if ((rc = unlink(s)) != 0)                  // q. regular file unlink?
    if (NOT (rc = chmod(s, S_IWRITE)))      // a. no .. q. change ok?
        rc = unlink(s);                     // a. yes .. try again

return(rc);                                 // return w/final status

}



/* ******************************************************************** *
 *
 *  extension_given() -- find the file extension in a filename
 *
 * ******************************************************************** */

char   *extension_given(char *s)            // source string
{
char   *p;                                  // work pointer


if ((p = strrchr(s, '.')) != 0)             // q. file extension given?
    return(p > strrchr(s, '\\')             // a. yes .. rtn w/address
             ? p : FALSE);                  // ..if no intervening \

return(FALSE);                              // else .. return empty handed

}



/* ******************************************************************** *
 *
 *  quit_with() -- give an error message, then return to DOS
 *
 * ******************************************************************** */

void    quit_with(char *msg, ...)           // quit with an error message
{
va_list list;                               // variable list


va_start(list, msg);                        // set up variable list
vprintf(msg, list);                         // give error message ..

change_dir(init_path);                      // back to initial drv and path
exit(rc);                                   // ..and then quit

}



/* ******************************************************************** *
 *
 *  select_file() -- select a file based on pattern
 *
 *  Returns: TRUE = file matches pattern
 *          FALSE = file does not match pattern
 *
 * ******************************************************************** */

int     select_file(SFS *sf0,               // target filename
                    SFS *sf1)               // match pattern
{
int     m = 0;                              // match flag
char   *s0, *s1;                            // work string pointers


if (sf1->dir[0] &&                          // q. directory given
            strncmp(sf0->dir, sf1->dir,     // ..and they don't match
                    strlen(sf1->dir)))      // ..upto pattern's length?
    return(FALSE);                          // a. yes .. return no match


for (s0 = sf0->fname, s1 = sf1->fname;      // loop thru the strings
            *s1; s0++, s1++)                // ..and check for matches
    {
    if (*s1 == '?')                         // q. any character match?
        continue;                           // a. yes .. next character

     else if (*s1 == '*')                   // q. match reset of string?
        {
        m = 1;                              // a. yes .. set flag
        break;                              // ..and exit loop
        }

     else if (*s0 != *s1)                   // q. match on this character?
        return(FALSE);                      // a. no .. rtn no match
    }

if (NOT m && *s0)                           // q. filename match?
    return(FALSE);                          // a. no .. return no match

if (NOT strcmp(s1 = sf1->ext, ".*"))        // q. use any extension?
    m = 1;                                  // a. yes .. set flag
 else
    {
    for (s0 = sf0->ext, m = 0;              // loop thru the strings
                *s1; s0++, s1++)            // ..and check for matches
        {
        if (*s1 == '?')                     // q. any character match?
            continue;                       // a. yes .. next character

         else if (*s1 == '*')               // q. match reset of string?
            {
            m = 1;                          // a. yes .. set flag
            break;                          // ..and exit loop
            }

         else if (*s0 != *s1)               // q. match on this character?
            return(FALSE);                  // a. no .. rtn no match
        }
    }

return(m || NOT *s0);                       // rtn match if ext match

}



/* ******************************************************************** *
 *
 *  parse_parms() -- parse command line against a command table
 *
 *  This routine takes the command line argument count and the command
 *  line arguments and processes them against a command table.  The
 *  command table contains the definitions for each of the programs
 *  options.  Each table entry contains the switch character, the
 *  address of switch variable and optionally a pointer to the switch
 *  data.  Positional parameter's address are stored in the positional
 *  parameter array and this array's address is returned to the caller
 *  in the parms_array variable.  Finally, the total count of positional
 *  parameters is returned from parse_parm.
 *
 * ******************************************************************** */

int     parse_parms(int  ac,                // command line arg count
                    char *av[],             // command line arguments
                    int  n,                 // command table entries
                    struct cmd_parm *t,     // command table
                    char ***parms_array)    // positional parms array
{
int     i, j,                               // loop counter
        parms_fnd = 0;                      // positional parms found
char    switch_fnd = 0,                     // switch found in token
        dash_valid = 1,                     // dash valid as switch char
       *p, *q, *r,                          // character pointers
        c;                                  // work character


*parms_array = (char **) malloc(            // set up for max nbr tokens
            sizeof(char *) * ac);           // ..from max nbr of args

for (i = 1; i < ac; i++)                    // for each cmd line token
    {
    p = av[i];                              // set up pointer to token

    while (*p)                              // process token
        {
        if (*p == '/' || switch_fnd ||      // q. slash option
                (dash_valid && *p == '-'))  // ..or dash option?
            {                               // a. yes .. process it
            if (NOT switch_fnd)             // q. embedded switch?
                p++;                        // a. no .. bump past switch

            switch_fnd = FALSE;             // reset switch
            c = toupper(*p);                // get char and upcase it

            if (c == '?')                   // q. help request?
                quit_with(help);            // a. yes .. give help ..

            for (j = 0; j < n; j++)         // check each table entry
                if (c == t[j].cp_ltr)       // q. find match?
                    break;                  // a. yes .. exit loop

            if (j == n)                     // q. no matches?
                {                           // a. no .. process error
                if ((q = strchr(p, '/'))    // q. any more slash switches?
                         != 0)
                    *q = 0;                 // a. yes .. isolate bad one

                if ((q = strchr(p, '-'))    // q. any more dash switches?
                         != 0)
                    *q = 0;                 // a. yes .. isolate bad one

                quit_with(bad_op, p);       // give error message & quit
                }

            q = p + 1;                      // point to next switch

            if (t[j].cp_flag)               // q. keyword with data?
                {                           // a. yes .. process arg
                if (NOT strlen(q))          // q. value given?
                    quit_with(bad_value,    // a. no .. give an error
                            *p);            // ..message and quit

                p = strcspn(q, "-") + q;    // get to next dash switch
                r = strcspn(q, "/") + q;    // ..and next slash switch

                if (p > r || NOT dash_valid)// q. slash appear next?
                    p = r;                  // a. yes .. use 1st occurence

                if (*p)                     // q. any more switches to use?
                    {
                    *p++ = 0;               // a. yes .. get to slash
                    switch_fnd = TRUE;      // ..show nxt char a switch
                    }

                *(char **)t[j].cp_entry = q;// save argument string
                }

             else
                {
                p++;                        // point to next switch
                (*t[j].cp_entry)++;         // show switch parm used

                if (*p == '-' && NOT dash_valid)    // q. ok to us a dash?
                    quit_with(bad_switch, p);       // a. no .. give error
                }
            }

         else
            {
            dash_valid = 0;                 // dashes not valid now

            (*parms_array)[parms_fnd++] = p;// save positional string

            if (*(p += strcspn(p, "/")))    // q. any switches left?
                {
                *p++ = 0;                   // a. yes .. make a string
                switch_fnd = TRUE;          // ..show nxt char a switch
                }
            }
        }

    }

return(parms_fnd);                          // rtn w/nbr of positionals

}



/* ******************************************************************** *
 *
 *  clear_memory() -- clear an area of memory
 *
 * ******************************************************************** */

void    clear_memory(char huge *p,          // address of memory block
                     long       n)          // length of block
{
UINT    clr_size = 0;                       // working chunk size


for (; n; n -= clr_size)                    // clear in big chunks
    {
    if (n > (65536L - 16))                  // q. more than 64k to do?
        clr_size = (UINT) 65536L - 16;      // a. yes .. just do some
     else
        clr_size = (UINT) n;                // else .. do what's left

    _fmemset(p, 0, (UINT) clr_size);        // clear to block to null
    p += clr_size;                          // point to next block
    }
}



/* ******************************************************************** *
 *
 *  malloc_chk() -- allocate memory with error processing
 *
 * ******************************************************************** */

void   *malloc_chk(UINT n)                  // size of block
{
void   *s;                                  // temporary pointers


if (NOT (s = malloc(n)))                    // q. enough memory?
    quit_with(no_memory);                   // a. no .. give error msg

clear_memory((char huge *) s, (long) n);    // else .. clear to nulls
return(s);                                  // ..and return w/address

}



/* ******************************************************************** *
 *
 *  huge_malloc() -- allocate huge memory with error processing
 *
 * ******************************************************************** */

void    huge *huge_malloc(long n)           // size of block
{
void    huge *s;                            // temporary pointer


if (NOT (s = farmalloc(n)))                 // q. enough memory?
    quit_with(no_memory);                   // a. no .. give error msg

clear_memory((char huge *) s, n);           // else .. clear to nulls
return(s);                                  // ..and return w/address

}



/* ******************************************************************** *
 *
 *  file_date() -- convert file date and time to printable string
 *
 *  Returns: a static string in the following format: "mm-dd-yy hh:mm"
 *
 * ******************************************************************** */

char   *file_date(UINT date, UINT time)     // target file's date and time
{
static
char    string[] = "xx-xx-xx xx:xx",        // return string
        fmt[] = "%02d-%02d-%02d %2d:%02d";  // format string

sprintf(string, fmt,                        // convert file's..
        (date >> 5) & 0xf,                  // ..date  month
        date & 0x1f,                        // ..      day
        ((date >> 9) & 0x7f)+ 80,           // ..      year
        (time >> 11) & 0x1f,                // ..time  hour
        (time >> 5) & 0x3f);                // ..      minute

return(string);                             // ..then return string

}



/* ******************************************************************** *
 *
 *  overwrite_prompt() -- give overwrite prompt; wait for response
 *
 * ******************************************************************** */

int     overwrite_prompt(void)              // 0 = ok to overwrite
{
int     reply;                              // reply to prompt


printf(overwrite_q);                        // give request to overwrite

for (;;)                                    // loop 'til good answer
    {
    while (NOT kbhit())                     // loop till key available
        ;

    reply = getch();                        // get the response

    switch (toupper(reply))                 // handle user's response
        {
        case 'A':                           // A = all
            sw_overwrite = 1;               // set up flag for later
            printf("All\n");                // show user's response
            return(0);                      // overwrite file on return

        case 'S':                           // S = stop
            printf("Stop\n");               // show user's response
            quit_with(stop_here);           // ..give error msg and quit

        case 0x1b:                          // ESC = stop, too
            printf("Esc\n");                // show user's response
            quit_with(stop_here);           // ..give msg and quit

        case 'N':                           // N = no
            printf("No\n");                 // show user's response
            return(1);                      // do not overwrite on rtn

        case 'Y':                           // Y = yes
        case 0x0a:                          // <CR> = yes, too
            printf("Yes\n");                // show user's response
            return(0);                      // overwrite file on return
        }
    }
}



/* ******************************************************************** *
 *
 *  ratio() -- determine ratio between two numbers
 *
 *  The ratio routine returns a value which is roughly equivalant to
 *
 *      big - little
 *     --------------  x  100
 *          big
 *
 * ******************************************************************** */

int     ratio(ULONG big, ULONG little)      // ratio input
{
ULONG   wr;                                 // working ratio


if (big < little || big == 0)               // q. flipped nbrs or divisor zero?
    return(0);                              // a. yes .. rtn w/ratio of zero

wr = big - little;                          // get numerator before scaling

while (wr > (0xffffffffUL / 1000))          // loop 'til nbrs are small enough
    {
    wr /= 10;                               // reduce both the
    big /= 10;                              // ..numerator and denominator
    }

wr = (((wr * 1000L) / big) + 9) / 10;       // determine ratio from 0 to 100
return((int) wr);                           // ..and return ratio to caller

}



/* ******************************************************************** *
 *
 *  touppers() -- uppercase a string
 *
 * ******************************************************************** */

void    touppers(char *s)                   // string to uppercase
{

while (*s)                                  // for the whole string..
    *s++ = toupper(*s);                     // ..uppercase each char

}


#if defined(_BC_VER)                        // BC++ 3.1 handlers

/* ******************************************************************** *
 *
 *  control_break() -- control break intercept routine
 *
 * ******************************************************************** */

int     control_break(void)
{

quit_with(stop_here);                       // give error msg and quit
return(0);                                  // ..and abort processing

}



/* ******************************************************************** *
 *
 *  critical_rtn() -- DOS critical error handler
 *
 * ******************************************************************** */

#pragma option -O2-b-e                      // no global reg allocation
                                            // ..or dead code elimination

void    interrupt critical_routine(...)
{

if (_AX & 0x800)                            // q. fail allowed?
    _AX = (_AX & 0xff00) | 3;               // a. yes .. show failed
 else
    _AX = (_AX & 0xff00) | 2;               // else .. abort

}

#endif
