{
Screen dump player to play back raw CGA screen dumps.  Sets composite color
mode.  Reprograms system timer to fire at the framerate we want, then
restores at exit.

USAGE:

  cgaview3 [filename.ext] [fps]

...where fps = desired frames per second.

Programming notes:

An elegant way to do the buffers is through a queue managed with pointers.  
To keep things simple, we'll just use an array of buffers.

This code does NOT rely on the Borland "CRT" unit to avoid "runtime error
200" problems when running on machines faster than 166MHz.

TODO:
- Display ring buffer legend if user holds down a shift key by setting
  bytes on top edge.  Color bytes according to cur, head, and tail
- Display maximum and average fps achieved on exit.

Drawbacks and future enhancements:
- Does not currently implement any form of audio playback.
- System uses an asynchronous method, which can drift out of sync with
  audio.
- Framerate is limited to an integer.  Future modification could allow for
  fractional rates, like 29.97 (actual NTSC display rate, not 30).
- Program only accepts command-line arguments.  Might be nice to display a
  list of *.CGA files for the user to pick from.
- Assumes a raw file of full 16K CGA screen dumps.  Future enhancement
  might be to only play back 8K of data, which would fill the screen
  skipping every other line but double playback speed.
}

{{$DEFINE DEBUG}
{If this flag is active then we are in debug mode for single-stepping through
the code with interrupts off.  The letter "f" is necessary to advance a frame}

program cgaview3;

uses
  tinterrupts;

const
  intMult:word=30; {needed to handle interrupts slower than 18.2Hz}

var
  fname:string;
  f:file;
  buffers:array[0..63] of pointer;   {ring buffer; an array of pointers to buffers}
  bufferMax,
  bufferHead,
  bufferTail:byte;             {indexes for managing the ring buffer}
  screenSize:word;             {size of a CGA screen}
  screenLoc:pointer;
  fps:byte;
  framesPending:word;          {How many frames in time are pending to play}
  intCounter:byte;
  playbackQuit,
  playbackPause:boolean;       {flags to pause or stop playback}
  result:word;
  lastFrameLoaded:boolean;

{$F+}
procedure FPSCounter; Interrupt;
{
Simple interrupt to increase a counter variable.  Ensures that system
time is kept in sync by checking if it is time to update the BIOS timer
tick counter and passing control to that routine if necessary.

Do not do any LONGINT stuff in an interrupt handler with Turbo Pascal 7
because the 32-bit-away routines do not preserve the upper bits of
ax/bx/cx/dx. If you must, and your code will run on 386s or higher,
make sure you PUSH EAX/EBX/ECX/EDX before starting and POP when done.
}

begin
  inc(intCounter); if intCounter >= intMult then intCounter:=0;

  {every time intCounter=0, we are at FPS timeslice and advance framesPending}
  if intCounter=0 then begin
    if not playbackPause {if the user hasn't paused playback}
    and (framesPending < bufferMax) {to prevent excessive queueing}
      then inc(framesPending);
  end;

  {We want to be nice, and will maintain the BIOS interrupt so time doesn't drift}
  inc(PITCycles,Chan0Counter); {Keep track of how many PIT cycles have gone by}
  if longrec(PITCycles).hi <> 0 then begin {Did we roll over?  Is it time to call the 18.2Hz BIOS handler?}
    longrec(PITCycles).hi:=0; {Update our PIT cycles counter}
    asm pushf end; {simulate an interrupt by pushing flags, then CALLing handler}
    BIOSTimerHandler; {this will acknowledge the interrupt}
  end
  else
    Port[$20] := $20; {send EndOfInterrupt to the PIC to ackn. the interrupt}
end;
{$F-}

Function KeyPressed:Boolean; Assembler;
Asm
  mov ah, 01h
  int 16h
  mov ax, 00h
  jz @1
  inc ax
  @1:
end;

Function ReadKeyChar:Char; Assembler;
Asm
  mov ah, 00h
  int 16h
  cmp al,0
  jne @endit
  mov al,ah
@endit:
end;

const
  {useful for interpreting keyboard status byte}
  rshift=01;
  lshift=02;
  ctrl  =04;
  alt   =08;

function KeyboardStatusFlags:byte; assembler;
asm
  mov ah,2
  int 16h
  {result in AL}
end;

Function StrToInt(s:string):longint;
var
  i:longint;
  foo:integer;
Begin
  val(s,i,foo);
  StrToInt:=i;
End;

procedure displayQueue;
{draws a small graphical legend of the buffer queue, head, and tail}
var
  w:word;
begin
  {draw buffer queue in white}
  for w:=0 to bufferMax do mem[$b800:w]:=$ff;
  {draw buffer head and tail in different solid colors}
  mem[$b800:bufferHead]:=$55;
  mem[$b800:bufferTail]:=$AA;
end;

procedure InitCompositeVideo;
const
  m6845_color_sel=$3d9; {color select register}
    {relevant bits}
    c_black=0; {just in case it wasn't obvious}
    c_blue=1;
    c_green=2;
    c_red=4;
    c_bright=8;
    c_alternate_intensity=16; {alt. intens. colors in graphics mode.  specs say "backgr. color" in text mode?  huh?}
    c_paletteCMW=32; {otherwise, red/green/yellow palette}
  m6845_mode_ctl=$3d8; {mode control register}
    {relevant bits}
    c_fast_char_clock=1; {use 160 bytes per line instead of 80; REQUIRED for 80x25 mode, otherwise 40x25 mode}
    c_graphics_enable=2; {otherwise, text mode}
    c_blackandwhite_enable=4; {otherwise, color signal}
    c_videosignal_enable=8; {otherwise, NO VIDEO SIGNAL}
    c_640x200_enable=16; {otherwise, 320x200}
    c_blinking_text=32; {otherwise, all background colors enabled}

begin
  asm
    mov ax,0006h
    int 10h

    mov dx,m6845_mode_ctl
    mov al,c_graphics_enable+c_videosignal_enable+c_640x200_enable
    out dx,al

    mov dx,m6845_color_sel
    mov al,15
    out dx,al
  end;
end;

procedure InitPlayer;
begin
  if paramcount=0 then begin
    writeln('Usage: cgaview3 [filename.ext] [fps]');
    halt(1);
  end;

  fname:=paramstr(1);
  assign(f,fname);
  reset(f,1);
  fps:=5; {if user doesn't supply an fps, assume 5 fps}
  if paramstr(2) <> '' then fps:=StrToInt(paramstr(2));
  if fps>60 then fps:=60; {no point in exceeding screen refresh rate!}
  if fps=0 then fps:=1;
  lastFrameLoaded:=false;

  screenSize:=16*1024;
  screenLoc:=ptr($b800,0000);

  {Initialize buffers; grab ram until there isn't enough left for a frame}
  bufferMax:=0; bufferHead:=0; bufferTail:=0;
  while maxavail > screenSize do begin
    getmem(buffers[bufferMax],screenSize);
    inc(bufferMax);
  end;
  dec(bufferMax); {adjust bufferMax to point to the last successful buffer}

  {Pre-fill the buffers before we start playing}
  for bufferHead:=0 to bufferMax do begin
    write(#13,'Preload buffers remaining: ',bufferMax-bufferHead,'  ');
    blockread(f,buffers[bufferHead]^,screenSize,result);
    if result<screenSize {Did we run out of frames to load?}
      then begin
        lastFrameLoaded:=true;
        dec(bufferHead); {roll back our Head}
        break; {get out of here}
      end;
   end;

  InitCompositeVideo;

  framesPending:=1; {Show the first preloaded frame}
  playbackQuit:=false; playbackPause:=false;
end;

procedure DoPlayer;

  {
  Procedures to manage our ring buffer queue.
  Makes the playback loop cleaner to read and understand.
  }
  procedure GrowHead;
  begin
    inc(bufferHead);
    if bufferHead>bufferMax then bufferHead:=0; {wrap around}
  end;

  procedure ShrinkHead;
  begin
    dec(bufferHead); {roll back our Head}
    if bufferHead>bufferMax then bufferHead:=bufferMax; {wrap around}
  end;

  procedure GrowTail;
  begin
    inc(bufferTail);
    if bufferTail>bufferMax then bufferTail:=0; {wrap around}
  end;

  function QueueEmpty:boolean;
  begin
    QueueEmpty:=(bufferTail=bufferHead);
  end;

  function QueueFull:boolean;
  var
    b,found:byte;
  begin
    found:=0; b:=bufferTail;
    repeat
      inc(found);
      inc(b); if b>bufferMax then b:=0;
    until b=bufferHead;
    QueueFull:=(found=bufferMax); {true if head-tail = total size of queue}
  end;

begin
  {
  Main playback routine.  Each iteration of the inner loop works like this:
    - If one or more frames are pending, display the next pending frame
    - If no frames are pending, or we must show a frame but the buffer is
      empty, load a frame from disk

  How many frames are pending is based on the framesPending variable which
  is incremented by the interrupt routine, and decremented by the frame
  display code.  Frames are loaded at the bufferHead, which grows until it
  bumps into the bufferTail.  Frames are displayed from the bufferTail,
  which advances until it bumps into the bufferHead.
  }

  {$IFNDEF DEBUG}
  {Initialize timer}
  intCounter:=0;
  {The IBM PC can't handle interrupt rates slower than 18.2 times a second
  so we have to cheat by using a multiplier.  If the FPS is > 18.2 Hz, we
  don't need a multiplier.}
  if fps<19
    then intMult:=300 div fps
    else intMult:=1;
  SetTimer(@FPSCounter, fps * intMult); {set our interrupt handler to fire at fps rate}
  {$ENDIF}

  while not playbackQuit do begin
    if (framesPending>0) {Do we have one or more frames to play?}
    and not QueueEmpty
    and not playbackPause
    then begin
      move(buffers[bufferTail]^,screenLoc^,screenSize);
      if (framesPending <> 0) then dec(framesPending);
      GrowTail;
    end else begin
      if not lastFrameLoaded and not QueueFull then begin
        GrowHead;
        blockread(f,buffers[bufferHead]^,screenSize,result);
        if result<screenSize {Did we run out of frames to load?}
          then begin
            ShrinkHead;
            lastFrameLoaded:=true;
          end;
      end;
    end;

    {if the buffer is empty and the last frame has been loaded, we're done}
    if QueueEmpty and lastFrameLoaded then playbackQuit:=true;

    {Handle user actions}

    {$IFNDEF DEBUG} if keypressed then {$ENDIF}
      case upcase(readkeychar) of
        #27 {ESCAPE}:playbackQuit:=true;
        #32 {SPACE} :playbackPause:=not playbackPause;
        'F':inc(FramesPending); {debug mode; force a frame to advance}
      end;

    {display, graphically, the state of the buffer queue}
    {$IFNDEF DEBUG} if (KeyboardStatusFlags AND rshift)=1 then {$ENDIF} displayQueue;

    {If user paused playback, keep resetting the amount of pending frames}
    if playbackPause then framesPending:=0;
    {$IFDEF DEBUG} write(#13,' ',framesPending,' '); {$ENDIF}
  end;
end;

procedure DonePlayer;
begin
  {$IFNDEF DEBUG}
  {restore interrupts}
  CleanUpTimer;
  {$ENDIF}

  {close file and free memory}
  close(f);
  for bufferTail:=0 to bufferMax do freemem(buffers[bufferTail],screenSize);

  {back to 80x25 text mode}
  asm
    mov ax,0003h
    int 10h
  end;
  {print out useful stats?}
end;

begin
  InitPlayer;
  DoPlayer;
  DonePlayer;
end.
