;*********************************************************************;
;*                              A T C L K                            *;
;*-------------------------------------------------------------------*;
;*    Task           : This program is a clock-driver which can be   *;
;*                     used by DOS for functions which access date   *;
;*                     and time on the battery powered clock         *;
;*                     of the AT.                                    *;
;*-------------------------------------------------------------------*;
;*    Author         : MICHAEL TISCHER                               *;
;*    developed on   : 8.4.87                                        *;
;*    last Update    : 9.21.87                                       *;
;*-------------------------------------------------------------------*;
;*    assembly      : MASM ATCLK;                                    *;
;*                    LINK ATCLK;                                    *;
;*                    EXE2BIN ATCLK ATCLK.SYS                        *;
;*-------------------------------------------------------------------*;
;*    Call         : Copy into root directory place the command      *;
;*                     DEVICE=ATCLK.SYS in the CONFIG.SYS file       *;
;*                     and then boot system.                         *;
;*********************************************************************;

code     segment

         assume cs:code,ds:code,es:code,ss:code

         org 0                   ;Program has no PSP, therefore
                                 ;beginning at offset address 0

;== Constants ========================================================

cmd_fld  equ 2                   ;Offset command-field in data block
status   equ 3                   ;Offset status field in data block
end_adr  equ 14                  ;Offset driver end-adr. in data block
num_db   equ 18                  ;Offset number in data block
b_adr    equ 14                  ;Offset buffer-address in data block

;== Data ==============================================================

;-- Header of Device-Driver -------------------------------------------

         dw -1,-1                ;Connection to next driver
         dw 1000000000001000b    ;Driver attribute
         dw offset strat         ;Pointer to strategy routine
         dw offset intr          ;Pointer to interrupt routine
         db "$CLOCK "            ;new clock driver

db_ptr   dw (?),(?)              ;address of data block passed

mon_tab  db 31                   ;Table with number of days in 
february db 28                  ;the months 
         db 31,30,31,30,31,31,30,31,30,31

;== Routines and functions of the Driver ==============================

strat    proc far                ;Strategy routine

         mov  cs:db_ptr,bx       ;Record address of the data block in
         mov  cs:db_ptr+2,es     ;the variable DB_PTR 

         ret                     ;back to caller 

strat    endp

;----------------------------------------------------------------------

intr     proc far                ;interrupt routine

         push ax                 ;Store registers on the stack 
         push bx
         push cx
         push dx
         push di
         push si
         push bp
         push ds
         push es
         pushf                   ;Store the flag register 

         cld                     ;increment for string commands 

         push cs                 ;Set data segment register 
         pop  ds                 ;Code is identical with data here

         les  di,dword ptr db_ptr;Address of data block to ES:DI
         mov  bl,es:[di+cmd_fld] ;Get command-code 
         cmp  bl,4               ;Should Time/Date be read?
         je   ck_read            ;YES --> CK_READ
         cmp  bl,8               ;Should Time/Date be written?
         je   ck_write           ;YES --> CK_WRITE
         or   bl,bl              ;should the driver be initialized ?
         jne  unk_fkt            ;NO --> unknown function

         jmp  init               ;initialize driver 

unk_fkt: mov  ax,8003h           ;Code for "unknown Command"

         ;-- Function Execution completed ---------------------------

intr_end label near
         or   ax,0100h           ;Set finished-bit 
         mov  es:[di+status],ax  ;store everything in status field 

         popf                    ;Restore flag register 
         pop  es                 ;Restore other registers 
         pop  ds
         pop  bp
         pop  si
         pop  di
         pop  dx
         pop  cx
         pop  bx
         pop  ax

         ret                     ;back to caller 

intr     endp

;-----------------------------------------------------------------------

ck_read  proc  near		 ;Read Time/Date from the clock

         mov  byte ptr es:[di+num_db],6 ;6 bytes are passed 
         les  di,es:[di+b_adr]   ;ES:DI points to the DOS-buffer

         mov  ah,4               ;Read function number for Date 
         int  1Ah                ;Call BIOS Time interrupt 
         call date_ofs           ;Change Date after offset to 1.1.1980 
         stosw                   ;store in buffer 

         mov  ah,2               ;Read function number for time 
         int  1Ah                ;Call BIOS Time interrupt 
         mov  bl,ch              ;Store hour in BL 
         call bcd_bin            ;convert minutes 
         stosb                   ;Store in buffer 
         mov  cl,bl              ;Hour to CL
         call bcd_bin            ;Convert hour 
         stosb                   ;Store in buffer 
         xor  al,al              ;Hundreth second is 0
         stosb                   ;Store in buffer 
         mov  cl,dh              ;Seconds to CL
         call bcd_bin            ;Convert seconds 
         stosb                   ;Store in buffer 

         xor  ax,ax              ;everything o.k.
         jmp  short intr_end     ;back to caller 

ck_read  endp

;-----------------------------------------------------------------------

ck_write proc near	      ;Write Time/Date into clock

         mov  byte ptr es:[di+num_db],6 ;6 bytes are read 
         les  di,es:[di+b_adr]   ;ES:DI points to the DOS buffer

         mov  ax,es:[di]         ;Get number of days since 1.1.1980 
         push ax                 ;store number 
         call ofs_date           ;convert into a date 
         mov  ch,19h             ;Year begins with 19..
         mov  ah,5               ;Set function number for dte 
         int  1AH                ;Call BIOS Time interrupt 

         mov  al,es:[di+2]       ;Get minute from buffer 
         call bin_bcd            ;convert to BCD 
         mov  cl,al              ;bring to CL 
         mov  al,es:[di+5]       ;Get seconds from buffer 
         call bin_bcd            ;convert to BCD 
         mov  dh,al              ;bring to DH 
         mov  al,es:[di+3]       ;Get hours from buffer 
         call bin_bcd            ;convert to BCD 
         mov  ch,al              ;bring to CH 
         xor  dl,dl              ;no summer time 
         mov  ah,3               ;Set function number for time 
         int  1AH                ;Call BIOS Time interrupt 

         ;-- Calculate Day of the Week  -------------------------------
         xor  dx,dx              ;HI-word for division
         pop  ax                 ;Get number of days from stack 
         or   ax,ax              ;is number 0?
         je   nodiv              ;Yes --> bypass division 
         xor  dx,dx              ;HI-word for division
         mov  cx,7               ;week has seven days 
         div  cx                 ;divide AX by 7 
nodiv:   add  dl,3               ;1.1.80 was a Tuesday (Day 3)
         cmp  dl,8               ;is it a Sunday or Monday?
         jb   nosomo             ;NO --> no correction necessary 
         sub  dl,cl              ;correct value 
nosomo:  mov  al,6               ;Location 6 in RTC is day of week 
         out  70h,al             ;Address to RTC-address register
         mov  al,dl              ;Day of the week to AL
         out  71h,al             ;Day of the week to RTC-data register

         xor  ax,ax              ;everything o.k.
         jmp  intr_end           ;back to caller 

ck_write endp

;-- OFS_DATE: Convert number of days since 1.1.1980 into date  -------
;-- Input  : AX = Number of days since 1.1.1980
;-- Output  : CL = Year, DH = Month and DL = Day
;-- Registers : AX, BX, CX, DX, SI and FLAGS are changed 
;-- Info     : For conversion of Offsets the Array MON_TAB
;--            is used 

ofs_date proc near

         mov  cl,80              ;Year 1980 
         mov  dh,01              ;January 
ly:      mov  bx,365             ;Number of days in a normal year 
         test cl,3               ;is year a leap year?
         jne  ly1                ;NO --> ly1
         inc  bl                 ;Leap Year has one day more 
ly1:     cmp  ax,bx              ;another year passed?
         jb   mo                 ;NO --> Calculate months 
         inc  cl                 ;YES --> Increment year 
         sub  ax,bx              ;deduct number of days in this year
         jmp  short ly           ;calculate next year 

mo:      mov  bl,28              ;Days in February in a normal year 
         test cl,11b             ;is the year a leap year?
         jne  nolp2              ;NO --> nolp2
         inc  bl                 ;in leap year February has 29 days 
nolp2: mov  february,bl          ;store number of days in February 

         mov  si,offset mon_tab  ;Address of months table 
         xor  bh,bh              ;every month has less than 256 days 
mo1:     mov  bl,[si]            ;Get number of days in month 
         cmp  ax,bx              ;another month passed?
         jb   day                ;NO --> calculate day 
         sub  ax,bx              ;YES --> deduct day of the month 
         inc  dh                 ;increment month 
         inc  si                 ;SI to next month in the table 
         jmp  short mo1          ;calculate next month 

day:     inc  al                 ;the remainder  + 1 is the day 
         call bin_bcd            ;Convert day to BCD 
         mov  dl,al              ;transmit to DL 
         mov  al,dh              ;transmit month to AL 
         call bin_bcd            ;convert to BCD 
         mov  dh,al              ;move to DH 
         mov  al,cl              ;move year to AL 
         call bin_bcd            ;convert to BCD 
         mov  cl,al              ;move to CL 

         ret                     ;back to caller 

ofs_date endp

;-- BIN_BCD: Convert Binary-Number to BCD ----------------------------
;-- Input  : AL = Binary value 
;-- Output  : AL = corresponding BCD-value 
;-- Register : AX, CX and FLAGS are changed 

bin_bcd  proc near

         xor  ah,ah              ;prepare 16 bit division 
         mov  ch,10              ;work in decimal system 
         div  ch                 ;divide AX by 10 
         shl  al,1               ;Shift quotient left 4 places 
         shl  al,1               
         shl  al,1
         shl  al,1
         or   al,ah              ;OR remainder 
         ret                     ;back to caller 

bin_bcd  endp

;-- DATE_OFS: Convert Date in number of days since 1.1.1980 -----
;-- Input  : CL = Year, DH = Month and DL = Day 
;-- Output  : AX = Number of days since 1.1.1980
;-- Register : AX, BX, CX, DX, SI and FLAGS are changed 
;-- Info     : For conversion of date, the Array MON_TAB
;--            is used 

date_ofs proc near

         call bcd_bin            ;Convert year to binary 
         mov  bl,al              ;transmit to BL 
         mov  cl,dh              ;transmit month to CL 
         call bcd_bin            ;Convert Month to binary 
         mov  dh,al              ;and transmit again to DH 
         mov  cl,dl              ;transmit day to CL 
         call bcd_bin            ;convert day to binary 
         mov  dl,al              ;and again transmit to DL 

         xor  ax,ax              ;0 days 
         mov  ch,bl              ;store year 
         dec  bl                 ;back one year 
year:    cmp  bl,80              ;counted back to year 1980 ?
	 jb   month		 ;YES --> convert month
         test bl,11b             ;is year a Leap year ?
         jne  nolpyr             ;NO --> NOLPYR
         inc  ax                 ;a leap year has one more day 
nolpyr:  add  ax,365             ;add days of year 
         dec  bl                 ;back one year 
         jmp  short year         ;process next year 

month:   mov  bl,28              ;Days in February in a normal year
         test ch,11b             ;is current year a Leap Year?
         jne  nolpyr1            ;NO --> NOLPYR1
         inc  bl                 ;in Leap Year February has 29 days 
nolpyr1: mov  february,bl        ;store in Month table 
         xor  ch,ch              ;every month has less than 256 days 
         mov  bx,offset mon_tab  ;Address of month table 
monat1:  dec  dh                 ;decrement number of months 
         je   add_day            ;all month calculated --> TAG
         mov  cl,[bx]            ;Get number of days in month 
         add  ax,cx              ;add to total-days 
         inc  bx                 ;BX to next month in the table 
         jmp  short monat1       ;calculate next month 

add_day: add  ax,dx              ;add current day 
         dec  ax                 ;deduct one day (1.1.80 = 0)
         ret                     ;back to caller 

date_ofs  endp

;-- BCD_BIN: Convert BCD to Binary Number ----------------------------
;-- Input  : CL = BCD-Value 
;-- Output  : AL = corresponding binary value 
;-- Register : AX, CX and FLAGS are changed 

bcd_bin  proc near               ;Convert BCD-value in CL to binary 
                                 ;return in AL 

         mov  al,cl              ;transmit value to AL 
         shr  al,1               ;shift 4 places right 
         shr  al,1
         shr  al,1
         shr  al,1
         xor  ah,ah              ;Set AH to 0 
         mov  ch,10              ;process in decimal system 
         mul  ch                 ;multiply AX by 10 
         mov  ch,cl              ;transmit CL to CH 
         and  ch,1111b           ;Set Hi-Nibble in CH to 0 
         add  al,ch              ;add AL and CH 
         ret                     ;back to caller 

bcd_bin  endp

;----------------------------------------------------------------------

init     proc near               ;Initialization routine

         ;-- the following code can be overwritten by DOS  ----------
         ;--  after installation of the clock

         mov  word ptr es:[di+end_adr],offset init  ;Set end address 
         mov  es:[di+end_adr+2],cs                  ;of the driver 

         mov  ah,9               ;Output installation message 
         mov  dx,offset initm    ;Address of the text 
         int  21h                ;Call DOS interrupt 

         xor  ax,ax              ;everything o.k.
         jmp  intr_end           ;back to caller 

initm    db 13,10,"**** ATCLK-Driver installed. (c) 1987 by"
         db " MICHAEL TISCHER",13,10,"$"

init     endp

;----------------------------------------------------------------------

code     ends
         end
