// File        : dmarecf.c
//
// Description : Function to record any Covox sound format to a file using dmaRecord().
//
// Written by  : Ryan Hanlon
// Description :
// Date        : Feb 19, 1992
//
//
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------

#include "cvxdigi.h"
#include "cvxutil.h"

#define   _DMA_REC_FILE_BUFFER_SIZE        0x8000
#define   _DMA_REC_FILE_BUFFER_COUNT       5
#define   _DMA_REC_FILE_BUFFER_SIZE_MAX    0xF000
#define   _DMA_REC_FILE_BUFFER_COUNT_MAX   5

// Flag used to determine when i/o is complete.
//
extern _ioStopFlag;

//
WORD dmaRecordFile( HANDLE fileHandle,   BYTE fileFormat, BYTE recordRate,
                    WORD   portIn,       WORD channel,  WORD irqNumber,
                    BOOL   silenceRange, BOOL hookKBFlag,
                    LONG   bufferSize,   WORD bufferCount )

{
   _CONVERT_DATA cnvrtData;

   LPSTR  dmaBuffer  [ _DMA_REC_FILE_BUFFER_COUNT_MAX];

   WORD  returnValue = _ERROR_NONE;

   WORD  portInAddress;

   WORD  currentBuffer;   // This variable is used to switch between the buffers
                          // being recorded via dmaRecord.

   LONG  bytesAvailable;

   LONG  bytesWritten;

   int   i;

   // If no buffer Size is given then use defaults.
   //
   if( !bufferSize || (bufferSize > ( LONG )_DMA_REC_FILE_BUFFER_SIZE_MAX) )
      bufferSize = ( LONG )_DMA_REC_FILE_BUFFER_SIZE;


   // 
   if( bufferCount )
      bufferCount -= 1;

   // If no buffer count is given then use defaults.
   //
   if( !bufferCount || (bufferCount > _DMA_REC_FILE_BUFFER_COUNT_MAX) )
      bufferCount = _DMA_REC_FILE_BUFFER_COUNT;


   // Allocate memory for buffers used during DMA recording.
   //
   for(i=0; ( (i < bufferCount) && !returnValue) ; i++)
   {
      dmaBuffer[i]= cvxBufferAlloc( bufferSize,
                                    &bytesAvailable );

      if( dmaBuffer[i] == _NULL )
      {
         returnValue = _ERROR_MEMORY_ALLOC;
      }
   }

   if( !returnValue )
   {
      // Allocate memory for buffer used to pack sound data.
      //
      cnvrtData.destinationPointer = cvxBufferAlloc( bufferSize,
                                                     &bytesAvailable );

      if( cnvrtData.destinationPointer == _NULL )
      {
         returnValue = _ERROR_MEMORY_ALLOC;
      }
   
      if( !returnValue )
      {
         // Set file format variables.
         //
         if( !fileFormat )
            fileFormat = _FORMAT_V8;
            
         // Set record rate.
         //
         if( !recordRate ) 
            recordRate = _CVX_RATE_DEFAULT;
 
         // Initialize DMA for record.
         //
         returnValue =  dmaInit( portIn, channel, irqNumber, &portInAddress );

         if( !returnValue )
         {
            // put header information into structure to be put in the file buffer by pack
            //
            cnvrtData.sourceLength  = ( LONG )_HEADER_LENGTH;   // length of header 
            cnvrtData.bufferFormat  = ( WORD )fileFormat;
            cnvrtData.sampleRate    = recordRate; 
            cnvrtData.silenceRange  = silenceRange;   
            cnvrtData.trebleFlag    = 0;         // unpack feature not used here 
            cnvrtData.noiseFlag     = 0;         // unpack feature not used here 
   
            // Set length to 16 so header only written on first pack 
            cnvrtData.destinationLength = ( LONG )_HEADER_LENGTH; 
   
            if(packFirst((_CONVERT_DATA far *) &cnvrtData))
            {
               returnValue = _ERROR_DURING_PACK_FIRST;
            }
            else
            {
               returnValue = cvxFileWrite( fileHandle,
                                           cnvrtData.destinationPointer,
                                           (LONG)_HEADER_LENGTH,
                                           &bytesWritten );

               // Set length of source buffer
               //
               cnvrtData.sourceLength = bufferSize;
   
               // Reset length of destination buffer. 
               cnvrtData.destinationLength = bufferSize;
   
               // Queue all buffers to begin DMA recording.
               //
               for(i=0; ( (i < bufferCount) && !returnValue ) ; i++)
               {
                  returnValue = dmaRecord( dmaBuffer[i], bufferSize, recordRate );
               }
   
               // Hook int 9h (keyboard interrupt).  This hook will set the 
               // code segment variable '_ioStopFlag' when a key has been
               // pressed.
               //
               if( hookKBFlag )
                  hookKeyboard();
   
               // Set currentBuffer to process first buffer with pack and cvxFileWrite.
               //
               currentBuffer = 0;
   
               // Do the following loop until all data has been recorded or 
               // until a key has been struck.
               //
               while( !returnValue && !_ioStopFlag )
               {
                  // wait for source DMA buffer to open up 
                  do
                  {
                     _asm nop
   
                  }while( ( dmaNumberInQueue() > ( bufferCount-1 ) ) && 
                          !_ioStopFlag  && !returnValue );
   
                  // Check flag that indicates record exit.
                  //
                  if( !_ioStopFlag && !returnValue )
                  {
                     // Pack data according to input format.
                     //
                     cnvrtData.sourcePointer = dmaBuffer[currentBuffer];
   
                     //

                     if( fileFormat == _FORMAT_V8 )
                     {
                        cnvrtData.destinationPointer = cnvrtData.sourcePointer;
                        cnvrtData.destinationFilled  = cnvrtData.sourceLength;
                     }
                     else
                     {
                        if(packNext((_CONVERT_DATA far *) &cnvrtData))
                        {
                           returnValue = _ERROR_DURING_PACK_NEXT;
                        }
                     }
                  
                     // Write sound data.
                     //
                     if( !returnValue )
                     {
                        returnValue =  cvxFileWrite( fileHandle,
                                                     cnvrtData.destinationPointer,
                                                     cnvrtData.destinationFilled,
                                                     &bytesWritten );

                        if( !returnValue )
                        {
                           // Queue next buffer to be recorded into.
                           //
                           returnValue =  dmaRecord( dmaBuffer[currentBuffer], bufferSize, recordRate );
                        }
                     }
   
                     // Increment currentBuffer so next buffer is accessed.
                     //
                     if(currentBuffer == bufferCount - 1)
                     {
                        currentBuffer=0;
                     }
                     else
                     {
                        currentBuffer++;
                     }
                  }
   
               } // end while
   
               // DMA has been halted because of keypress. Find out 
               // how much of the buffer was actually recorded and
               // set the sourceLength variable.
               //
               cnvrtData.sourceLength = (LONG)( bufferSize - ( LONG )dmaBytesRemaining() );
   
               //
               cnvrtData.sourcePointer = dmaBuffer[currentBuffer];
               
               //
               if( fileFormat == _FORMAT_V8 )
               {
                  cnvrtData.destinationPointer = cnvrtData.sourcePointer;
                  cnvrtData.destinationFilled  = cnvrtData.sourceLength;
               }
               else
               {
                  if(packNext((_CONVERT_DATA far *) &cnvrtData))
                  {
                     returnValue = _ERROR_DURING_PACK_NEXT;
                  }
               }
                  
               // Write sound data.
               //
               if( !returnValue )
               {
                  returnValue =  cvxFileWrite( fileHandle,
                                               cnvrtData.destinationPointer,
                                               cnvrtData.destinationFilled,
                                               &bytesWritten );
               }


               dmaFlushQueue();
   
               dmaUninit();
   
               // Restore original keyboard interrupt.
               //
               if( hookKBFlag )
                  unhookKeyboard();
   
            } // end if pack ok 
   
         } // end if DMA Init ok 

         // Free memory allocated for the conversion buffer.
         //
         if( cvxBufferFree( cnvrtData.destinationPointer ) )
         {
            returnValue = _ERROR_MEMORY_DEALLOC;
         }
      }
      
      // Free allocated buffer memory used for the record buffers.
      //
      for( i=3; ((i>=0) && !returnValue); i-- )
      {
         if( cvxBufferFree( dmaBuffer[i]) )
         {
            returnValue = _ERROR_MEMORY_DEALLOC;
         }
      }
   }

   return(returnValue);
}



