// DMASETRT.C
// This program reads 8 bit PCM sound data from a file.
// The 8 bit sound data is then played back using DMA.
// The rate of buffer being played is increased every time
// a key is struck on the keyboard.
//
// The buffer that is used in this program is allocated 
// using cvxBufferAlloc. The buffer is set to 64K. 

#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include "cvxdigi.h"
#include "cvxutil.h"


#define _BUFFER_SIZE  0xFFFF     // Size of each DMA buffer.
#define _PCM_FILE     "TEST1.V8"  // File containing sound data.
#define _RATE_STEP    5          // Value used to increase rate.

extern _dmaInProgressFlag;          
extern _dmaDevice;

VOID main( VOID )
{
   HANDLE  pcmFileHandle;
   LPSTR   pcmBuffer;
   LONG    bytesAvailable;
   LONG    bytesRead;
   WORD    portAddress;
   BYTE    dmaRate;
   WORD    repeatCount = 1;
   WORD    initError;

   // Allocate memory for buffer used to read 8 bit PCM
   // data from file.
   pcmBuffer = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );

   if( pcmBuffer == _NULL )
   {
      printf( "ERROR : Cannot allocate memory!\n" );
      exit( 0 );
   }

   // Initialize DMA. Setting each parameter to _AUTODETECT 
   // causes dmaInit to perform a search for the Port,
   // DMA channel, and IRQ setting respectively.
   initError = dmaInit( _AUTODETECT, _AUTODETECT, _AUTODETECT, 
                        &portAddress );

   // If the variable _dmaDevice equals 0 then the DMA
   // sub-system was not initialized correctly.
   if( _dmaDevice == 0 )
      printf( "ERROR = %d : dmaInit failed\n", initError );
   else
   {
      // Open a file containing 8 bit PCM data.
      if( cvxFileOpen( _PCM_FILE, _OPEN_R_ONLY, &pcmFileHandle ) )
         printf( "ERROR : %s not found.\n", _PCM_FILE );
      else
      {
         // Fill buffer with sound data.
         cvxFileRead( pcmFileHandle, pcmBuffer,
                      ( LONG )_BUFFER_SIZE, &bytesRead );

         // Get rate from header.
         dmaRate = pcmBuffer[ _HEADER_RATE_OFFSET ];
      
         // Bypass header and insert buffer into DMA queue.
         dmaPlay( ( LPSTR )pcmBuffer + _HEADER_LENGTH,
                  ( LONG )bytesRead - _HEADER_LENGTH,
                  dmaRate, repeatCount );

         printf( "Press key increase rate by %d.\n", _RATE_STEP );

         // Loop until DMA has completed.
         while( _dmaInProgressFlag )
         {
            // If a key is struck,increase rate of DMA output.
            if( kbhit() )
            {
               // Make sure that rate does not go above maximum.
               if( ( dmaRate + _RATE_STEP) < _CVX_RATE_IN_MAXIMUM )
               {
                  // Increase rate
                  dmaRate = dmaRate + _RATE_STEP;

                  // Set new DMA rate.
                  dmaSetRate( dmaRate );
               }
               else
                  // Display that the rate can be set no higher.
                  printf( "Rate at maximum.\n" );
               
               // Clear all key stokes.
               while( kbhit() )
                  getch();
            }
         }
      }
   }

   // Uninitialize the DMA system.
   if( dmaUninit() )
      printf( "DMA uninit failure.\n" );
   
   // Close the sound file.
   cvxFileClose( pcmFileHandle );

   // Free memory used by 8 bit PCM buffer.
   if( cvxBufferFree( pcmBuffer ) )
      printf( "ERROR : Cannot free memory!\n" );
}


