/*
  nntplib.c -- Waterloo TCP NNTP client libraries.

  Copyright (C) 1997-1999  Rod Whitby
  Copyright (C) 1992       Walter Andrew Nolan

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version
  2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be 
  useful, but WITHOUT ANY WARRANTY; without even the implied
  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more
  details.

  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
  USA.

  $Source: A:/SRC/TCP/MTA/RCS/NNTPLIB.C $
  $Id: NNTPLIB.C 1.23 2000/04/09 07:02:50 rwhitby Exp $
*/

#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <tcp.h>
#include "socklib.h"

/* Global Variables */

static int NNTP_stat;

/*-------------------------------------------------------------
 -                                                            <
 -  NNTP Stuff                                                <
 -                                                            <
/------------------------------------------------------------*/

#define SOCK_RESPONSE(PROTOCOL) \
  do { \
    SOCK_GETS(PROTOCOL); \
  } while (Sock_buffer[3] == '-')

#define SOCK_COMMAND(PROTOCOL) \
  strcpy(Sock_last, Sock_buffer); \
  SOCK_PUTS(PROTOCOL); \
  SOCK_GETS(PROTOCOL); \
  if (!strcmp(Sock_buffer, Sock_last)) { \
    Sock_echo = 1; \
    SOCK_GETS(PROTOCOL); \
  } \
  while (Sock_buffer[3] == '-') { \
    SOCK_GETS(PROTOCOL); \
  }

tcp_Socket * nntp_init(char *proxy, char *host, int port)
{
  tcp_Socket *NNTP_sock=NULL;
  char *p;

  NNTP_sock=(tcp_Socket *)malloc(sizeof(tcp_Socket));

  if (!socklib_connect( "NNTP", NNTP_sock, &NNTP_stat, proxy, host, port)) {
    fprintf(stderr, "NNTP> Unable to connect to %s\n", host);
    return 0;
  }

  if (proxy) {
    if (((p = strstr(Sock_buffer, "200 ")) != NULL) ||
        ((p = strstr(Sock_buffer, "201 ")) != NULL)) {
      strcpy(Sock_buffer, p);
    }
    else {
      SOCK_RESPONSE( NNTP );
    }
  }
  else {
    SOCK_RESPONSE( NNTP );
  }

  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_QUIT_ON(NNTP, NNTP_BYE, );

  SOCK_READ_ERR(NNTP, );
  return NNTP_sock;
}

int nntp_auth(tcp_Socket *NNTP_sock, char *username, char *password)
{
  sprintf(Sock_buffer, "AUTHINFO USER %s", username);
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_PASS, );

  sprintf(Sock_buffer, "AUTHINFO PASS %s", password);
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_AUTHOK, );

  SOCK_READ_ERR(NNTP, );
  return 1;
}

int nntp_post(tcp_Socket *NNTP_sock, FILE *f, char eol)
{
  int in_header=1;
  char *temp;

  rewind(f);

  sprintf(Sock_buffer, "POST");
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_GIMME, );

  while(!feof(f)) {
    if (!fgets(Sock_buffer, sizeof( Sock_buffer ), f)) break;
    strrep(Sock_buffer, '\n', '\0');
    strrep(Sock_buffer, '\r', '\0');

    trim(temp=strdup(Sock_buffer));
    if (strlen(temp)==0) in_header=0;
    free(temp);

    if (*Sock_buffer=='.') {
      movmem(Sock_buffer, Sock_buffer+1, sizeof( Sock_buffer )-1 );
      *Sock_buffer='.';
    }

    if (in_header && eol) {
      if (!strncmp(Sock_buffer, "Newsgroups:", 11)) {
        fprintf(stderr, "%s\n", Sock_buffer); clreol();
      }
      else if (!strncmp(Sock_buffer, "Subject:", 8)) {
        fprintf(stderr, "%s\n", Sock_buffer); clreol();
      }
    }

    SOCK_PUTS( NNTP );
    if (Sock_echo) SOCK_GETS( NNTP );
  }

  sprintf(Sock_buffer, ".");
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_OK, );

  SOCK_READ_ERR(NNTP, );
  return 1;
}

int nntp_list(tcp_Socket *NNTP_sock, FILE *f)
{
  sprintf(Sock_buffer, "LIST");
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_LIST, );

  do {
    SOCK_GETS( NNTP );
    if ((Sock_buffer[0] == '.') && (Sock_buffer[1] == 0)) {
      break;
    }
    if (fprintf(f, "%s\n", Sock_buffer) < 0) return 0;
  } while (1);

  SOCK_READ_ERR(NNTP, );
  return 1;
}
 
int nntp_group(tcp_Socket *NNTP_sock, char *group,
               long *count, long *first, long *last)
{
  sprintf(Sock_buffer, "GROUP %s", group);
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_GROUP, );

  if (getnumbers(Sock_buffer, count, first, last, NULL) != 3) {
    SOCK_FAIL( NNTP, );
  }

  SOCK_READ_ERR(NNTP, );
  return 1;
}
 
int nntp_stat(tcp_Socket *NNTP_sock, long msg_num)
{
  sprintf(Sock_buffer, "STAT %ld", msg_num);
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  SOCK_FAIL_UNLESS(NNTP, NNTP_STAT, );

  SOCK_READ_ERR(NNTP, );
  return 1;
}

static int nntp_start_article(FILE *f)
{
  /* MMDF format */
  fprintf(f, "\001\001\001\001\n");
  return;
}

static int nntp_end_article(FILE *f)
{
  /* MMDF format */
  fprintf(f, "\001\001\001\001\n");
  return;
}

int nntp_article(tcp_Socket *NNTP_sock, FILE *f, long msg_num,
                 long body_lines, char eol)
{
  int in_body = 0;
  int flush = 0;
  int status = 1;

  if (body_lines == 0) {
    sprintf(Sock_buffer, "HEAD %ld", msg_num);
  }
  else {
    sprintf(Sock_buffer, "ARTICLE %ld", msg_num);
  }
  SOCK_COMMAND( NNTP );
  SOCK_QUIT_ON(NNTP, NNTP_OOPS, );
  if (body_lines == 0) {
    SOCK_FAIL_UNLESS(NNTP, NNTP_HEAD, );
  }
  else {
    SOCK_FAIL_UNLESS(NNTP, NNTP_ARTICLE, );
  }

  nntp_start_article(f);

  do {
    SOCK_GETS( NNTP );
    if ((Sock_buffer[0] == '.') && (Sock_buffer[1] == 0)) break;
    if (flush) continue;
    if (in_body) {
      if (body_lines == 0) {
        fprintf(f, "[Truncated]\n");
        flush = 1;
        continue;
      }
      body_lines--;
    }
    else if (*Sock_buffer == 0) {
      in_body = 1;
    }
    else if (eol && !strncmp(Sock_buffer, "From:", 5)) {
      fprintf(stderr, "%c%s", eol, Sock_buffer); clreol();
    }
    else if (eol && !strncmp(Sock_buffer, "Subject:", 8)) {
      fprintf(stderr, "%c%s", eol, Sock_buffer); clreol();
    }
    if (fprintf(f, "%s\n", Sock_buffer) < 0) {
      flush = 1;
      status = 0;
    }
  } while (1);

  SOCK_READ_ERR(NNTP, nntp_end_article(f));

  nntp_end_article(f);

  return status;
}

int nntp_shutdown(tcp_Socket *NNTP_sock)
{
  if (sock_established(NNTP_sock)) {
    sprintf(Sock_buffer, "QUIT");
    SOCK_COMMAND( NNTP );
    sock_close( NNTP_sock );
  }
  else {
    sock_abort( NNTP_sock );
  }
  SOCK_READ_ERR(NNTP, sock_abort(NNTP_sock));
  return 1;
}

/* End of nntplib.c */
